package org.springframework.prospring.ticket.web;

import java.util.*;

import javax.servlet.http.*;

import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.domain.support.*;
import org.springframework.prospring.ticket.service.*;
import org.springframework.web.bind.*;
import org.springframework.web.servlet.*;
import org.springframework.web.servlet.mvc.*;

/**
 * Kontroler pobierajcy przedstawienie z kalendarza zdarze (EventsCalendar)
 * w celu jego przedstawienia na stronie.<p>
 * Dane wejciowe: showId (identyfiaktor przedstawienia)<p>
 * Model:
 * <ul>
 *  <li><code>show</code>: przedstawienie jakie bdzie prezentowane
 *  <li><code>performances</code>: lista obiektw PerformancesWithAvailability
 *  <li><code>showInfoUrl</code>: adres URL do strony zawierajcej informacje o przedstawieniu.
 *  <li><code>seatingPlanUrl</code>: adres URL do strony zawierajcej plan miejsc dla przedstawienia.
 *  <li><code>seatClasses</code>: lista wszystkich kategorii miejsc
 * </ul>
 */
public class DisplayShowController extends AbstractController {

    private final static String STATIC_CONTENT_SUFFIX = ".htm";
    private final static String URL_PATH_SEPERATOR = "/";

    // usuga BoxOffice uywana przez ten kontroler.
    private BoxOffice boxOffice;

    // usuga kalendarza zdarze uywana przez ten kontroler.
    private EventsCalendar eventsCalendar;

    // nazwa zwracanego widoku.
    private String viewName;

    // pooenie plikw z informacjami o przedstawieniu.
    private String showInfosLocation;

    // pooenie plikw z informacjami o planie miejsc dla tego przedstawienia.
    private String seatingPlansLocation;

    /**
     * @see AbstractController#handleRequestInternal(
     *      javax.servlet.http.HttpServletRequest,
            javax.servlet.http.HttpServletResponse)
     */
    protected ModelAndView handleRequestInternal(HttpServletRequest httpRequest, HttpServletResponse httpResponse)
        throws Exception {

        // jeli w sesji jest jaka rezerwacja to odwoujemy j i usuwamy z sesji.
        cancelBookingIfOneExists(httpRequest);

        long showId = RequestUtils.getRequiredLongParameter(httpRequest, "showId");

        Show show = eventsCalendar.getShow(showId);
        PerformanceWithAvailability[] availability = boxOffice.getAvailabilityForPerformances(show);

        Date lastPerformanceDate = (availability.length == 0) ? null : availability[availability.length-1].getDateAndTime();

        ModelAndView mav = new ModelAndView(viewName);
        mav.addObject("performances", availability);
        mav.addObject("performancesCount", new Integer(availability.length));
        mav.addObject("lastPerformanceDate", lastPerformanceDate);
        mav.addObject("show", show);
        mav.addObject("showInfoUrl", getInfoUrl(show));
        mav.addObject("seatingPlanUrl", getSeatingPlanUrl(show));
        mav.addObject("seatClasses", getSeatClasses(availability));

        return mav;
    }

    /**
     * Sets the box office service this controller should use.
     * Ustawia obiekt usugi BoxOffice jakiego powinien uywa ten kontroler.
     * @param boxOffice Obiekt usugi BoxOffice jakiego powinien uywa ten kontroler.
     */
    public void setBoxOffice(BoxOffice boxOffice) {
        this.boxOffice = boxOffice;
    }

    /**
     * Ustawia kalendarz zdarze jakiego powinien uywa ten kontroler.
     * @param eventsCalendar Kalendarz zdarze jakiego powinien uywa ten kontroler.
     */
    public void setEventsCalendar(EventsCalendar eventsCalendar) {
        this.eventsCalendar = eventsCalendar;
    }

    /**
     * Ustawia lokalizacj polikw zawierajcych informacje o przedstawieniu. 
     * Pooenie to jest okrelane wzgldem katalogu gwnego aplikacji.
     * @param showInfosLocation Pooenie plikw z informacjami o przedstawieniu.
     */
    public void setShowInfosLocation(String showInfosLocation) {
        this.showInfosLocation = showInfosLocation;
    }

    /**
     * Ustawia lokalizacj polikw zawierajcych informacje o planie miejsc. 
     * Pooenie to jest okrelane wzgldem katalogu gwnego aplikacji.
     * @param seatingPlansLocation Pooenie plikw zawierajacych informacje o planie miejsc.
     */
    public void setSeatingPlansLocation(String seatingPlansLocation) {
        this.seatingPlansLocation = seatingPlansLocation;
    }

    /**
     * Ustawia nazw widoku zwracanego przez kontroler.
     * @param viewName Nazwa widoku zwracanego przez ten kontroler.
     */
    public void setViewName(String viewName) {
        this.viewName = viewName;
    }

    /**
     * Sprawdza czy w sesji jest jaka rezerwacja. Jeli jest, to rezerwacja ta zostaje 
     * anulowana i usunita z sesji.
     * Czynnoci te s wykonywane w celu usunicia rezerwacji wykonanej przeaz uytkownika, ktry
     * nie doszed do etapu przeprowadzenia patnoci i zdecycowaa si na wybranie innej daty 
     * lub grupy cenowej biletw.
     * @param request
     */
    protected void cancelBookingIfOneExists(HttpServletRequest request) {
        HttpSession session = request.getSession(true);
        Reservation reservation = (Reservation)session.getAttribute("reservation");
        if (reservation == null) {
            return;
        }
        session.removeAttribute("reservation");
        session.removeAttribute("reservationRequest");
        Booking booking = reservation.getBooking();
        boxOffice.cancelBooking(booking);
    }

    // zwraca adres URL pliku zawierajcego informacje o tym przedstawieniu.
    private String getInfoUrl(Show show) {
        return showInfosLocation + URL_PATH_SEPERATOR + show.getId() + STATIC_CONTENT_SUFFIX;
    }

    // zwraca adres URL pliku z informacjami o planie miejsc dla tego przedstawienia.
    private String getSeatingPlanUrl(Show show) {
        return seatingPlansLocation + URL_PATH_SEPERATOR + show.getSeatingPlan().getId() + STATIC_CONTENT_SUFFIX;
    }

    // zwraca list wszystkich dostpnych kategorii miejsc.
    private SeatClass[] getSeatClasses(PerformanceWithAvailability[] performances) {
        Set seatClasses = new HashSet();
        for (int i=0; i<performances.length; i++) {
            PerformanceWithAvailability performance = performances[i];
            List priceBands = performance.getPriceBandWithAvailability();
            for (Iterator iter = priceBands.iterator(); iter.hasNext();) {
                PriceBandWithAvailability priceBand = (PriceBandWithAvailability)iter.next();
                seatClasses.add(priceBand.getSeatClass());
            }
        }
        return (SeatClass[])seatClasses.toArray(new SeatClass[seatClasses.size()]);
    }
}
