package org.springframework.prospring.ticket.web;

import java.util.*;
import java.math.*;

import javax.servlet.http.*;

import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.service.*;
import org.springframework.validation.*;
import org.springframework.web.bind.*;
import org.springframework.web.servlet.*;
import org.springframework.web.servlet.view.*;
import org.springframework.web.servlet.mvc.*;
import org.springframework.web.util.*;

/**
 * Form controller issuing a ReservationRequest on behalf of the user.
 * The user has selected a show, a price band and a date in the previous
 * screen(s) and this form controller will create the reservation
 * request based on this. Together with the amount of seats the user will
 * be selected, this form controller completes the request.<p>
 * 
 * Kontroler formularza zgaszajcego w imieniu uytkownika danie 
 * rezerwacji (ReservationRequest). Uytkownik, na wczeniejszych stronach, 
 * wybra ju spektakl, grup cenow biletw oraz dat. Po okreleniu 
 * iloci miejsc, ten kontroler bdzie mg zakoczy obsug dania rezerwacji.
 *  
 * Dane wejciowe:
 * <ul>
 *   <li><code>performanceId</code>: identyfikator przedstawienia</li>
 *   <li><code>priceBandId</code>: identyfikator grupy cenowej</li>
 * </ul>
 * Warunki bdw:
 * <ul>
 *  <li>Brak danej iloci dostpnych miejsc</li>
 * </ul>
 */
public class BookSeatsController extends SimpleFormController {

    private final static int DEFAULT_MINUTES_TO_HOLD = 5;
    private final static int DEFAULT_DEFAULT_NUMBER_OF_SEATS_SELECTED = 1;
    private final static int DEFAULT_MAXIMUM_NUMBER_OF_SEATS = 12;
    private final static BigDecimal DEFAULT_BOOKING_FEE = new BigDecimal(0.0);

    // obiekt usugi BoxService uywany przez ten kontroler.
    private BoxOffice boxOffice;

    // kalendarz zdarze uywany przez ten kontroler.
    private EventsCalendar eventsCalendar;

    // ilo domylnie rezerwowanych miejsc.
    private int defaultNumberOfSeatsSelected = DEFAULT_DEFAULT_NUMBER_OF_SEATS_SELECTED;

    // maksymalna ilo miejsc jakie uytkownik moe zarezerwowa.
    private int maximumNumberOfSeats = DEFAULT_MAXIMUM_NUMBER_OF_SEATS;

    // ilo minut okrelajca jak dugo miejsca pozostan zarezerwowane.
    private int minutesReservationWillBeValid = DEFAULT_MINUTES_TO_HOLD;

    // opata za obsug rezerwacji.
    // UWAGA: Tak na prawd, ta informacja nie powinna by umieszczona w warstwie prezentacji.
    // Naley ona do zagadnie cile zwizanych z logik biznesow i tam powinna by 
    // podana (w odpowiedniej usudze - BoxOffice). Umiecilimy j tutaj, aby zapewni 
    // zgodno z fragmentami kodw zamieszczonymi w tekcie ksiki.
    private BigDecimal bookingFee = DEFAULT_BOOKING_FEE;

    /*
	 * @see org.springframework.web.servlet.mvc.AbstractFormController#formBackingObject(javax.servlet.http.HttpServletRequest)
	 */
    protected Object formBackingObject(HttpServletRequest req) throws Exception {

        long performanceId = RequestUtils.getRequiredLongParameter(req, "performanceId");
        Performance performance = eventsCalendar.getPerformance(performanceId);

        long priceBandId = RequestUtils.getRequiredLongParameter(req, "priceBandId");
        PriceBand priceBand = boxOffice.getPriceBand(priceBandId);

        return createReservationRequest(performance, priceBand, minutesReservationWillBeValid);
    }

    /**
     * Binds the maximum number of seats the user will be able to select
     * Kojarzy maksymaln ilo miejsc jakie uytkownik moe zarezerwowa
     * (klucz = <code>maximumSeats</code>) oraz domyln ilo rezerwowanych miejsc
     * (<code>defaultSelected</code>). 
     */
    protected Map referenceData(HttpServletRequest request, Object o, Errors errors) throws Exception {
        ReservationRequest reservationRequest = (ReservationRequest)o;
        int availableSeatsCount = boxOffice.getAvailableSeatsCount(
            reservationRequest.getPerformance(), reservationRequest.getPriceBand().getSeatClass());

        int maximumSeats = Math.min(maximumNumberOfSeats, availableSeatsCount);

        int defaultSelected = Math.min(defaultNumberOfSeatsSelected, availableSeatsCount);

        Map m = new HashMap();
        m.put("maximumSeats", new Integer(maximumSeats));
        m.put("defaultSelected", new Integer(defaultSelected));
        return m;
    }

    /**
     * @see SimpleFormController#onSubmit(
     *      javax.servlet.http.HttpServletRequest,
            javax.servlet.http.HttpServletResponse,
            Object,
            org.springframework.validation.BindException)
     */
    protected ModelAndView onSubmit(
        HttpServletRequest httpRequest,
        HttpServletResponse httpResponse,
        Object o,
        BindException errors) throws Exception {

        // obiekt to danie rezerwacji zawierajce ju stosowne dane
        ReservationRequest request = (ReservationRequest)o;
        Reservation reservation = null;
        try {
            reservation = boxOffice.allocateSeats(request);
        } catch (RequestedSeatNotAvailableException e) {
            errors.reject(e.getErrorCode(), "dane miejsca nie s dostpne");
        } catch (NotEnoughSeatsException e) {
            errors.reject(e.getErrorCode(), "Nie mona byo zarezerwowa wystarczajcej iloci miejsc");
        }

        // jeli pojawiy si bdy, to wywietlamy widok formularza ze stosownym komunikatem
        if (reservation == null) {
            return showForm(httpRequest, errors, getFormView());
        }

        // umieszczamy rezerwacj oraz spektakl w sesji (do wykorzystania w przyszoci).
        WebUtils.setSessionAttribute(httpRequest, "reservation", reservation);
        WebUtils.setSessionAttribute(httpRequest, "reservationRequest", request);

        // kierujemy uytkownika do innego kontrolera, ktry wywietli rezerwacj.
        // Ma to na celu uniemoliwienie ponownego przesania formularza.
        return new ModelAndView(new RedirectView(getSuccessView()));
    }

    /**
     * Ustaiwa usug BoxOffice jakiej powinien uywa ten kontroler.
     * @param boxOffice Usuga BoxOffice jakiej powinien uywa ten kontroler.
     */
    public void setBoxOffice(BoxOffice boxOffice) {
        this.boxOffice = boxOffice;
    }

    /**
     * Ustawia usuge kalendarza zdarze jakiej powinien uywa ten kontroler.
     * @param eventsCalendar Usuga kalendarza zdarze jakiej powinien uywa ten kontroler.
     */
    public void setEventsCalendar(EventsCalendar eventsCalendar) {
        this.eventsCalendar = eventsCalendar;
    }

    /**
     * Ustawia ilo minut przez ktre miejsca bd rezerwowane zanim zostan zamwione.
     * @param minutesReservationWillBeValid Ilo minut przez ktre miejsca bd rezerwowane zanim zostan zamwione.
     */
    public void setMinutesReservationWillBeValid(int minutesReservationWillBeValid) {
        this.minutesReservationWillBeValid = minutesReservationWillBeValid;
    }

    /**
     * Ustawia domyln ilo rezerwowanych miejsc.
     * @param defaultNumberOfSeatsSelected Domylna ilo rezerwowanych miejsc.
     */
    public void setDefaultNumberOfSeatsSelected(int defaultNumberOfSeatsSelected) {
        this.defaultNumberOfSeatsSelected = defaultNumberOfSeatsSelected;
    }

    /**
     * Ustawia maksymaln ilo miejsc jakie mona zarezerwowa w ramach jednej rezerwacji.
     * @param maximumNumberOfSeats Maksymalna ilo miejsc jakie mona zarezerwowa w ramach jednej rezerwacji.
     */
    public void setMaximumNumberOfSeats(int maximumNumberOfSeats) {
        this.maximumNumberOfSeats = maximumNumberOfSeats;
    }

    /**
     * Ustawia opat za obsug rezerwacji.
     * @param bookingFee Opata za obsug rezerwacji.
     */
    public void setBookingFee(BigDecimal bookingFee) {
        this.bookingFee = bookingFee;
    }

    //====================================== Metody pomocnicze (uywane do testowania) ====================================

    /**
     * Tworzy nowy obiekt ReservationRequest o podanych wartociach. Ta metoda moe by
     * uywana w testach do tworzenia da rezerwacji.
     *
     * @param performance Spektak.
     * @param priceBand Grupa cenowa.
     * @param minutesToHold Ilo minut przez ktre miejsca maj by rezerwowane.
     * @return Nowe danie rezerwacji.
     */
    protected ReservationRequest createReservationRequest(
        Performance performance,
        PriceBand priceBand,
        int minutesToHold) {

        ReservationRequest request = new ReservationRequest();
        request.setPerformance(performance);
        request.setPriceBand(priceBand);
        request.holdFor(minutesToHold);
        request.setBookingFee(bookingFee);

        return request;
    }

}
