package org.springframework.prospring.ticket.service;

import java.math.*;
import java.util.*;

import org.apache.commons.logging.*;
import org.springframework.prospring.ticket.dao.*;
import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.domain.support.*;
import org.springframework.prospring.ticket.service.payment.*;

/**
 * Domylna implementacja usugi BoxOffice.
 */
public class BoxOfficeImpl implements BoxOffice {

    private final static Log logger = LogFactory.getLog(BoxOfficeImpl.class);

    // Obiekt DAO uywany przez t usuge do pobierania informacji z bazy danych.
    private BoxOfficeDao boxOfficeDao;

    // mechanizm obsugi patnoci uywany do przetwarzania patnoci wykonywanych przy uyciu karty kredytowej.
    private PaymentProcessor paymentProcessor;

    /**
     * Zwraca list spektakli danego przedstawienia wraz z informacjami o dostpnych miejscach.
     * @param show Przedstawienie.
     * @return Lista spektakli podanego przedstawienia wraz z informacjami o dostpnych miejscach.
     *         Lista jest posortowana wedug daty spektaklu.
     */
    public PerformanceWithAvailability[] getAvailabilityForPerformances(Show show) {
        PerformanceWithAvailability[] performances = boxOfficeDao.getAvailabilityForPerformances(show);
        Arrays.sort(performances, new Comparator() {
            public int compare(Object o1, Object o2) {
                Performance p1 = (Performance)o1;
                Performance p2 = (Performance)o2;
                return p1.getDateAndTime().compareTo(p2.getDateAndTime());
            }
        });
        return performances;
    }

    /**
     * Przydziela miejsca na spektakl. Rezerwacja wraz z odpowiednimi informacjami szczegowymi jest 
     * tworzona na podstawie podanego dania rezerwacji. Miejsca s rezerwowane dla danego uytkownika na 
     * pewnien okres czasu, dziki czemu bdzie on mia szans dokonania patnoci.
     *
     * @param reservationRequest danie rezerwacji.
     * @return Nowa rezerwacja.
     * @throws org.springframework.prospring.ticket.service.RequestedSeatNotAvailableException
     *          Zgaszany gdy dane miejsca nie s dostpne.
     * @throws org.springframework.prospring.ticket.service.NotEnoughSeatsException
     *          Zgaszany gdy nie ma wystarczajcej iloci wolnych miejsc.
     */
    public Reservation allocateSeats(ReservationRequest reservationRequest) throws RequestedSeatNotAvailableException, NotEnoughSeatsException {
        Performance performance = reservationRequest.getPerformance();
        SeatClass seatClass = reservationRequest.getPriceBand().getSeatClass();

        Seat[] seats = boxOfficeDao.getAvailableSeats(performance, seatClass);
        if (seats.length < reservationRequest.getNumberOfSeatsRequested()) {
            throw new NotEnoughSeatsException(seatClass.getId(), seats.length, reservationRequest.getNumberOfSeatsRequested());
        }

        Seat[] seatsRequested = selectSeatsFromAvailableSeats(seats, reservationRequest.getNumberOfSeatsRequested());

        BigDecimal price = boxOfficeDao.getCostOfSeats(performance, seatsRequested);

        // Nowe szczegy rezerwacji
        Booking booking = createBooking(
            reservationRequest.getNumberOfSeatsRequested(),
            reservationRequest.getBookingFee().add(price),
            reservationRequest.getHoldUntil()
        );

        // Dodanie szczegw rezerwacji i aktualizacja statusu miejsc.
        boxOfficeDao.reserveSeats(seatsRequested, performance, booking);

        // Utworzenie obiektu Reservation.
        return new Reservation(seatsRequested, booking);
    }

    /**
     * Wykonuje zamwienie biletw na podstawie podanego dania zakupu.
     * @param purchaseRequest danie zakupu.
     * @return Pomylnie wykonane zamwienie.
     * @throws org.springframework.prospring.ticket.service.payment.PaymentException
     *          Zgaszany jeli podczas procesu obsugi patnoci wystpiy jakiekolwiek bdy.
     * @throws org.springframework.prospring.ticket.service.RequestedSeatNotAvailableException
     *          Zgaszany jeli podczas dokonywania rezerwacji, dane miejsca nie s ju dostpne.
     */
    public Purchase purchaseTickets(PurchaseRequest purchaseRequest)
        throws PaymentException, CreditCardValidationException, RequestedSeatNotAvailableException {

        // w pierwszej kolejnoci sprawdzamy poprawno dania zakupu.
        paymentProcessor.validate(purchaseRequest);

        // sprawdzenie czy wszystkie miejsca s jeszcze dostpne.
        Booking booking = purchaseRequest.getReservation().getBooking();
        if (!areSeatsStillAvailable(booking)) {
            if (logger.isDebugEnabled()) {
                logger.debug("Dokonanie rezerwacij niemoliwe, dane miejsca nie s ju  sostpne.; Nr rezerwacji #" +
                    booking.getId());
            }
            throw new RequestedSeatNotAvailableException("Miejsca nie s ju dostpne", purchaseRequest.getPerformance());
        }

        Purchase purchase = createPurchase();
        purchase.setBillingAddress(purchaseRequest.getBillingAddress());
        Address deliveryAddress = purchaseRequest.getDeliveryAddress();
        if (deliveryAddress == null) {
            deliveryAddress = purchaseRequest.getBillingAddress();
        }

        purchase.setPaymentAuthorizationCode("");
        purchase.setDeliveryAddress(deliveryAddress);
        purchase.setEmail(purchaseRequest.getEmail());
        purchase.setPurchaseDate(new Date());
        purchase.setCustomerName(purchaseRequest.getCreditCardDetails().getNameOnCard());
        purchase.setEncryptedCardNumber(purchaseRequest.getCreditCardDetails().getEncryptedCardNumber());
        purchase.setUserCollected(purchaseRequest.isCollect());
        booking.setPurchase(purchase);
        boxOfficeDao.savePurchaseForBooking(booking);

        String code = paymentProcessor.process(purchaseRequest);
        purchase.setPaymentAuthorizationCode(code);
        boxOfficeDao.updatePurchaseAuthorizationCode(purchase);

        return purchase;
    }

    /**
     * Sprawdza czy miejsca skojarzone z dan rezerwacj s jeszcze dostpne.
     * @param booking Szczegwe dane rezerwacji.
     * @return True miejsca s jeszcze dostpne,, false w przeciwnym przypadku.
     */
    public boolean areSeatsStillAvailable(Booking booking) {
        return boxOfficeDao.checkAvailabilityForBooking(booking);
    }

    /**
     * Zwraca wszystkie miejsca danej kategorii dostpne na podany spektak.
     * @param performance Spektakl.
     * @param seatClass   Klasa danych miejsc.
     * @return Wszystkie miejsca podanej kategorii, ktre na podany spektak s dostpne.
     */
    public Seat[] getAvailableSeats(Performance performance, SeatClass seatClass) {
        return boxOfficeDao.getAvailableSeats(performance, seatClass);
    }

    /**
     * Zwraca ilo dostpnych miejsc danej kategorii na podany spektakl.
     * @param performance Spektakl.
     * @param seatClass Kategoria miejsc.
     * @return Ilo dostpnych miejsc danej kategorii na podany spektakl.
     */
    public int getAvailableSeatsCount(Performance performance, SeatClass seatClass) {
        return boxOfficeDao.getAvailableSeatsCount(performance, seatClass);
    }

    /**
     * Zwraca grup cenow skojarzon o podanym identyfikatorze.
     * @param priceBandId Identyfikator grupy cenowej.
     * @return Grupa cenowa o podanym identyfikatorze.
     */
    public PriceBand getPriceBand(long priceBandId) {
        return boxOfficeDao.getPriceBand(priceBandId);
    }

    /**
     * Anuluje rezerwacj.
     * @param booking Rezerwacja jak naley anulowa.
     */
    public void cancelBooking(Booking booking) {
        boxOfficeDao.removeBooking(booking);
    }

    /**
     * Zwraca szczegy rezerwacji o podanym identyfiaktorze.
     * @param bookingId Idnetyfikator 
     * @return Szczegy rezerwaci o podanym identyfikatorze.
     */
    public Booking getBooking(long bookingId) {
        return boxOfficeDao.getBooking(bookingId);
    }

    /**
     * Zwraca zarezerwowane miejsca.
     * @param booking Podane szczegy rezerwacji.
     * @return Zarezerwowane miejsca.
     */
    public Seat[] getBookedSeats(Booking booking) {
        List seats = boxOfficeDao.getSeatsForBooking(booking);
        return (Seat[])seats.toArray(new Seat[seats.size()]);
    }

    /**
     * Zwraca spektakl na jaki zostay zarezerwowane miejsca.
     * @param booking Rezerwacja.
     * @return Spektakl, ktrego dotyczy rezerwacja.
     */
    public Performance getPerformanceOfBooking(Booking booking) {
        return boxOfficeDao.getPerformanceForBooking(booking);
    }

    //================================== Metody pobierajce/ustawiajce ===============================

    /**
     * Ustawia obiekt BoxOfficeDao z ktrego usuga bdzie korzysta przy wymianie informacji z baz danych.
     * @param boxOfficeDao Obiekt BoxOfficeDao z ktrego bdzie korzysta ta usuga.
     */
    public void setBoxOfficeDao(BoxOfficeDao boxOfficeDao) {
        this.boxOfficeDao = boxOfficeDao;
    }

    /**
     * Ustawia mechanizm obsugi patnoci przy uyciu kart z jakiego bdzie korzysta usuga.
     * @param paymentProcessor Mechanizm obsugi patnoci z ktrego bdzie korzysta ta aplikac.
     */
    public void setPaymentProcessor(PaymentProcessor paymentProcessor) {
        this.paymentProcessor = paymentProcessor;
    }

    //============================== Metody pomocnicze (przydatne podczas testowania) =================================

    /**
     * Wybiera podan ilo miejsc ze wskazanej listy wszystkich dostpnych miejsc.
     * Realnie wykorzystywana aplikacja zazwyczaj byaby lepsza i posiadaa interfejs strategii odpowiedzialny
     * za wykonywanie tej operacji oraz rne algorytmy dla rnych miejsc.
     *
     * @param availableSeats Lista dostpnych miejsc.
     * @param numberOfSeats Ilo wybieranych miejsc..
     * @return The selected seats.
     */
    protected Seat[] selectSeatsFromAvailableSeats(Seat[] availableSeats, int numberOfSeats) {
        Seat[] seatsRequested = new Seat[numberOfSeats];
        System.arraycopy(availableSeats, 0, seatsRequested, 0, numberOfSeats);
        return seatsRequested;
    }

    /**
     * Tworzy nowy obiekt szczegw rezerwacji o podanych wartociach.
     * @param price Koszt zamwienia.
     * @param holdUntil Data wyganicia wanoci rezerwacji przy zaoeniu, e rezerwacja ta nie zostaa rozliczona.
     * @return Nowy obiekt szczegw rezerwacji.
     */
    protected Booking createBooking(int seatCount, BigDecimal price, Date holdUntil) {
        return new Booking(seatCount, holdUntil, price);
    }

    /**
     * Tworzy nowy obiekt zakupu.
     * @return Nowy obiekt zakupu.
     */
    protected Purchase createPurchase() {
        return new Purchase();
    }


}
