package org.springframework.prospring.ticket.service;

import org.springframework.prospring.ticket.domain.*;
import org.springframework.prospring.ticket.domain.support.*;
import org.springframework.prospring.ticket.service.payment.*;

/**
 * Reprezentuje cay system kasowy. Zapewnia usugi o nastpujcych moliwociach:<br/>
 *
 * - wyszukiwanie dostpnych miejsc<br/>
 * - dokonywanie rezerwacji<br/>
 * - zarzdzanie patnociami<br/>
 * - inne usugi zwizane z procesem rezerwacji.
 */
public interface BoxOffice {

    /**
     * Zwraca list spektakli danego przedstawienia wraz z informacjami o dostpnych miejscach.
     * @param show Przedstawienie.
     * @return Lista spektakli wraz z informacjami o dostpnych miejscach.
     */
    public PerformanceWithAvailability[] getAvailabilityForPerformances(Show show);

    /**
     * Przydziela miejsca dla spektaklu. Rezerwacja o okrelonych danych szczegowych jest tworzona
     * na podstawie podanego dania rezerwacji. Rezerwacja jest tworzona dla uytkownika na pewien okres 
     * czasu, by pozwoli uytkownikowi na dokoczenie procesu patnoci.
     *
     * @param request danie rezerwacji.
     * @return Nowa rezerwacja.
     * @throws RequestedSeatNotAvailableException Zgaszany gdy dane miejsca nie s dostpne.
     * @throws NotEnoughSeatsException Zgaszany gdy nie ma tylu miejsc ilu zada uytkownik.
     */
    public Reservation allocateSeats(ReservationRequest request)
            throws RequestedSeatNotAvailableException, NotEnoughSeatsException;

    /**
     * Realizuje zamwienie biletw na podstawie dania zakupu.
     *
     * @param purchaseRequest danie zakupu.
     * @return Pomylnie przeprowadzone zamwienie.
     * @throws PaymentException Zgaszany gdy podczas realizacji patnoci pojawiy si jakiekolwiek bdy.
     * @throws RequestedSeatNotAvailableException Zgaszany gdy podczas zamawiania, ktrekolwiek z danych miejsc
     * nie byo ju dostpne.
     */
    public Purchase purchaseTickets(PurchaseRequest purchaseRequest)
        throws PaymentException, CreditCardValidationException, RequestedSeatNotAvailableException;

    /**
     * Sprawdza czy miejsca skojarzone z podanymi szczegami rezerwacji wci s dostpne.
     * @param booking Szczegy rezerwacji.
     * @return True jeli miejsca wci s dostpne, false w przeciwnym przypadku.
     */
    public boolean areSeatsStillAvailable(Booking booking);

    /**
     * Zwraca dostpne miejsca podanej kategorii na podany spektak.
     * @param performance Spektakl.
     * @param seatClass Kategoria miejsc.
     * @return Dostpne miejsca podanej klasy na wskazny spektakl.
     */
    public Seat[] getAvailableSeats(Performance performance, SeatClass seatClass);

    /**
     * Zwraca ilo dostpnych miejsc podanej klasy na podany spektakl.
     * @param performance Spektakl.
     * @param seatClass Kategoria miejsc.
     * @return Ilo miejsc podanej kategorii na wskazny spektakl.
     */
    public int getAvailableSeatsCount(Performance performance, SeatClass seatClass);

    /**
     * Zwraca grup cenow skojarzon z podanym identyfikatorem.
     * @param priceBandId Identyfikator grupy cenowej.
     * @return Grupa cenowa o podanym identyfikatorze.
     */
    public PriceBand getPriceBand(long priceBandId);

    /**
     * Anuluje rezerwacj.
     * @param booking Rezerwacja jak naley anulowa.
     */
    public void cancelBooking(Booking booking);

    /**
     * Zwraca rezerwacj o podanym identyfikatorze.
     * @param bookingId Identyfikator rezerwacji.
     * @return Rezerwacja o podanym identyfikatorze.
     */
    public Booking getBooking(long bookingId);

    /**
     * Zwraca zarezerwowane miejsca skojarzone z dan rezerwacj.
     * @param booking Szczegowe dane rezerwacji.
     * @return Zarezerwowane miejsca.
     */
    public Seat[] getBookedSeats(Booking booking);

    /**
     * Zwraca spektakl skojarzony z dan rezerwacj.
     * @param booking Szczegowe dane rezerwacji.
     * @return Skojarzony spektakl.
     */
    public Performance getPerformanceOfBooking(Booking booking);
}