package org.springframework.prospring.ticket.domain;

import java.io.*;

/**
 * Reprezentuje status (dostpny, zarezerwowane, itp.) miejsca dla konkretnego spektaku.
 */
public class SeatStatus implements Serializable {

    // skojarzone miejsce.
    private Seat seat;

    // grupa cenowa dla miejsca.
    private PriceBand priceBand;

    // skojarzony spektakl.
    private Performance performance;

    // rezerwacja skojarzona ze statusem miejsca
    private Booking booking;

    /**
     * Pusty konstruktor wymagany przez specyfikacj javabean.
     */
    public SeatStatus() {
    }

    /**
     * Konstruktor - tworzy nowy statu miejsca dla danego miejsca, grupy cenowej 
     * oraz spektaklu.
     * @param seat Miejsce skojarzone z tym statusem miejsca.
     * @param priceBand Grupa cenowa skojarzona z tym statusem miejsca.
     * @param performance Spektakl skojarzony z tym statusem miejsca.
     */
    public SeatStatus(Seat seat, PriceBand priceBand, Performance performance) {
        this.seat = seat;
        this.priceBand = priceBand;
        this.performance = performance;
    }

    /**
     * Zwraca miejsce skojarzone z tym statusem miejsca.
     * @return Miejsce skojarzone z tym statusem miejsca.
     */
    public Seat getSeat() {
        return seat;
    }

    /**
     * Ustawia miejsce skojarzone z tym statusem miejsca.
     * @param seat Miejsce skojarzone z tym statusem miejsca.
     */
    public void setSeat(Seat seat) {
        this.seat = seat;
    }

    /**
     * Grupa cenowa skojarzona z tym statusem miejsca.
     * @return Grupa cenowa skojarzona z tym statusem miejsca.
     */
    public PriceBand getPriceBand() {
        return priceBand;
    }

    /**
     * Ustawia grup cenow skojarzon z tym statusem miejsca.
     * @param priceBand Grupa cenowa skojarzona z tym statusem miejsca.
     */
    public void setPriceBand(PriceBand priceBand) {
        this.priceBand = priceBand;
    }

    /**
     * Zwraca spektakl skojarzony z tym statusem miejsca.
     * @return Spektakl skojarzony z tym statusem miejsca.
     */
    public Performance getPerformance() {
        return performance;
    }

    /**
     * Ustawia spektakl skojarzony z tym statusem miejsca.
     * @param performance Spektakl skojarzony z tym statusem miejsca.
     */
    public void setPerformance(Performance performance) {
        this.performance = performance;
    }

    /**
     * Zwraca rezerwacj (jeli jest) skojarzon z tym satusem miejsca.
     * @return Rezerwacja skojarzona z tym statusem miejsca. Null jeli taka rezerwacja nie istnieje.
     */
    public Booking getBooking() {
        return booking;
    }

    /**
     * Ustawia rezerwacj skojarzon z tym statusem miejsca.
     * @param booking Rezeracja skojarzona z tym statusem miejsca.
     */
    public void setBooking(Booking booking) {
        this.booking = booking;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        SeatStatus seatStatus = (SeatStatus) o;

        if (performance != null ? performance.getId() != seatStatus.performance.getId() : seatStatus.performance != null) {
            return false;
        }

        return (seat != null) ? seat.getId() == seatStatus.seat.getId() : seatStatus.seat == null;
    }

    public int hashCode() {
        int hashcode = (seat != null ? (int)seat.getId() : 0);
        hashcode = 31 * hashcode + (performance != null ? (int)performance.getId() : 0);
        return hashcode;
    }

}
