package org.springframework.prospring.ticket.domain;

import java.math.*;
import java.util.*;

/**
 * Reprezentuje szczegowe dane rezerwacji.
 */
public class Booking {

    // identyfikator rezerwacji.
    private long id;

    // data utworzenia rezerwacji.
    private Date dateMade;

    // data wanoci rezerwacji.
    private Date reservedUntil;

    // cena rezerwowanych biletw.
    private BigDecimal price;

    // dane zakupu dla tej rezerwacji
    private Purchase purchase;

    // ilo zarezerwowanych miejsc.
    // w rzeczywistej aplikacji byaby zapewne przechowywana lista wszystkich 
    // zarezerwowanych miejsc.
    // ta warto bdzie uywana przy sprawdzaniu czy wszystkie zarezerwowane
    // miejsca s wci dostpne.
    private int seatCount;

    /**
     * Pusty konstruktor wymagany przez specyfikacj javabean.
     */
    public Booking() {
    }

    /**
     * Konstruktor - tworzy now rezerwacj.
     * @param reserveUntil Data wanoci rezerwacji.
     * @param price Cena rezerwacji.
     */
    public Booking(int seatCount, Date reserveUntil, BigDecimal price) {
        this(seatCount, new Date(), reserveUntil, price);
    }

    /**
     * Konstruktor - tworzy now rezerwacj.
     * @param dateMade Data utworzenia rezerwacji.
     * @param reserveUntil Data wanoci rezerwacji.
     * @param price Cena rezerwacji.
     */
    public Booking(int seatCount, Date dateMade, Date reservedUntil, BigDecimal price) {
        this.seatCount = seatCount;
        this.dateMade = dateMade;
        this.reservedUntil = reservedUntil;
        this.price = price;
    }

    /**
     * Identyfikator tej rezerwacji.
     * @return Identyfikator rezerwacji.
     */
    public long getId() {
		return id;
	}

    /**
     * Ustawia identyfikator rezerwacji
     * @param id Identyfikator rezerwacji.
     */
    public void setId(long id) {
		this.id = id;
	}

    /**
     * Zwraca dat utworzenia rezerwacji.
     * @return Data utworzenia rezerwacji.
     */
    public Date getDateMade() {
		return dateMade;
	}

    /**
     * Ustawia dat utworzenia rezerwacji.
     * @param dateMade Data utworzenia rezerwacji.
     */
    public void setDateMade(Date dateMade) {
		this.dateMade = dateMade;
	}

    /**
     * Zwraca cen rezerwacji.
     * @return Cena rezerwacji.
     */
    public BigDecimal getPrice() {
		return price;
	}

    /**
     * Ustawia cen rezerwacji
     * @param price Cena rezerwacji.
     */
    public void setPrice(BigDecimal price) {
		this.price = price;
	}

    /**
     * Zwraca zakup dla tej rezerwacji, jeli takie istnieje.
     * @return Zakup dla tej rezerwacji, jeli taki istnieje.
     */
    public Purchase getPurchase() {
		return purchase;
	}

    /**
     * Ustawia zakup dla tej rezerwacji.
     * @param purchase Zakup dla tej rezerwacji.
     */
    public void setPurchase(Purchase purchase) {
		this.purchase = purchase;
	}

    /**
     * Zwraca dat wanoci rezerwacji.
     * @return Data wanoci rezerwacji.
     */
    public Date getReservedUntil() {
		return reservedUntil;
	}

    /**
     * Ustawia dat wanoci rezerwacji
     * @param reservedUntil Data wanoci rezerwacji.
     */
    public void setReservedUntil(Date reservedUntil) {
		this.reservedUntil = reservedUntil;
	}

    /**
     * Zwraca ilo zarezerwowanych miejsc.
     * @return Ilo zarezerwowanych miejsc.
     */
    public int getSeatCount() {
        return seatCount;
    }

    /**
     * Ustawia ilo zarezerwowanych miejsc.
     * @param seatCount Ilo zarezerwowanych miejsc.
     */
    public void setSeatCount(int seatCount) {
        this.seatCount = seatCount;
    }

}
