/*
 Serwer WWW GET/POST z odczytem kart SD
 Kontekst: Arduino
 Odczytuje czujnik temperatury TMP36 i przekazuje wynik 
 na stronie WWW. Umożliwia ustawienie termostatu
 z formularza WWW.
 */
#include <SD.h>
#include <EEPROM.h>
#include <SPI.h>
#include <Ethernet.h>
#include <TextFinder.h>

// konfiguracja połączenia Ethernet:
byte mac[] = {  
  0x00, 0xAA, 0xBB, 0xCC, 0xDE, 0x01 };
IPAddress gateway(192,168,1,1);
IPAddress subnet(255,255,255,0);
IPAddress ip(192,168,1,20);

// zainicjuj bibliotekę serwera Ethernet:
EthernetServer server(80);

const int fileStringLength = 16;       // długość żądanego pliku
const int typeLength = 6;              // długość GET lub POST
const int sdChipSelect = 4;            // chipSelect karty SD
const int relayPin = 2;                // pin do którego podlączony jest przekaźnik
const long tempCheckInterval = 10000;  // czas między sprawdzeniami (w ms)
const int thermostatAddress = 10;      // adres EEPROM  termostatu


char fileString[fileStringLength];     // dla danych wejściowych z przeglądarki 
char requestTypeString[typeLength];    // typ żądania: GET lub POST

long now;                              // ostatnie sprawdzenie temperatury 
int thermostat = EEPROM.read(thermostatAddress);  // punkt aktywacji termostatu


void setup() {
  // zainicjuj komunikację szeregową:
  Serial.begin(9600);
  // zainicjuj wyjście przekaźnika:
  pinMode(relayPin, OUTPUT);

  // sprawdź, czy jest karta SD:
  Serial.print(F("Inicjowanie karty SD..."));
  if (!SD.begin(sdChipSelect)) {
    //jeśli nie można odczytać karty SD, wydrukuj błąd i przejdź dalej:
    Serial.println(F("Inicjowanie nie powiodło się!"));
  }
  else {
    Serial.println(F("Inicjowanie zakończone."));
  }
  
  // daj kontrolerowi Ethernet czas na uruchomienie:
  delay(1000);
  Serial.println(F("Próba uzyskania adresu"));
  // próba uruchomienia przez DHCP. Jeśli nie powiedzie się, zrób to ręcznie:
  if (!Ethernet.begin(mac)) {
    Ethernet.begin(mac, ip, gateway, subnet);
  }
  // wyświetl adres IP i uruchom serwer:
  Serial.println(Ethernet.localIP());
  server.begin();
}

void loop() {
  String fileName = "";        // nazwa pliku żądanego przez klienta 
  char inChar = 0;             // znak przychodzący od klienta 
  int requestType = 0;         // typ żądania (GET lub POST);
  int requestedFileLength = 0; // długość nazwy żądnego pliku

  // nasłuchuj przychodzących klientów:
  EthernetClient client = server.available();  

  if (client) {
    // stwórz instancję TextFinder, by przeszukać dane od klienta:
    TextFinder finder(client );  

    while (client.connected()) {      
      if (client.available()) {      
        // poszukaj tego, co znajduje się przed  /. Powinno to być GET lub POST:
        if(finder.getString("","/", requestTypeString,typeLength)){
          // zrób coś innego dla GET i POST:
          if(String(requestTypeString) == "GET " ) {
            requestType = 1;
          }
          else if(String(requestTypeString) == "POST ") {
            requestType = 2;
          }

          // zbierz to co przychodzi po / do tablicy,
          // jest to nazwa pliku, którego żąda klient:
          requestedFileLength = finder.getString("", " ", 
          fileString, fileStringLength);

          // po zakończeniu wiersza GET/POST przetwórz to, co masz:
          switch (requestType) {
          case 1:    // GET
            // nie rób nic z GET poza wysłaniem pliku, poniżej:
            break;
          case 2:    //POST
            // pomiń resztę nagłówka,
            // który kończy się znakiem nowego wiersza i powrotu karetki:
            finder.find("\n\r");
            // jeśli klient wysyła wartość dla termostatu, pobierz ją:
            if (finder.find("thermostat")) {
              int newThermostat = finder.getValue('=');
              // jeśli wartość została zmieniona, zapisz ją:
              if (thermostat != newThermostat) {
                thermostat = newThermostat;
                // ogranicz zakres od 20 do 40 stopni:
                thermostat = constrain(thermostat, 20, 40);
                // zapisz ją w EEPROM:
                EEPROM.write(thermostatAddress, thermostat);
              }
            }
            break; 
          }

          // niezależnie czy jest to GET, czy POST, zwróć żądany ciąg.
          // Jeśli brak pozycji po /,
          // to klient chce plik indeksu:
          if (requestedFileLength < 2) {
            sendFile(client, "index.htm");
          }             
          // w przeciwnym razie wyślij żądany plik:
          else  {
            sendFile(client, fileString);
          }
        }
        // daj czas klientowi na odebranie danych:
        delay(1);
        // zamknij połączenie:
        Serial.println(F("Closing the connection"));
        client.stop();
      }
    }
  }

  // okresowo sprawdzaj temperaturę, aby określić,
  // czy należy włączyć termostat:
  if (millis() - now > tempCheckInterval) {
    Serial.print("Temperatura: ");
    Serial.println(readSensor());
    if (checkThermostat()) {
      Serial.println("Termostat jest włączony");
    }
    else {
      Serial.println("Termostat jest wyłączony");
    }
    now = millis();
  }
}

// odczytaj czujnik temperatury:
float readSensor() {
  // odczytaj wartość z czujnika:
  int sensorValue = analogRead(A0);
  // skonwertuj odczyt na wolty:
  float voltage = (sensorValue *  5.0) / 1024.0; 
  // skonwertuj napięcie do temperatury w st. Celsujsza
  // (100 mv na stopień - 500mV przesunięcie):
  float temperature = (voltage - 0.5) * 100;
  // zwróć temperaturę:
  return temperature; 
}

// sprawdź temperaturę i odpowiednio steruj przekaźnikiem:
boolean checkThermostat() {
  // załóżmy, że przekaźnik powinien być wyłączony:
  boolean relayState = LOW;
  // jeżeli temperatura jest większa niż punkt aktywacji 
  // termostatu, przekaźnik powinien być włączony:
  if(readSensor() > thermostat) {
    relayState = HIGH;
  }
  // włącz lub wyłącz przekaźnik:
  digitalWrite(relayPin, relayState);
  return relayState;
}

// Wyślij żądany plik:
void sendFile(EthernetClient thisClient, char thisFile[]) {
  String outputString = "";      // ciąg dla każdego wiersza pliku

  // otwórz plik do odczytu:
  File myFile = SD.open(thisFile);
  if (myFile) {
    // wyślij nagłówek OK:
    sendHttpHeader(thisClient, 200);
    // odczytaj z pliku, dopóki nie zostanie w nim nic innego:
    while (myFile.available()) {
      // dodaj bieżący znak do ciągu wyjściowego:
      char thisChar = myFile.read();
      outputString += thisChar; 
      // sprawdź zmienną temperatury i zastąp
      // (typ float nie może być konwertowany na ciągi znaków, więc prześlij go bezpośrednio):
      if (outputString.endsWith("$temperature")) {
        outputString = "";
        // ogranicz wynik do 2 miejsc dziesiętnych:
        thisClient.print(readSensor(), 2);
      } 

      // sprawdź zmienną termostatu i zastąp:
      if (outputString.endsWith("$thermostat")) {
        outputString.replace("$thermostat", String(thermostat));
      } 

      // sprawdź zmienną status przekaźnika i zastąp:
      if (outputString.endsWith("$status")) {
        String relayStatus = "wyłączony";
        if (checkThermostat()) {
          relayStatus = "włączony";
        } 
        outputString.replace("$status", relayStatus);
      } 

      // gdy pojawi się nowy wiersz, wyślij i wyczyść outputString:
      if (thisChar == '\n') {
        thisClient.print(outputString);
        outputString = "";
      } 
    }
    // jeśli plik nie jest zakończony znakiem nowego wiersza, wyślij ostatni wiersz:
    if (outputString != "") {
      thisClient.print(outputString); 
    }

    // zamknij plik:
    myFile.close();
  } 
  else {
    // jeśli plik nie został otwarty:
    sendHttpHeader(thisClient, 404);
  } 
}


// wyślij nagłówek HTTP do klienta:
void sendHttpHeader(EthernetClient thisClient, int errorCode) {
  thisClient.print(F("HTTP/1.1 "));
  switch(errorCode) {
  case 200: // OK
    thisClient.println(F("200 OK"));
    thisClient.println(F("Content-Type: text/html"));
    break;
  case 404: // nie można odnaleźć pliku 
    thisClient.println(F("404 Not Found"));
    break;
  }
  // nagłówek odpowiedzi kończy się dodatkowym wysuwem wiersza:
  thisClient.println();
}













