/* 
  Program do odczytu pakietów XBee i rysowania wykresu.
  Kontekst: Processing
  Odczytuje pakiet z radia XBee przez UDP i analizuje go.
  Rysuje wykresy wyników w czasie. Pakiet powinien mieć 22 bajtów, 
  ułożonych następująco:
 bajt 1:     0x7E, wartość początkowa
 bajty 2-3:  rozmiar pakietu, wartość 2-bajtowa (nie używane tutaj)
 bajt 4:     wartość identyfikatora API, kod określający jaka to jest odpowiedź (nie używane tutaj)
 bajty 5-6:  adres nadawcy
 bajt 7:     siła sygnału, Received Signal Strength Indicator (nie używane tutaj)
 bajt 8:     opcje rozgłaszania (nie używane tutaj)
 bajt 9:     liczba próbek 
 bajty 10-11: sygnalizacja aktywnych kanałów (nie używane tutaj)
 bajty 12-21: 5 wartości 10-bitowych, poszczególne próbki ADC od nadawcy
*/

import hypermedia.net.*;
import processing.serial.*;

UDP udp;                 // zdefiniuj obiekt UDP 
int queryPort = 43770;   // numer portu dla kwerendy urz±dzenia 
int hPos = 0;            // pozycja pozioma na wykresie
int lineHeight = 14;     // a variable to set the line height


void setup() {
  // ustaw rozmiar okna
  size(400,300);

  // utwórz nowe połączenie do nasłuchiwania 
  // datagramów UDP na porcie kwerendy:
  udp = new UDP(this, queryPort);

  // nasłuchuj przychodzących pakietów:
  udp.listen( true );

  // ustaw czarne to
  background(0);
  // pokaż tekst odczytu:
  drawReadings(0,0);
}

void draw() {
  // nic się tutaj nie dzieje.
}

/*
  Nasłuchuj odpowiedzi UDP 
*/
void receive( byte[] data, String ip, int port ) {
  int[] inString = int(data);   // przychodzące dane konwertowane na ciąg 
  parseData(inString);
}

/*
 Po otrzymaniu pakietu, musisz wydobyć z niego użyteczne dane.
 Ta metoda bierze adres nadawcy i 5 odczytów ADC.
 Następnie uśrednia odczyty ADC i prezentuje wynik. 
 */

void parseData(int[] thisPacket) {
  // najpierw upewnij się, że pakiet ma długość 22 bajtów:
  if (thisPacket.length >= 22) {
    int adcStart = 11;                      // odczyt ADC zaczyna się od 12 bajtu 
    int numSamples = thisPacket[8];         // liczba próbek w pakiecie 
    int[] adcValues = new int[numSamples];  // tablica do przechowywania 5 odczytów
    int total = 0;                          // suma wszystkich odczytów ADC 

    // odczytaj adres. Jest to wartość dwubajtowa, więc musisz 
    //dodać dwa bajty w następujący sposób:
    int address = thisPacket[5] + thisPacket[4] * 256;

    // odczytaj siłę odebranego sygnału:
    int signalStrength = thisPacket[6];

    // odczytaj <numSamples> wartości analogowe 10-bitowe, dwie naraz,
    // ponieważ każdy odczyty ma dwa bajty:
    for (int i = 0; i < numSamples * 2; i=i+2) {
      // wartość 10-bitowa = wyższy bajt * 256 + niski bajt:
      int thisSample = (thisPacket[i + adcStart] * 256) +
      thisPacket[(i + 1) + adcStart];
      // umieść wynik w jednym z 5 bajtów:
      adcValues[i/2] = thisSample;
      // dodaj wyniku do łącznej wartości dow celu późniejszego uśredniania:
    total = total + thisSample;
  }
  // uśrednij wynik:
  int average = total / numSamples;
    // narysuj linię na wykresie i odczyty:
    drawGraph(average);
    drawReadings(average, signalStrength);
  }
}

/*
  update the graph 
 */
void drawGraph(int thisValue) {
  // rysuj linię:
  stroke(#4F9FE1);
  // mapuj daną wartość do wysokośćci okna:
  float graphValue = map(thisValue, 0, 1023, 0, height);
  // określ wysokość linii dla wykresu:
  float graphLineHeight = height - (graphValue);
  // rysuj linię:
  line(hPos, height, hPos, graphLineHeight);
  // przy krawędzi ekranu wróć do początku:
  if (hPos >= width) {
    hPos = 0;
    // wyczyść ekran:
    background(0);
  }
  else {
    // zmień położenie na osi  poziomej do rysowania następnej linii:
    hPos++;
  }
}

/*
  draw the date and the time
 */
void drawReadings(int thisReading, int thisSignalStrength) {
  // skonfiguruj tablicę, aby uzyskać nazwy miesięcy
  // z ich wartości liczbowych:
  String[] months = {
    "Sty", "Lut", "Mar", "Kwi", "Maj", "Cze", "Lip", "Sie",
    "Wrz", "Paź", "Lis", "Gru"
  };

  // sformatuj ciąg daty :
  String date = day() + " " + months[month() -1] + " " + year() ;

  // format ciągu czasu 
  // wszystkie liczby są sformatowane jako dwie cyfry:
  String time = nf(hour(), 2) + ":" + nf(minute(), 2) + ":" + nf(second(),2);

  // oblicz napięcie z odczytu:
  float voltage = thisReading * 3.3 / 1024;
  
  // wybierz pozycję dla tekstu:
  int xPos = 20;
  int yPos = 20;

  // wyczyść poprzednie odczyty:
  noStroke();
  fill(0);
  rect(xPos,yPos, 180, 80);
  // zmień kolor wypełnienia tekstu :
  fill(#4F9FE1);
  // wyświetl odczyty:
  text(date, xPos, yPos + lineHeight);
  text(time, xPos, yPos + (2 * lineHeight));
  text("Napięcie: " + voltage + "V", xPos, yPos + (3 * lineHeight));
  text("Siła sygnału: -" + thisSignalStrength + " dBm", xPos,
  yPos + (4 * lineHeight));
}

