/*
  Analogowe przesyłanie dwukierunkowe Bluetooth
  Kontekst: Arduino
 Ten szkic steruje radiem Bluetooth przez port szeregowy, 
 wysyla wartość sensora analogowego i nasluchuje danych
 z radia, używając ich do ustawienia wyjścia PWM. 
*/
#include <TextFinder.h>

const int sensorPin = A0;         // czujnik wejścia analogowego
const int analogLed = 3;          // dioda LED, która zmienia jasność
const int threshold = 20;         // próg zmiany czujnika
const int debounceInterval = 15;  // używane do wygładzania odczytów 
                                  // przycisku 
const int connectButton = 2;      // przycisk do podłączania 
int lastButtonState = 0;          // poprzedni stan przycisku 
int lastSensorReading = 0;        // poprzedni stan czujnika 
long lastReadingTime = 0;         // czas wcześniejszego odczytu czujnika 

// adres zdalnego radia BT. Zamień na adres 
// swojego zdalnego radia
String remoteAddress = "112233445566";
String messageString = "";       // wiadomości przychodzące 
                                 // na port szeregowy 

boolean connected = false;       // jesteś połączony czy nie 
boolean commandMode = false;     // jesteś w trybie poleceń czy danych

TextFinder finder(Serial);       // do przeszukiwania danych 
                                 // z portu szeregowego

void setup() {
  // konfiguracja komunikacji szeregowej:
  Serial.begin(115200);

  // konfiguracja pinów wyjściowych:
  pinMode (analogLed, OUTPUT);

  // zamigaj diodą LED TX, sygnalizując, że 
  // główny program zaraz się zacznie:
  blink(analogLed, 3);
}

void loop() {
  // przeczytaj przychodzące dane szeregowe i sparsuj je:
  handleSerial();

  // sprawdź, czy przycisk jest wciśnięty:
  boolean buttonPushed = buttonRead(connectButton);

  // jeśli przycisk został właśnie naciśnięty:
  if (buttonPushed) {
    // jeśli klient jest podłączony, rozłącz:
    if (connected) {
      BTDisconnect();
    } // jeśli klient jest rozłączony, spróbuj połączyć się:
    else {
      BTConnect();
    }
  }
  
  // jeśli połączony, odczytaj czujnik:
  if (connected) {
    // zapisz bieżący czas w milisekundach:
    long currentTime = millis();
    // jeżeli upłynęło wystarczająco dużo czasu od ostatniego odczytu:
    if (currentTime - lastReadingTime > debounceInterval) {
      // odczytaj czujnik analogowy, podziel przez 4, 
      // aby uzyskać zakres 0 - 255:
      int sensorValue = analogRead(A0)/4;
      // jeśli istnieje istotna różnica między 
      //bieżącym i ostatnim odczytem czujnika, wyślij go:
      if (abs(sensorValue - lastSensorReading) > threshold) {
        Serial.println(sensorValue, DEC);
      }
      // zapisz czas ostatniego odczytu 
      // i ostatni odczyt czujnika:
      lastReadingTime = currentTime;
      lastSensorReading = sensorValue;
    }
  }
}

// zamigaj diodą LED:
void blink(int thisPin, int howManyTimes) {
  for (int blinks=0; blinks< howManyTimes; blinks++) {
    digitalWrite(thisPin, HIGH);
    delay(200);
    digitalWrite(thisPin, LOW);
    delay(200);
  }
}

void BTConnect() {
  // jeśli w trybie danych, wyślij $$$
  if (!commandMode) {
    Serial.print("$$$");
    // czekaj na odpowiedź :
    if (finder.find("CMD")) {
      commandMode = true;
    }
  }
  // kiedy jesteś w trybie poleceń, wyślij polecenie połączenia:
  if (commandMode) {
    Serial.print("C," + remoteAddress + "\r");
    // czekaj na odpowiedź:
    finder.find("CONNECT");
    // jeśli wiadomość to "CONNECT failed":
    if (finder.find("failed")) {
      connected = false;
    }
    else {
      connected = true;
      // radio automatycznie przejdzie w tryb danych,
      // kiedy się połączy:
      commandMode = false;
    }
  }
}

void BTDisconnect() {
  // jeśli w trybie danych, wyślij $$$
  if (!commandMode) {
    Serial.print("$$$");
    // czekaj na odpowiedź:
    if (finder.find("CMD")) {
      commandMode = true;
    }
  }
  // kiedy jesteś w trybie poleceń,
  // wyślij polecenie rozłącz:
  if (commandMode) {
    // próba połączenia: 
    Serial.print("K,\r");
    // poczekaj na komunikat o pomyślnym rozłączeniu:
    if (finder.find("BTDISCONNECT")) {
      connected = false;
      // radio automatycznie przejdzie w tryb danych, 
      // kiedy jest rozłączone:
      commandMode = false;
    }
  }
}

void handleSerial() {
  // sprawdź ciąg wiadomości:
  // jeśli to BTCONNECT, connected = true;
  // jeśli to BTDISCONNECT, connected = false;
  // jeśli to CONNECT failed, connected = false;
  // jeśli to liczba, ustaw LED
  char inByte = Serial.read();

  // dodaj dowolne znaki alfanumeryczne ASCII 
  // do ciągu wiadomości:
  if (isAscii(inByte)) {
    messageString = messageString + inByte;
  }

  // obsłuż wiadomości CONNECT i DISCONNECT:
  if (messageString == "BTDISCONNECT") {
    connected = false;
  }
  if (messageString == "BTCONNECT") {
    connected = true;
  }
  if (connected) {
    // konwertuj ciąg znaków na liczbę:
    int brightness = messageString.toInt();
    // ustaw analogowe wyjście LED:
    if (brightness > 0) {
      analogWrite(analogLed, brightness);
    }
  }

  // jeśli otrzymasz znak ASCII powrotu karetki:
  if (inByte == '\r') {
    // wyczyść ciąg wejściowy dla 
    // następnej wartości:
    messageString = "";
  }
}


// ta metoda odczytuje przycisk, aby zobaczyć, 
// czy właśnie się zmienił ze stanu niskiego 
// na wysoki, i filtruje odbicie przycisku w przypadku 
// zakłóceń elektrycznych:

boolean buttonRead(int thisButton) {
  boolean result = false;
  // tymczasowy stan przycisku:
  int currentState = digitalRead(thisButton);
  // końcowy stan przycisku:
  int buttonState = lastButtonState;
  // uzyskaj bieżący czas do odliczenia czasu interwału odbijania:
  long lastDebounceTime = millis();
  
  while ((millis() - lastDebounceTime) < debounceInterval) {
    // odczytaj stan przełącznika do zmiennej lokalnej:
    currentState = digitalRead(thisButton);
    // jeśli przycisk zmienił się z powodu zakłóceń:
    if (currentState != buttonState) {
      // resetuj czasomierz odbijania
      lastDebounceTime = millis();
    }
    // niezależnie od odczytu, czekamy już dłużej
    // niż opóźnienie odbijania, dlatego należy 
    // przyjąć go jako rzeczywisty bieżący stan:
    buttonState = currentState;
  }
  // jeśli stan przycisku zmienił się na wysoki:
  if(buttonState != lastButtonState && buttonState == HIGH) {
    result = true;
  }
  // zapisz bieżący stan na następny raz:
  lastButtonState = buttonState;
  return result;
}

