package cwp;

import java.sql.*;

public class NarzedziaBazyDanych {
  
  /** Tworzy poczenie z baz danych, wykonuje okrelone zapytanie
   *  i gromadzi wyniki w obiekcie klasy BDRezultat.
   *  Jeli poczenie z baz danych jest ju otwarte (w tym celu naley uy argumentu
   *  zamknij), moliwe jest pobranie poczenia za pomoc
   *  metody BDRezultat.dajPolaczenie.
   */
  
  public static BDRezultat dajRezultat(String sterownik,
                                       String url,
                                       String nazwaUzytkownika,
                                       String haslo,
                                       String zapytanie,
                                       boolean zamknij) {
    try {
      Class.forName(sterownik);
      Connection polaczenie =
        DriverManager.getConnection(url, nazwaUzytkownika, haslo);
      return(dajRezultat(polaczenie, zapytanie, zamknij));
    } catch(ClassNotFoundException cnfe) {
      System.err.println("Blad wczytywania sterownika: " + cnfe);
      return(null);
    } catch(SQLException sqle) {
      System.err.println("Blad polaczenia: " + sqle);
      return(null);
    }
  }

  /** Podobnie jak poprzednia, metoda pobiera wyniki, lecz zamiast
   *  tworzenia nowego poczenia uywa ju istniejcego.
   */
  
  public static BDRezultat dajRezultat(Connection polaczenie,
                                       String zapytanie,
                                       boolean zamknij) {
    try {
      DatabaseMetaData dbMetadane = polaczenie.getMetaData();
      String nazwaBazy =
        dbMetadane.getDatabaseProductName();
      String wersjaBazy =
        dbMetadane.getDatabaseProductVersion();
      Statement instrukcja = polaczenie.createStatement();
      ResultSet zestawWynikow = instrukcja.executeQuery(zapytanie);
      ResultSetMetaData metadaneZestawu =
        zestawWynikow.getMetaData();
      int liczbaKolumn = metadaneZestawu.getColumnCount();
      String[] nazwyKolumn = new String[liczbaKolumn];
      // indeks kolumn rozpoczyna si od 1 (SQL), a nie od 0 (Java).
      for(int i=1; i<liczbaKolumn+1; i++) {
        nazwyKolumn[i-1] =
          metadaneZestawu.getColumnName(i).trim();
      }
      BDRezultat wynikiDB =
        new BDRezultat(polaczenie, nazwaBazy, wersjaBazy,
                      liczbaKolumn, nazwyKolumn);      
      while(zestawWynikow.next()) {
        String[] wiersz = new String[liczbaKolumn];
        // Ponownie indeks obiektu klasy ResultSet rozpoczyna sie od 1, a nie od 0.
        for(int i=1; i<liczbaKolumn+1; i++) {
          String wpis = zestawWynikow.getString(i);
          if (wpis != null) {
            wpis = wpis.trim();
          }
          wiersz[i-1] = wpis;
        }
        wynikiDB.dodajWiersz(wiersz);
      }
      if (zamknij) {
        polaczenie.close();
      }
      return(wynikiDB);
    } catch(SQLException sqle) {
      System.err.println("Blad polaczenia: " + sqle);
      return(null);
    } 
  }

  /** Tworzy tabele majce okrelony format oraz wiersze. */
  
  public static Connection stworzTabele(String sterownik,
                                        String url,
                                        String nazwaUzytkownika,
                                        String haslo,
                                        String nazwaTabeli,
                                        String formatTabeli,
                                        String[] wierszeTabeli,
                                        boolean zamknij) {
    try {
      Class.forName(sterownik);
      Connection polaczenie =
        DriverManager.getConnection(url, nazwaUzytkownika, haslo);
      return(stworzTabele(polaczenie, nazwaUzytkownika, haslo,
                          nazwaTabeli, formatTabeli,
                          wierszeTabeli, zamknij));
    } catch(ClassNotFoundException cnfe) {
      System.err.println("Blad wczytywania sterownika: " + cnfe);
      return(null);
    } catch(SQLException sqle) {
      System.err.println("Blad polaczenia: " + sqle);
      return(null);
    } 
  }

  /** Podobnie jak w przypadku poprzedniej metody, lecz uywa istniejcego poczenia. */
  
  public static Connection stworzTabele(Connection polaczenie,
                                        String nazwaUzytkownika,
                                        String haslo,
                                        String nazwaTabeli,
                                        String formatTabeli,
                                        String[] wierszeTabeli,
                                        boolean zamknij) {
    try {
      
      Statement instrukcja = polaczenie.createStatement();
      // Usuwa poprzedni tabel, lecz nie zwraca bdu w przypadku 
      // gdy taka nie istnieje. Dlatego te w tym miejscu s oddzielne bloki try ? catch.
      try {
        instrukcja.execute("DROP TABLE " + nazwaTabeli);
      } catch(SQLException sqle) {}
      String instrukcjaCreate =
        "CREATE TABLE " + nazwaTabeli + " " + formatTabeli;
      instrukcja.execute(instrukcjaCreate);
      String instrukcjaInsert =
        "INSERT INTO " + nazwaTabeli + " VALUES";
      for(int i=0; i<wierszeTabeli.length; i++) {
        instrukcja.execute(instrukcjaInsert + wierszeTabeli[i]);
      }
      if (zamknij) {
        polaczenie.close();
        return(null);
      } else {
        return(polaczenie);
      }
    } catch(SQLException sqle) {
      System.err.println("Blad tworzenia tabeli: " + sqle);
      return(null);
    } 
  }

  public static void pokazTabele(String sterownik,
                                String url,
                                String nazwaUzytkownika,
                                String haslo,
                                String nazwaTabeli,
                                int szerokoscWpisu,
                                boolean zamknij) {
    String zapytanie = "SELECT * FROM " + nazwaTabeli;
    BDRezultat wyniki =
      dajRezultat(sterownik, url, nazwaUzytkownika,
                      haslo, zapytanie, zamknij);
    pokazDaneTabeli(nazwaTabeli, wyniki, szerokoscWpisu, true);
  }

  /** Wypisuje wszystkie rekordy tabeli. Kady wpis bdzie 
   *  wywietlony w kolumnie o liczbie znakw okrelonej w szerokoscWpisu
   *  dlatego te naley upewni si, e okrelie warto rwn najszerszemu
   *  wynikowi.
   */

  public static void pokazTabele(Connection polaczenie,
                                 String nazwaTabeli,
                                 int szerokoscWpisu,
                                 boolean zamknij) {
    String zapytanie = "SELECT * FROM " + nazwaTabeli;
    BDRezultat wyniki =
      dajRezultat(polaczenie, zapytanie, zamknij);
    pokazDaneTabeli(nazwaTabeli, wyniki,szerokoscWpisu, true);
  }

  public static void pokazDaneTabeli(String nazwaTabeli,
                                    BDRezultat wyniki,
                                    int szerokoscWpisu,
                                    boolean wypiszMetadane) {
    if (wyniki == null) {
      return;
    }
    if (wypiszMetadane) {
      System.out.println("Baza danych: " +
                         wyniki.dajNazwaBazy());
      System.out.println("Wersja: " +
                         wyniki.dajWersjaBazy());
      System.out.println();
    }
    System.out.println(nazwaTabeli + ":");
    String podkreslenie =
      dopelnij("", nazwaTabeli.length()+1, "=");        
    System.out.println(podkreslenie);
    int liczbaKolumn = wyniki.dajLiczbaKolumn();
    String separator =
      stworzOdstep(szerokoscWpisu, liczbaKolumn);
    System.out.println(separator);
    String wiersz = stworzWiersz(wyniki.dajNazwyKolumn(), szerokoscWpisu);
    System.out.println(wiersz);
    System.out.println(separator);
    int liczbaWierszy = wyniki.dajLiczbaWierszy();
    for(int i=0; i<liczbaWierszy; i++) {
      wiersz = stworzWiersz(wyniki.dajWiersz(i), szerokoscWpisu);
      System.out.println(wiersz);
    }
    System.out.println(separator);
  }
  
  // acuch znakowy postaci "|  xxx |  xxx |  xxx |"

  private static String stworzWiersz(String[] wpisy,
                                int szerokoscWpisu) {
    String wiersz = "|";
    for(int i=0; i<wpisy.length; i++) {
      wiersz = wiersz + dopelnij(wpisy[i], szerokoscWpisu, " ");
      wiersz = wiersz + " |";
    }
    return(wiersz);
  }
    
  // acuch znakowy postaci "+------+------+------+"
  
  private static String stworzOdstep(int szerokoscWpisu,
                                      int liczbaKolumn) {
    String wpis = dopelnij("", szerokoscWpisu+1, "-");
    String separator = "+";
    for(int i=0; i<liczbaKolumn; i++) {
      separator = separator + wpis + "+";
    }
    return(separator);
  }

  private static String dopelnij(String oryginalny, int rozmiar,
                                  String przesuniecie) {
    if (oryginalny == null) {
      oryginalny = "<null>";
    }
    // Uywa bufora StringBuffer, a nie zwykego doczania acuchw znakowych
    // w celu uniknicia tworzenia zbyt duej liczby tymczasowych acuchw znakowych.
    StringBuffer bufor = new StringBuffer("");
    int dodatkoweZnaki = rozmiar - oryginalny.length();
    for(int i=0; i<dodatkoweZnaki; i++) {
      bufor.append(przesuniecie);
    }
    bufor.append(oryginalny);
    return(bufor.toString());
  }
}
