/** Kodowanie liczb naturalnych w systemie typów, podobne do liczb Churcha */
sealed trait Nat {
  // Ten typ pozwala na rozwijanie typów dla liczb naturalnych. 
  // Up to ograniczenie górne dla typów wynikowych Expand.
  // NonZero to konstruktor typu, któremu należy przekazać poprzednią liczbę naturalną. 
  // IfZero to typ używany dla 'bazowej' liczby naturalnej _0.
  type Expand[NonZero[N <: Nat] <: Up, IfZero <: Up, Up] <: Up
}

object Nat {
  // Kodowanie 'zera' w systemie typów.
  sealed trait _0 extends Nat {
    // Expand pobiera także przypadek IfZero
    type Expand[NonZero[N <: Nat] <: Ret, IfZero <: Ret, Ret] = IfZero
  }
  // Kodowanie liczb naturalnych różnych od zera w oparciu o kolejne liczby.
  // Prev to poprzednia liczba w systemie typów.
  sealed trait Succ[Prev <: Nat] extends Nat {
    type Expand[NonZero[N <: Nat] <: Ret, IfZero <: Ret, Ret] = NonZero[Prev]
  }
  // Kodowanie liczb 1->22 w systemie typów.   
  // Uwaga: Możesz używać tylko liczb od _1 do _22.
  type _1 = Succ[_0]
  type _2 = Succ[_1]  
  type _3 = Succ[_2]
  type _4 = Succ[_3]
  type _5 = Succ[_4]
  type _6 = Succ[_5]
  type _7 = Succ[_6]
  type _8 = Succ[_7]
  type _9 = Succ[_8]
  type _10 = Succ[_9]
  type _11 = Succ[_10]
  type _12 = Succ[_11]
  type _13 = Succ[_12]
  type _14 = Succ[_13]
  type _15 = Succ[_14]
  type _16 = Succ[_15]
  type _17 = Succ[_16]
  type _18 = Succ[_17]
  type _19 = Succ[_18]
  type _20 = Succ[_19]
  type _21 = Succ[_20]
  type _22 = Succ[_21]
}

/** Definiuje heterogeniczną listę typowaną. FullType zawiera pełen typ HList */
trait HListLike[FullType <: HList] extends HList {
  def viewAt[Idx <: Nat](implicit in : FullType => FullType#ViewAt[Idx]) = in(this.asInstanceOf[FullType])
  def ::[T](v : T) = HCons(v, this.asInstanceOf[FullType])
}

/**Typ bazowy dla list HList. Typ możemy przechwycić stosując FullType <: HList i różne algorytmy rekurencyjne. 
 * Może się to przydać podczas przekazywania list heterogenicznie typowanych o wspólnej klasie typu.
 */
sealed trait HList {
  /**Definiuje typ widoku View listy na danej pozycji. 
   * Widoki są wykorzystywane do modyfikowania listy i do uzyskania dostępu do wartości w oparciu o indeks.
   */
  type ViewAt[N <: Nat] <: IndexedView
}

/**
 * Ta klasa reprezentuje węzeł wiążący w heterogenicznej liście wiązanej. Typ H to typ głowy listy, 
 * a typ T to typ ogona.
 */
final case class HCons[H, T <: HList](head : H, tail : T) extends HListLike[HCons[H,T]] {
  /**
   * Typ widoku listy na pozycji N. 
   * Jest tworzony poprzez rekurencyjne dodawanie klas HListViewX z wykorzystaniem liczb naturalnych zakodowanych w systemie typów.   
   */
  type ViewAt[N <: Nat] = N#Expand[
    ({ type Z[P <: Nat] = HListViewN[H, T#ViewAt[P]] })#Z,
    HListView0[H,T], 
    IndexedView]
  override def toString = head + " :: " + tail
}

/**
 * Pusta lista heterogeniczna.
 */
class HNil extends HListLike[HNil] {
  type ViewAt[N <: Nat] = Nothing
  override def toString = "HNil"
}

/**
 * Obiekt towarzyszacy HList zawiera wygodne definicje.
 */
object HList {
  // Operator typu do konstrukcji HList za pomocą ::
  type ::[H, T <: HList] = HCons[H,T]
  // Alias dla HCons, dzięki któremu listy HList można konstruować za pomocą ::
  val :: = HCons
  // Obiekt singletonowy HNil stosowany do konstrukcji list HList. 
  // HNil to zarówno typ listy pustej jak i wartość listy pustej, w zależności od kontekstu.
  val HNil = new HNil
  // Ta metoda rekurencyjnie tworzy indeksowany widok listy za pomocą systemu domniemań.
  def indexedView[List <: HList,
                  Idx <: Nat](list : List)(implicit in : List => List#ViewAt[Idx]) = in(list)
}

import HList._

/** Cecha bazowa widoków indeksowanych. 
 * System typów otrzymuje tu podstawowy typ, na którym można wywoływać metody.
 */
sealed trait IndexedView {
 type Before <: HList
 type After <: HList
 type At
 def fold[R](f : (Before, At, After) => R) : R
 def get = fold( (_, value, _) => value)
}

/** Obiekt towarzyszący, który przechowuje domniemane fabryki klas IndexedView */
object IndexedView {
  /** Przypadek bazowy:  widok indexedview głowy listy */
  implicit def index0[H, T <: HList](list : H :: T) : HListView0[H,T] =
    new HListView0[H,T](list)
  /** Przypadek rekurencyjny: widok indexedView obsługujący głowę i delegujący do widoku ogona. */
  implicit def indexN[H, T <: HList, Prev <: IndexedView](
     list : (H :: T))(implicit indexTail : T => Prev) : HListViewN[H,Prev] =
       new HListViewN[H, Prev](list.head, indexTail(list.tail))
}


/**
 * Widok listy HList z głowy listy. Przechwycono zarówno typ głowy jak i typ ogona.
 * Metoda fold została zdefiniowana dla głowy listy.
 */
final class HListView0[H, T <: HList](val list : H :: T) extends IndexedView {
  type Before = HNil
  type After = T
  type At = H
  def fold[R](f : (Before, At, After) => R): R =
    f(HNil, list.head, list.tail)
}

/**
 * Widok listy HList w przypadku, gdy aktualna pozycja (indeks) *nie* jest pożądaną pozycją dla widoku.
 * Widok jest tworzony w oparciu o ogon listy. Przechwytywany jest jeden element poprzedzający ogon.
 * W ten sposób możemy dostać się do dowolnej pozycji listy, tworząc HListView0 na pożądanej pozycji
 * i tworząc widoku HListViewN przechwytujące wartości poprzedzające naszą wartość.
 */
final class HListViewN[H, NextIdxView <: IndexedView](h : H, next : NextIdxView) extends IndexedView {
  type Before = H :: NextIdxView#Before
  type At = NextIdxView#At
  type After = NextIdxView#After
  def fold[R](f : (Before, At, After) => R) : R = 
    next.fold( (before, at, after) =>
      f(HCons(h, before), at, after) )
}


