# == Opis programu
# 
#  photo_utility: Manipulowanie zdjciami w celu zmiany ich rozmiaru, 
#  dodania znakw wodnych lub utworzenia galerii
# 
# 
# == Sposb uycia
# 
# photo_utility [OPCJE] ... ZDJCIE
# 
# -h, --help
#   Wywietla pomoc
# 
# --bw, -b
#    Przeksztaca zdjcie na czarno-biae
# 
# --gallery, -g
#    Tworzy galeri zdj. Jeli uywasz tej opcji, wpisz "temp" jako argument ZDJCIE
# 
# --info, -i
#    Pobiera informacje o zdjciu
# 
# --resize rozmiar, -r rozmiar
#    Zmienia rozmiar pliku na podstawie podanych wymiarw
# 
# --watermark tekst, -w tekst
#    Dodaje do zdjcia znak wodny w postaci okrelonego tekstu
# 
# ZDJCIE: Zdjcie, ktre chcesz zmodyfikowa

require 'getoptlong'
require 'rdoc/ri/ri_paths'
require 'rdoc/usage'
require 'RMagick'
require 'exifr'
require 'ftools'
include EXIFR
include Magick

#Przeksztacanie zdjcia na czarno-biae.
def bw(file)
  new_img = "bw_#{file}"
  img = Image.read(file).first

  img = img.quantize(256, GRAYColorspace)

  if File.exists?(new_img)
    puts "Nie mona zapisa zdjcia - plik ju istnieje."
    exit
  end
                
  img.write(new_img)
end

#Tworzenie internetowej galerii zdj.
def gallery()
  photos_row = 4
  table_border = 1
  html_rows = 1

  #Tworzenie potrzebnych katalogw.
  File.makedirs("gallery/thumbs", "gallery/resize")
  output = File.new("gallery/index.html","w+b")

  output.puts <<EOF
            <html>
                <head>
                    <title>Zdjcia</title>
                </head>
                <body bgcolor="#d0d0d0">
                    <h1>Witaj w galerii zdj</h1>
                    <table border=#{table_border}>
EOF

  Dir['*.[Jj][Pp]*[Gg]'].each do |pic|
    # Tworzenie miniatur.
    thumb = Image.read(pic)[0]
    thumb.change_geometry!('150x150') do |cols, rows, img|
      thumb.resize!(cols, rows)
    end
    if File.exists?("gallery/thumbs/th_#{pic}")
      puts "Nie mona zapisa pliku th_#{pic} miniatura ju istnieje.  Zmiana nazwy na new_th_#{pic}"
      thumb.write "gallery/thumbs/new_th_#{pic}"
    else
      # Zapisz zdjcia w odrbnym katalogu, jeli chcesz je uporzdkowa.
      # Moesz te po prostu posortowa fotografie wedug nazw plikw.
      thumb.write "gallery/thumbs/th_#{pic}"
    end

    # Zmiana wielkoci zdj.
    resize = Image.read(pic)[0]
    resize.change_geometry!('800x600') do |cols, rows, img|
      resize.resize!(cols, rows)
    end
    if File.exists?("gallery/resize/resize_#{pic}")
      puts "Nie mona zapisa pliku resize_#{pic} - zdjcie o zmienionych wymiarach ju istnieje. Zmiana nazwy na new_resize_#{pic}"
      resize.write("gallery/resize/new_resize_#{pic}")
    else
      resize.write("gallery/resize/resize_#{pic}")
    end
                
    if html_rows % photos_row == 1
      output.puts "\n<tr>"
    end
                
    output.puts <<EOF
                    <td><a href="resize/resize_#{pic}/" title="#{pic}" target="_blank"><img src="thumbs/th_#{pic}" alt="#{pic}"/></a></td>
EOF
                    
    if html_rows % photos_row == 0
      output.puts "</tr>"
    end
    html_rows+=1
  end

  unless html_rows % photos_row == 1
    output.puts "</tr>"
  end

  output.puts "</body>\n</html>"
  output.puts "<!-- Dziki uprzejmoci No Starch Press (skrypt z ksiki Wicked Cool 
      Ruby Scripts) -->"
  output.close
end

#Pobieranie informacji o zdjciu.
def info(file)
  info = JPEG.new(file)

  File.open("info_#{File.basename(file)}.txt", "w") do |output|
    output.puts info.exif.to_hash.map{ |k,v| "#{k}:  #{v}"}
  end
end

#Zmienianie wymiarw zdjcia.
def resize(file, arg)
  size = arg.chomp
  img = Image.read(file).first
  width = nil
  height = nil
            
  img.change_geometry!("#{size}x#{size}") do |cols, rows, img|
    img.resize!(cols, rows)
    width = cols
    height = rows
  end

  file_name = "#{width}x#{height}_#{file}"

  if File.exists?(file_name)
    puts "Zdjcie ju istnieje - nie mona zapisa pliku."
    exit
  end

  img.write(file_name)
end

#Dodawanie znaku wodnego do zdjcia.
def watermark(file, arg)
  text = arg.chomp
  img = Image.read(file).first
  new_img = "wm_#{file}"

  if File.exists?(new_img)
    puts "Zdjcie ju istnieje - nie mona zapisa pliku."
    exit
  end

  watermark = Image.new(600, 50)

  watermark_text = Draw.new
  watermark_text.annotate(watermark, 0,0,0,0, text) do
    watermark_text.gravity = CenterGravity
    self.pointsize = 50
    self.font_family = "Arial"			#Jeli masz wymyln czcionk, moesz uy jej w tym miejscu.
    self.font_weight = BoldWeight
    self.stroke = "none"
  end

  watermark.rotate!(45)
  watermark = watermark.shade(true, 310, 30)
  img.composite!(watermark, SouthWestGravity, HardLightCompositeOp)		#Oznaczenie w lewym dolnym rogu.
  watermark.rotate!(-90)
  img.composite!(watermark, NorthWestGravity, HardLightCompositeOp)		#Oznaczenie w lewym grnym rogu.
  watermark.rotate!(90)
  img.composite!(watermark, NorthEastGravity, HardLightCompositeOp)		#Oznaczenie w prawym grnym rogu.
  watermark.rotate!(-90)
  img.composite!(watermark, SouthEastGravity, HardLightCompositeOp)		#Oznaczenie w prawym dolnym rogu.

  puts "Zapisywanie pliku #{new_img}"
  img.write(new_img)
end

#Definiowanie argumentw podawanych w wierszu polece.
opts = GetoptLong.new(
  [ '--help', '-h', GetoptLong::NO_ARGUMENT ],
  [ '--black', '-b', GetoptLong::NO_ARGUMENT ],
  [ '--gallery', '-g', GetoptLong::NO_ARGUMENT ],
  [ '--info', '-i', GetoptLong::NO_ARGUMENT ],
  [ '--resize', '-r', GetoptLong::REQUIRED_ARGUMENT ],
  [ '--watermark', '-w', GetoptLong::REQUIRED_ARGUMENT ]
)

filename = ARGV[-1].chomp

#Instrukcja case zawiera blok powizany z kadym z wymienionych wczeniej argumentw.
opts.each do |opt, arg|
  case opt
  when '--help'
    RDoc::usage
  when '--black'
    bw(filename)
  when '--gallery'
    gallery()
  when '--info'
    info(filename)
  when '--resize'
    resize(filename, arg)
  when '--watermark'
    watermark(filename, arg)
  else
    RDoc::usage
  end
end
