# Definicja metody Module.synchronize_method, ktra tworzy acuchy aliasw metod egzemplarza,
# dziki czemu s one synchronizowane na egzemplarzu przed wykonaniem.
class Module
  # Metoda pomocnicza dla tworzenia acucha aliasw.
  # Majc podan nazw metody (jako acuch lub symbol) i prefiks, tworzy
  # unikatowy alias dla tej metody i zwraca nazw tego aliasu jako symbol.
  # Wszystkie znaki interpunkcyjne w oryginalnej nazwie metody
  # s konwertowane na liczby, dziki czemu mona tworzy aliasy operatorw.
  def create_alias(original, prefix="alias")
    # Przykleja prefiks do oryginalnej nazwy i konwertuje znaki interpunkcyjne.
    aka = "#{prefix}_#{original}"
    aka.gsub!(/([\=\|\&\+\-\*\/\^\!\?\~\%\<\>\[\]])/) {
      num = $1[0]                       # Znak Ruby 1.8 -> porzdkowy.
      num = num.ord if num.is_a? String # Znak Ruby 1.9 -> porzdkowy.
      '_' + num.to_s
    }

    # Dodawanie znakw podkrelenia, a powstanie nazwa, ktra nie jest jeszcze uywana.
    aka += "_" while method_defined? aka or private_method_defined? aka
    aka = aka.to_sym            # Konwersja nazwy aliasu na symbol.
    alias_method aka, original  # Rzeczywiste utworzenie aliasu.
    aka                         # Zwrcenie nazwy aliasu.
  end
  # Dodanie metody do acucha aliasw, aby umoliwi synchronizacj.
  def synchronize_method(m)
    # Najpierw trzeba utworzy alias dla niesynchronizowanej wersji metody.
    aka = create_alias(m, "unsync")
    # Teraz naley przedefiniowa orygina, aby wywoywa alias w synchronizowanym bloku.
    # Chcesz, aby zdefiniowana metoda przyjmowaa bloki, a wic
    # nie moesz uy metody define_method, konieczne jest wykonanie acucha za pomoc
    # metody class_eval. Wszystko pomidzy znakami %Q{ i pasujcym znakiem }
    # jest acuchem w podwjnych cudzysowach, nie blokiem.
    class_eval %Q{
      def #{m}(*args, &block)
        synchronized(self) { #{aka}(*args, &block) }
      end
    }
  end
end
# Niniejszej globalnej synchronizowanej metody mona teraz uywa na trzy rne sposoby.
def synchronized(*args)
  # Przypadek 1.: z jednym argumentem i blokiem, synchronizacja na obiekcie
  # i wykonanie bloku.
  if args.size == 1 && block_given?
    args[0].mutex.synchronize { yield }
  # Przypadek 2.: z jednym argumentem niebdcym symbolem i bez bloku.
  # Zwraca obiekt klasy SynchronizedObject.
  elsif args.size == 1 and not args[0].is_a? Symbol and not block_given?
    SynchronizedObject.new(args[0])
  # Przypadek 3.: w przypadku wywoania na rzecz moduu bez bloku wyznaczone metody czone s w acuch
  # w celu umoliwienia synchronizacji. Lub, jeli nie ma adnych argumentw,
  # do acucha zostaje dodana kolejna zdefiniowana metoda.
  elsif self.is_a? Module and not block_given?
    if (args.size > 0) # Synchronizacja wyznaczonych metod.
      args.each {|m| self.synchronize_method(m) }
    else
      # Jeli nie zostan okrelone adne metody, synchronizacji podlega kolejna zdefiniowana metoda.
      eigenclass = class<<self; self; end
      eigenclass.class_eval do # Uycie klasy eigenclass do zdefiniowania metod klasowych.
        # Definicja metody method_added powiadamiajcej o zdefiniowaniu kolejnej metody.
        define_method :method_added do |name|
          # Najpierw trzeba usun t metod zwrotn.
          eigenclass.class_eval { remove_method :method_added }
          # Nastpnie synchronizujesz metod, ktra wanie zostaa dodana.
          self.synchronize_method name
        end
      end
    end
  # Przypadek 4.: kade inne wywoanie jest bdem.
  else
    raise ArgumentError, "Nieprawidowe argumenty metody synchronize()"
  end
end
