# Metoda trace dowolnego obiektu tworzy nowy obiekt, ktry zachowuje si
# dokadnie tak samo jak orygina, ale ledzi wszystkie wywoania metod
# na rzecz tego obiektu. W przypadku ledzenia wicej ni jednego obiektu,
# naley poda nazw, ktra ma zosta uyta w danych wyjciowych. Domylnie
# komunikaty bd wysyane do strumienia STDERR, ale mona wyznaczy dowolny strumie
# (lub dowolny obiekt przyjmujcy acuchy jako argumenty do <<).
class Object
  def trace(name="", stream=STDERR)
    # Zwraca ledzcy i delegujcy obiekt klasy TracedObject.
    TracedObject.new(self, name, stream)
  end
end
# Niniejsza klasa ledzi wywoania metod za pomoc metody method_missing i
# deleguje je do jakiego innego obiektu. Usuwa wikszo wasnych
# metod egzemplarza, aby nie wchodziy w konflikty z metod method_missing.
# Zauwa, e ledzone bd tylko metody wywoywane poprzez obiekt klasy TracedObject.
# Jeli obiekt delegowany wywouje metod na swoj rzecz, ledzenia te nie
# s ledzone.
class TracedObject
  # Wszystkie mao wane publiczne metody egzemplarza zostaj oddefiniowane.
  # Zauwa uycie metod Module.instance_methods i Module.undef_method.
  instance_methods.each do |m|
    m = m.to_sym  # Ruby 1.8 zwraca acuchy zamiast symboli.
    next if m == :object_id || m == :__id__ || m == :__send__
    undef_method m
  end
  # Inicjalizacja egzemplarza klasy TracedObject.
  def initialize(o, name, stream)
    @o = o            # Obiekt, do ktrego delegujemy.
    @n = name         # Nazwa obiektu, ktra ma pojawia si w komunikatach.
    @trace = stream   # Miejsce, do ktrego maj by wysyane komunikaty.
  end
  # To jest kluczowa metoda klasy TracedObject. Jest wywoywana dla
  # kadego wywoania metody na rzecz obiektu klasy TracedObject.
  def method_missing(*args, &block)
    m = args.shift         # Pierwszy argument jest nazw metody.
    begin
      # ledzenie wywoa tej metody.
      arglist = args.map {|a| a.inspect}.join(', ')
      @trace << "Wywoywanie: #{@n}.#{m}(#{arglist}) w #{caller[0]}\n"
      # Wywoanie metody na rzecz delegowanego obiektu i odebranie wartoci zwrotnej.
      r = @o.send m, *args, &block
      # ledzenie normalnego zwrotu metody.
      @trace << "Zwracanie: #{r.inspect} z #{@n}.#{m} do #{caller[0]}\n"
      # Zwrcenie wartoci zwrconej przez obiekt delegowany.
      r
    rescue Exception => e
      # ledzenie nienormalnego zwrotu metody.
      @trace << "Zgoszenie: #{e.class}:#{e} z #{@n}.#{m}\n"
      # Ponowne zgoszenie wyjtku zgoszonego przez obiekt delegowany.
      raise
    end
  end
  # Zwrcenie obiektu, do ktrego delegujesz.
  def __delegate
    @o
  end
end
