{*****************************************************}
{*                                                   *}
{* Nazwa modulu  : UFUNKCJE                          *}
{* Wersja        : 1.00                              *}
{* Kompilowane w : Turbo Pascal version 7.0          *}
{* Procesor      : 8086 lub lepszy                   *}
{* Co-procesor   : zalecany                          *}
{* Autor         : Buk Mariusz                       *}
{* Data          : 15 VIII 1994                      *}
{* Miejsce       : Polska                            *}
{*                                                   *}
{* Opis:                                             *}
{*                                                   *}
{*   Modul UFUNKCJE jest uniwersalnym narzedziem do  *}
{*   obliczania dowolnych wyrazen matematycznych.    *}
{*                                                   *}
{*****************************************************}

{$A+,B-,D-,E+,I+,L+,N+,O+,Q-,R-,S-,T-,V+,X+}

Unit UFunkcje;

interface

uses Objects;

type


  { Typ wyliczeniowy zawierajacy identyfikatory elementow kodu }
  { wstepnego i wynikowego.                                    }

  TKod = (reZmienna, reStala, reFunkcja, reOperator, reKoniec);


  { Typ wyznaczajacy jeden element kodu wynikowego.  }
  { Pola:                                            }
  {   typ - typ elementu (np. funkcja lub stala)     }
  {   nr  - numer elementu (np. nr operatora,        }
  {         dodawania, nr funkcji itd)               }

  TElementKodu = record
    typ : TKod;
    nr  : Integer;
  end;


  { Typ wyznaczajacy jeden element kodu wstepnego. }
  { Pola:                                          }
  {   typ - typ elementu (np. zmienna lub stala)   }
  {   nr  - numer elementu (np. nr operatora,      }
  {         dodawania, nr funkcji itd)             }
  {   priorytet - priorytet operatora lub funkcji  }
  {               (funkcja ma zawsze priorytet     }
  {               rowny 255)                       }

  TElementKoduWstepnego = record
    typ       : TKod;
    nr        : Integer;
    priorytet : Word;
  end;


const

  { Stale wielkosci poszczegolnych elementow. }

  WlkElKodu    = SizeOf(TElementKodu);

  WlkElKoduWst = SizeOf(TElementKoduWstepnego);

  WlkLiczby    = SizeOf(Extended);


type


  { Typ identyfikujacy tablice kodu wynikowego. }

  PTablicaKodu = ^TTablicaKodu;
  TTablicaKodu = array [1..(65520 div WlkElKodu)] of TElementKodu;


  { Typ uzywany do przechowywania kodu wstepnego. }

  PKodWstepny  = ^TKodWstepny;
  TKodWstepny  = array [1..(65520 div WlkElKoduWst)] of TElementKoduWstepnego;


  { Typ wyznaczajacy tablice liczb typu Extended. }

  PTablicaLiczb = ^TTablicaLiczb;
  TTablicaLiczb = array [1..(65520 div WlkLiczby)] of Extended;


type


  { Ponizszy typ jest przodkiem obiektow takich jak TZmienna, TStala, }
  { TOperator i TFunkcja. Zawiera podstawowe metody przydzielania     }
  { i zwalniania pamieci.                                             }
  { Pola:                                                             }
  {   Nazwa - wskaznik na lancuch zawierajacy nazwe obiektu           }
  {   Id    - identyfikator obiektu (w celu przyspieszenia operacji)  }
  {   Stan  - stan obiektu (zablokowany/odblokowany)                  }

  PElement = ^TElement;
  TElement = object (TObject)
    Nazwa : PString;
    Id    : Word;
    Stan  : Boolean;
    constructor Init(Anazwa : string; Aid : Word);
    constructor Load(var S : TStream);
    destructor Done; virtual;
    procedure Store(var S : TStream); virtual;
  end;


  { Typ TZmienna reprezentuje jeden obiekt, tzn. zmienna zdefiniowana }
  { przez uzytkownika.                                                }
  { Pola:                                                             }
  {   nr_pom - numer pomocniczy, uzywany do przyspieszenia dostepu    }
  {            do danej zmiennej                                      }

  PZmienna = ^TZmienna;
  TZmienna = object (TElement)
    nr_pom : Word;
    constructor Load(var S : TStream);
    procedure Store(var S : TStream); virtual;
  end;


  { Typ TStala reprezentuje jeden obiekt, tzn. stala zdefiniowana }
  { przez uzytkownika.                                            }
  { Pola:                                                         }
  {   Wartosc - zawiera wartosc danej stalej                      }

  PStala = ^TStala;
  TStala = object (TElement)
    Wartosc : Extended;
    constructor Init(Anazwa : string; Aid : Word; Awartosc : Extended);
    constructor Load(var S : TStream);
    procedure Store(var S : TStream); virtual;
  end;


  { Typ TFunkcja reprezentuje jeden obiekt, tzn. funkcje zdefiniowana }
  { przez uzytkownika.                                                }
  { Obiekt ten nie ma dodatkowych pol.                                }

  PFunkcja = ^TFunkcja;
  TFunkcja = object (TElement)
  end;


  { Typ TOperator reprezentuje jeden obiekt, tzn. operator zdefiniowany }
  { przez uzytkownika.                                                  }
  { Pola:                                                               }
  {   Priorytet - okresla priorytet danego operatora wzgledem innych    }

  POperator = ^TOperator;
  TOperator = object (TElement)
    Priorytet : Byte;
    constructor Init(Anazwa : string; Aid : Word; Apriorytet : Byte);
    constructor Load(var S : TStream);
    procedure Store(var S : TStream); virtual;
  end;


  { Obiekt TAnalizator spelnia funkcje kompilatora. Analizuje podany }
  { ciag znakow i przeksztalca go na swoj wlasny jezyk, tak aby      }
  { korzystanie z juz tak utworzonego kodu bylo bardzo latwe.        }
  { Pola:                                                            }
  {   Zmienne     - pole zawierajace wskaznik na kolekcje obiektow   }
  {                 typu TZmienna                                    }
  {   Stale       - pole zawierajace wskaznik na kolekcje obiektow   }
  {                 typu TStala                                      }
  {   Funkcje     - pole zawierajace wskaznik na kolekcje obiektow   }
  {                 typu TFunkcja                                    }
  {   Operatory   - pole zawierajace wskaznik na kolekcje obiektow   }
  {                 typu TOperator                                   }
  {   TabStalych  - wskaznik na tablice stalych wartosci             }
  {   WskTabStalych - aktualny numer nastepnego elementu TabStalych  }
  {   KodWstepny  - wskaznik na tablice kodu wstepnego               }
  {   WskKodWst   - aktualny numer nastepnego elementu KodWstepny    }
  {   Pozycja     - aktualna pozycja w analizowanym lancuchu         }
  {   prior       - prywatna zmienna zawierajaca ilosc otw. nawiasow }

  PAnalizator = ^TAnalizator;
  TAnalizator = object (TObject)
    Zmienne, Stale,
      Funkcje, Operatory : PCollection;
    TabStalych           : PTablicaLiczb;
    WskTabStalych        : Word;
    Wyrazenie            : PString;
    KodWstepny           : PKodWstepny;
    WskKodWst            : Word;
    Pozycja              : Byte;
    private
    prior                : Byte;
    public
    constructor Init(Awyrazenie : string);
    constructor Load(var S : TStream);
    destructor Done; virtual;
    procedure Store(var S : TStream); virtual;
    procedure Analizuj;
    procedure CzytajLiczbe;
    function CzytajLancuch : string;
    procedure CzytajOperator;
    procedure CzytajSkladnik;
    procedure CzytajWyrazenie;
    procedure Dodaj(Atyp : TKod; Anr, Apriorytet : Word);
    procedure DodajStala(liczba : Extended);
    function Identyfikator : Boolean;
    function NastepnyZnak(przesuniecie : Boolean) : Char;
  end;


  { Typ TKoder sluzy do zakodowania w odpowiedni sposob kodu wstepnego      }
  { (idea kodowania opiera sie na obliczaniu danych w jezyku FORTH, gdzie   }
  { stosuje sie stos do obliczen matematycznych). Typ ten posiada wszystkie }
  { niezbedne metody do obslugi elementow takich jak: zmienne, funkcje      }
  { operatory i stale. Wykorzystuja go wszystkie inne typy powodujace       }
  { obliczanie jakichkolwiek wyrazen pokrywajac jedynie kilka metod.        }
  { Pola:                                                                   }
  {   Analizator - lacznik z obiektem analizujacym ciag znakow wejsciowych  }
  {   Kod        - wskaznik na tablice zawierajaca kod wynikowy             }
  {   WskKodu    - aktualny numer nastepnego elementu tablicy Kod           }

  PKoder = ^TKoder;
  TKoder = object (TObject)
    Analizator : PAnalizator;
    Kod        : PTablicaKodu;
    WskKodu    : Word;
    constructor Init(wyrazenie : string);
    constructor Load(var S : TStream);
    destructor Done; virtual;
    procedure Store(var S : TStream); virtual;
    procedure Analizuj;
    procedure UtworzKod; virtual;
    procedure NoweWyrazenie(wyrazenie : string);
    procedure DodajElementy; virtual;
    function WynikFunkcji(nr : Word; arg : Extended) : Extended; virtual;
    function WynikOperatora(nr : Word; arg1, arg2 : Extended) : Extended;
      virtual;
    function DodajZmienna(nazwa : string; id : Word) : Boolean;
    function DodajStala(nazwa : string; id : Word; wartosc : Extended) :
      Boolean;
    function DodajFunkcje(nazwa : string; id : Word) : Boolean;
    function DodajOperator(nazwa : string; id : Word; priorytet : Byte) : Boolean;
    function Rodzaj(typ : Byte) : PCollection;
    function Usun(typ : Byte; nazwa : string) : Boolean;
    function Blokuj(typ : Byte; nazwa : string) : Boolean;
    function Odblokuj(typ : Byte; nazwa : string) : Boolean;
    function Istnieje(typ : Byte; nazwa : string) : Boolean;
    function ZmienNazwe(typ : Byte; NazwaZrodlowa, NazwaDocelowa : string) :
      Boolean;
    function ZmienId(typ : Byte; nazwa : string; id : Word) : Boolean;
    function ZmienPriorytet(nazwa : string; priorytet : Byte) : Boolean;
    private
    function IsLongInt(var x : Extended) : Boolean;
  end;


  { Typ TWyrazenie jest typem abstrakcyjnym. Posiada on w sobie metody }
  { umozliwiajace obliczanie danego wyrazenia analizujac kod wynikowy  }
  { utworzony przez przodka TKoder, dodatkowo modyfikujac go aby obli- }
  { czanie trwalo szybciej. Wyprowadzone z tego typu obiekty beda      }
  { zazwyczaj pokrywaly od jednej do trzech metod.                     }
  { Pola:                                                              }
  {   TablicaZmiennych - tablica zawierajaca wartosci poszczegolnych   }
  {                      zmiennych                                     }
  {   IloscZmiennych   - aktualna ilosc zmiennych                      }

  PWyrazenie = ^TWyrazenie;
  TWyrazenie = object (TKoder)
    TablicaZmiennych : PTablicaLiczb;
    IloscZmiennych   : Word;
    constructor Load(var S : TStream);
    destructor Done; virtual;
    procedure Store(var S : TStream); virtual;
    procedure UtworzKod; virtual;
    function Oblicz : Extended;
    function NowaWartosc(id : Word; wartosc : Extended) : Boolean;
  end;


  { Typ TWyrPodst realizuje obliczanie wyrazen posiadajacych nieskomplikowana }
  { budowe, tzn. kilka podstawowych funkcji, operatory *, /, +, -, ^, mod,    }
  { div.                                                                      }

  PWyrPodst = ^TWyrPodst;
  TWyrPodst = object (TWyrazenie)
    procedure DodajElementy; virtual;
    function WynikFunkcji(nr : Word; arg : Extended) : Extended; virtual;
    function WynikOperatora(nr : Word; arg1, arg2 : Extended) : Extended;
      virtual;
  end;


  { Typ TWyrLog realizuje dodatkowo obliczanie wyrazen logicznych, tzn.   }
  { dopuszcza dzialania na operatorach takich jak >, <, =, <>, #, !=, and }
  { or, >=, <=.                                                           }

  PWyrLog = ^TWyrLog;
  TWyrLog = object (TWyrPodst)
    procedure DodajElementy; virtual;
    function WynikOperatora(nr : Word; arg1, arg2 : Extended) : Extended;
      virtual;
  end;


  { Typ TFunkcjeTryg posiada wszelkie cechy swoich przodkow dodatkowo }
  { umozliwiajac obliczanie dowolnych funkcji trygonometrycznych.     }
  { Obiekt ten dodaje tez dwie stale: Pi, oraz E.                     }

  PFunkcjeTryg = ^TFunkcjeTryg;
  TFunkcjeTryg = object (TWyrLog)
    procedure DodajElementy; virtual;
    function WynikFunkcji(nr : Word; arg : Extended) : Extended; virtual;
  end;


{ typy rejestracyjne wszystkich powyzszych obiektow (od 65000 do 65010) }

const

  RElement : TStreamRec = (
    ObjType : 65000;
    VmtLink : Ofs(TypeOf(TElement)^);
    Load    : @TElement.Load;
    Store   : @TElement.Store);


  RZmienna : TStreamRec = (
    ObjType : 65001;
    VmtLink : Ofs(TypeOf(TZmienna)^);
    Load    : @TZmienna.Load;
    Store   : @TZmienna.Store);


  RStala : TStreamRec = (
    ObjType : 65002;
    VmtLink : Ofs(TypeOf(TStala)^);
    Load    : @TStala.Load;
    Store   : @TStala.Store);


  RFunkcja : TStreamRec = (
    ObjType : 65003;
    VmtLink : Ofs(TypeOf(TFunkcja)^);
    Load    : @TFunkcja.Load;
    Store   : @TFunkcja.Store);


  ROperator : TStreamRec = (
    ObjType : 65004;
    VmtLink : Ofs(TypeOf(TOperator)^);
    Load    : @TOperator.Load;
    Store   : @TOperator.Store);


  RAnalizator : TStreamRec = (
    ObjType : 65005;
    VmtLink : Ofs(TypeOf(TAnalizator)^);
    Load    : @TAnalizator.Load;
    Store   : @TAnalizator.Store);


  RKoder : TStreamRec = (
    ObjType : 65006;
    VmtLink : Ofs(TypeOf(TKoder)^);
    Load    : @TKoder.Load;
    Store   : @TKoder.Store);


  RWyrazenie : TStreamRec = (
    ObjType : 65007;
    VmtLink : Ofs(TypeOf(TWyrazenie)^);
    Load    : @TWyrazenie.Load;
    Store   : @TWyrazenie.Store);


  RWyrPodst : TStreamRec = (
    ObjType : 65008;
    VmtLink : Ofs(TypeOf(TWyrPodst)^);
    Load    : @TWyrPodst.Load;
    Store   : @TWyrPodst.Store);


  RWyrLog : TStreamRec = (
    ObjType : 65009;
    VmtLink : Ofs(TypeOf(TWyrLog)^);
    Load    : @TWyrLog.Load;
    Store   : @TWyrLog.Store);


  RFunkcjeTryg : TStreamRec = (
    ObjType : 65010;
    VmtLink : Ofs(TypeOf(TFunkcjeTryg)^);
    Load    : @TFunkcjeTryg.Load;
    Store   : @TFunkcjeTryg.Store);


const


  { Stale rodzajow obiektow wykorzystywane przy operacjach na nich. }

  teZmienna  = $00;  { stala oznaczajaca obiekt typu TZmienna  }
  teStala    = $01;  { stala oznaczajaca obiekt typu TStala    }
  teFunkcja  = $02;  { stala oznaczajaca obiekt typu TFunkcja  }
  teOperator = $03;  { stala oznaczajaca obiekt typu TOperator }


  { Stale kodow bledow powstalych podczas analizy ciagu wejsciowego znakow }
  { lub podczas obliczania danego wyrazenia (np. dzielenie przez zero).    }

  bsOk                 = 00;  { operacja przebiegla pomyslnie          }
  bsBrakWyrazenia      = 01;  { brak wyrazenia do analizowania         }
  bsBrakPamieci        = 02;  { niedostateczna wielkosc wolnej pamieci }
  bsBlednaLiczba       = 03;  { bledny format liczby w lancuchu wejsc. }
  bsBlednyOperator     = 04;  { nieznany operator                      }
  bsNiespKoniecLinii   = 05;  { niespodziewany koniec linii            }
  bsBrakOperMnoz       = 06;  { usunieto, badz nie zadeklarowano op. mnozenia }
  bsZbytDuzoNawiasow   = 07;  { liczba nawiasow przekroczyla 253       }
  bsBrakNawiasuZamyk   = 08;  { nie zapisano nawiasu zamykajacego      }
  bsBlednyId           = 09;  { nieznany identyfikator (ew.bledna lb.) }
  bsDzieleniePrzezZero = 10;  { niedozwolone dzielenie przez zero      }
  bsLiczbaUjemna       = 11;  { liczba ujemna w niewlasciwym miejscu   }
  bsZbednyNawiasZamyk  = 12;  { zbedny nawias zamykajacy               }
  bsBlednyArgFunkcji   = 13;  { nieprawidlowy argument funkcji         }
  bsBrakRealizacjiFunk = 14;  { nie zrealizowano obliczenia funkcji    }
  bsBrakRealizacjiOper = 15;  { nie zrealizowano obliczenia operatora  }


const


  { Stale standardowych operatorow znajdujacych sie w poszczegolnych }
  { obiektach.                                                       }

  opMinus            = 00;  { sekcja dla obiektu TWyrPodst }
  opPlus             = 01;
  opMod              = 02;
  opDiv              = 03;
  opDzielenie        = 04;
  opMnozenie         = 05;
  opPotegowanie      = 06;

  opMniejszeLubRowne = 07;  { sekcja dla obiektu TWyrLog }
  opWiekszeLubRowne  = 08;
  opMniejsze         = 09;
  opWieksze          = 10;
  opRowne            = 11;
  opOr               = 12;
  opAnd              = 13;
  opRozne            = 14;


  { Stale standardowych funkcji znajdujacych sie w poszczegolnych obiektach. }

  fnSec    = 00;  { sekcja TFunkcjeTryg }
  fnCSC    = 01;
  fnSqrt   = 02;
  fnTan    = 03;
  fnCTg    = 04;
  fnArcCos = 05;
  fnArcTan = 06;
  fnSin    = 07;
  fnCos    = 08;
  fnArcSin = 09;
  fnArcCTg = 10;
  fnSinH   = 11;
  fnCosH   = 12;
  fnTanH   = 13;
  fnExp    = 14;
  fnLN     = 15;

  fnRound  = 16;  { sekcja TWyrPodst }
  fnAbs    = 17;
  fnInt    = 18;
  fnFrac   = 19;
  fnTrunc  = 20;
  fnOdd    = 21;
  fnSgn    = 22;


  { Stale standardowych stalych zdefiniowanych w obiekcie TFunkcjeTryg. }

  stPi = 00;
  stE  = 01;


function AdresElementu(C : PCollection; nazwa : string; sprawdz : Boolean) :
  PElement;
function AdrElWgId(C : PCollection; id : Word; sprawdz : Boolean) : PElement;
function Status : Byte;
function Komunikat(nr : Byte) : string;
function PozycjaBledu : Byte;
procedure Blad(nr, pozycja : Byte);
procedure RejestrujUFunkcje;

implementation

{ Zmienna KodBledu zawiera aktualny kod bledu lub wartosc bsOk. }
{ Zmienna PozBledu zawiera pozycje bledu w wyrazeniu tekstowym. }

var
  KodBledu, PozBledu : Byte;


{ Stala oznaczajaca wielkosc o jaka zwiekszane lub zmniejszane sa }
{ dynamiczne tablice podczas operacji dodawania lub usuwania ele- }
{ mentow.                                                         }

const
  RMEM = 50;


{ Funkcje wewnetrzne modulu ------------------------------------------- }


Function StrPas(P : PChar) : string; assembler;
asm
   push ds
   lds  si, p
   les  di, @Result
   inc  di
   mov  cx, 255
@hop:
   lodsb
   cmp  al, 00h
   je   @koniec
   stosb
   loop @hop
@koniec:
   mov  ax, 255
   sub  ax, cx
   les  di, @Result
   mov  byte ptr es:[di], al
   pop  ds
end;


{ Zwiekszenie indeksu dynamicznej tablicy. }

Function Incr(P : Pointer; Size : Word; var Counter : Word;
  Delta : Word) : Pointer;
var
  r  : Word;
  p1 : Pointer;
begin
  Inc(Counter);
  p1:=P;
  if Counter mod Delta=0 then
    begin
      r:=Counter div Delta;
      GetMem(p1, Size*Delta*(r+1));
      if p1<>nil then
        begin
          Move(P^, p1^, Size*Delta*r);
          FreeMem(P, Size*Delta*r)
        end
       else
        begin
          Dec(Counter);
          p1:=nil
        end
    end;
  Incr:=p1
end;


{ Zmniejszenie indeksu dynamicznej tablicy. }

Function Decr(P : Pointer; Size : Word; var Counter : Word;
  Delta : Word) : Pointer;
var
  r  : Word;
  p1 : Pointer;
begin
  Dec(Counter);
  p1:=P;
  if Counter mod Delta=Delta-1 then
    begin
      r:=Counter div Delta;
      GetMem(p1, Size*Delta*r);
      if p1<>nil then
        begin
          Move(P^, p1^, Size*Delta*r);
          FreeMem(P, Size*Delta*(r+1))
        end
       else
        begin
          Inc(Counter);
          p1:=nil
        end
    end;
  Decr:=p1
end;


{ Przydzielenie pamieci dla dynamicznej tablicy. }

Function Przydziel(P : Pointer; Size : Word; var Counter : Word;
  Delta : Word) : Pointer;
begin
  GetMem(P, Size*Delta*(Counter div Delta+1));
  Przydziel:=P
end;


{ Zwolnienie pamieci zajetej przez dynamiczna tablice. }

Function Zwolnij(P : Pointer; Size : Word; var Counter : Word;
  Delta : Word) : Pointer;
begin
  if P<>nil then
    FreeMem(P, Size*(Counter div Delta+1)*Delta);
  Zwolnij:=nil;
  Counter:=1
end;


{ Przeksztalcenie wszystkich malych liter w lancuchu do duzych. }

Function UpStr(s : string) : string;
var
  i : Byte;
begin
  for i:=1 to Length(s) do s[i]:=UpCase(s[i]);
  UpStr:=s
end;


{ Zwrocenie wartosci funkcji SGN (znak); -1 dla x<0; 1 dla x>0; 0 dla x=0. }

Function Sgn(x : Extended) : ShortInt;
begin
  if x<0 then Sgn:=-1 else Sgn:=Ord(x>0)
end;


{ Usuniecie wszystkich spacji i znakow tabulacji z lancucha znakowego. }

Procedure ClearSpaces(var s : string);
var
  s1 : string;
  i  : Byte;
begin
  s1:='';
  for i:=1 to Length(S) do
    if (s[i]<>#32) and (s[i]<>#9) then s1:=s1+s[i];
  s:=s1
end;


{ Stwierdzenie poprawnosci danego elementu (funkcja, stala, zmienna). Nie }
{ odnosi sie do sprawdzania operatorow.                                   }

Function PoprawnoscElementu(var nazwa : string) : Boolean;
var
  i : Byte;
begin
  PoprawnoscElementu:=False;
  if (nazwa='') or (nazwa[1] in ['0'..'9']) then Exit;
  nazwa:=UpStr(nazwa);
  for i:=1 to Length(nazwa) do
    if not (nazwa[i] in ['0'..'9', 'A'..'Z', '_']) then Exit;
  PoprawnoscElementu:=True
end;


{ Stwierdzenie poprawnosci nazwy operatora. }

Function PoprawnyOperator(var nazwa : string) : Boolean;
var
  i : Byte;
begin
  PoprawnyOperator:=False;
  if nazwa='' then Exit;
  nazwa:=UpStr(nazwa);
  if nazwa[1] in ['0'..'9', 'A'..'Z', '_'] then
    begin
      for i:=2 to Length(nazwa) do
        if not (nazwa[i] in ['0'..'9', 'A'..'Z', '_']) then Exit
    end
   else
    for i:=2 to Length(nazwa) do
      if nazwa[i] in ['0'..'9', 'A'..'Z', '_', ' ', '(', ')'] then Exit;
  PoprawnyOperator:=True
end;


{ Funkcje globalne modulu --------------------------------------- }


{ Zarejestrowanie wszystkich obiektow modulu w celu umozliwienia ich }
{ pozniejszego odczytu/zapisu z/do strumienia.                       }

Procedure RejestrujUFunkcje;
begin
  RegisterType(RElement);
  RegisterType(RZmienna);
  RegisterType(RStala);
  RegisterType(RFunkcja);
  RegisterType(ROperator);
  RegisterType(RAnalizator);
  RegisterType(RKoder);
  RegisterType(RWyrazenie);
  RegisterType(RWyrPodst);
  RegisterType(RWyrLog);
  RegisterType(RFunkcjeTryg)
end;


{ Pobranie adresu elementu o nazwie NAZWA z kolekcji zawierajacej }
{ elementy typu PElement lub jego potomkow. Jezeli parametr       }
{ SPRAWDZ=True, to nastepuje sprawdzenie, czy dany element        }
{ jest zablokowany, w przeciwnym przypadku taki test nie jest     }
{ przeprowadzany.                                                 }

Function AdresElementu(C : PCollection; nazwa : string; sprawdz : Boolean) :
  PElement;

  Function Testuj(P : Pointer) : Boolean; far;
  var
    El : PElement;
  begin
    El:=PElement(P);
    if (El^.Nazwa^=nazwa) and (not Sprawdz or
      (Sprawdz and (El^.Stan=True))) then
      Testuj:=True
     else
      Testuj:=False
  end;

begin
  if C<>nil then
    AdresElementu:=C^.FirstThat(@Testuj)
   else
    AdresElementu:=nil
end;


{ Pobranie adresu elementu o identyfikatorze ID z kolekcji zawierajacej }
{ elementy typu PElement lub jego potomkow.                             }

Function AdrElWgId(C : PCollection; id : Word; sprawdz : Boolean) : PElement;

  Function Testuj(P : Pointer) : Boolean; far;
  var
    El : PElement;
  begin
    El:=PElement(P);
    if (El^.Id=id) and (not Sprawdz or
      (Sprawdz and (El^.Stan=True))) then
      Testuj:=True
     else
      Testuj:=False
  end;

begin
  if C<>nil then
    AdrElWgId:=C^.FirstThat(@Testuj)
   else
    AdrElWgId:=nil
end;


{ Funkcje obslugi bledow analizy, badz tez wykonania programu. ----------- }


{ Funkcja zwracajaca komunikat o bledzie w postaci lancucha znakow. }

Function Komunikat(nr : Byte) : string;
const
  mess : array [0..15] of PChar =
    ('Ok', 'Brak wyrazenia', 'Brak pamieci', 'Bledna liczba',
     'Bledny operator', 'Niespodziewany koniec linii',
     'Niezdefiniowany operator mnozenia', 'Zbyt duzo nawiasow',
     'Brak nawiasu zamykajacego', 'Bledny identyfikator',
     'Dzielenie przez zero', 'Liczba ujemna', 'Zbedny nawias zamykajacy',
     'Nieprawidlowy argument funkcji', 'Nie zrealizowano obslugi funkcji',
     'Nie zrealizowano obslugi operatora');
begin
  if nr in [1..15] then
    Komunikat:=StrPas(mess[nr])
   else
    Komunikat:='Nieznany blad'
end;


{ Funkcja zwracajaca pozycje bledu w lancuchu tekstowym wyrazenia, jezeli }
{ taki wystapil.                                                          }

Function PozycjaBledu : Byte;
begin
  PozycjaBledu:=PozBledu
end;


{ Procedura przypisujaca zmiennej KodBledu, kod bledu wykonania. }

Procedure Blad(nr, pozycja : Byte);
begin
  KodBledu:=nr;
  PozBledu:=pozycja
end;


{ Funkcja Status zwraca kod bledu i jednoczesnie zeruje jego zawartosc, }
{ aby mogly byc przeprowadzane dalsze operacje.                         }

Function Status : Byte;
begin
  Status:=KodBledu;
  KodBledu:=bsOk
end;


{ TElement ---------------------------------------------------- }


{ Konstruktor inicjujacy jeden element (np. zmienna). }

Constructor TElement.Init(Anazwa : string; Aid : Word);
begin
  inherited Init;
  Nazwa:=NewStr(UpStr(Anazwa));
  Id:=Aid;
  Stan:=True
end;


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TElement.Load(var S : TStream);
begin
  Nazwa:=S.ReadStr;
  S.Read(Id, SizeOf(Id));
  S.Read(Stan, SizeOf(Stan))
end;


{ Destruktor zwalniajacy przydzielona pamiec dla nazwy elementu. }

Destructor TElement.Done;
begin
  if Nazwa<>nil then DisposeStr(Nazwa);
  inherited Done
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TElement.Store(var S : TStream);
begin
  S.WriteStr(Nazwa);
  S.Write(Id, SizeOf(Id));
  S.Write(Stan, SizeOf(Stan))
end;


{ TStala ------------------------------------------------------------- }


{ Konstruktor dla obiektu typu TStala umozliwiajacy wlaczenie wartosci }
{ danej stalej.                                                        }

Constructor TStala.Init(Anazwa : string; Aid : Word; Awartosc : Extended);
begin
  inherited Init(Anazwa, Aid);
  Wartosc:=Awartosc
end;


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TStala.Load(var S : TStream);
begin
  inherited Load(S);
  S.Read(Wartosc, SizeOf(Wartosc))
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TStala.Store(var S : TStream);
begin
  inherited Store(S);
  S.Write(Wartosc, SizeOf(Wartosc))
end;


{ TZmienna ----------------------------------------------------------- }


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TZmienna.Load(var S : TStream);
begin
  inherited Load(S);
  S.Read(nr_pom, SizeOf(nr_pom))
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TZmienna.Store(var S : TStream);
begin
  inherited Store(S);
  S.Write(nr_pom, SizeOf(nr_pom))
end;


{ TOperator -------------------------------------------------------- }


{ Konstruktor dla obiektu TOperator umozliwiajacy dodanie priorytetu }
{ danego operatora.                                                  }

Constructor TOperator.Init(Anazwa : string; Aid : Word; Apriorytet : Byte);
begin
  inherited Init(Anazwa, Aid);
  Priorytet:=Apriorytet
end;


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TOperator.Load(var S : TStream);
begin
  inherited Load(S);
  S.Read(Priorytet, SizeOf(Priorytet))
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TOperator.Store(var S : TStream);
begin
  inherited Store(S);
  S.Write(Priorytet, SizeOf(Priorytet))
end;


{ TAnalizator ---------------------------------------------------- }


{ Konstruktor inicjujacy obiekt typu TAnalizator. Wszystkie pola zostaja    }
{ zainicjowane, a w szczegolnosci kolekcje elementow (zmienne, stale itd.). }
{ Podstawowy test poprawnosci danych.                                       }

Constructor TAnalizator.Init(Awyrazenie : string);
begin
  inherited Init;
  Wyrazenie:=NewStr(UpStr(Awyrazenie));
  Zmienne:=New(PCollection, Init(50, 10));
  Stale:=New(PCollection, Init(50, 10));
  Funkcje:=New(PCollection, Init(50, 10));
  Operatory:=New(PCollection, Init(50, 10));
  KodBledu:=bsOk;
  WskTabStalych:=1;
  WskKodWst:=1;
  ClearSpaces(Awyrazenie);
  if (Awyrazenie<>'') and (Wyrazenie=nil) or (Zmienne=nil) or (Stale=nil) or
    (Funkcje=nil) or (Operatory=nil) then Blad(bsBrakPamieci, 1)
end;


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TAnalizator.Load(var S : TStream);
begin
  Zmienne:=PCollection(S.Get);
  Stale:=PCollection(S.Get);
  Funkcje:=PCollection(S.Get);
  Operatory:=PCollection(S.Get);
  S.Read(WskTabStalych, SizeOf(WskTabStalych));
  TabStalych:=Przydziel(TabStalych, WlkLiczby, WskTabStalych, RMEM);
  S.Read(TabStalych^, (WskTabStalych-1)*WlkLiczby);
  Wyrazenie:=S.ReadStr;
  S.Read(WskKodWst, SizeOf(WskKodWst));
  KodWstepny:=Przydziel(KodWstepny, WlkElKoduWst, WskKodWst, RMEM);
  S.Read(KodWstepny^, (WskKodWst-1)*WlkElKoduWst);
  S.Read(Pozycja, SizeOf(Pozycja));
  S.Read(Prior, SizeOf(Prior))
end;


{ Zwolnienie pamieci dla elementow obiektu typu TAnalizator. }

Destructor TAnalizator.Done;
begin
  TabStalych:=Zwolnij(TabStalych, WlkLiczby, WskTabStalych, RMEM);
  if Wyrazenie<>nil then DisposeStr(Wyrazenie);
  KodWstepny:=Zwolnij(KodWstepny, WlkElKoduWst, WskKodWst, RMEM);
  if Zmienne<>nil then Dispose(Zmienne, Done);
  if Stale<>nil then Dispose(Stale, Done);
  if Funkcje<>nil then Dispose(Funkcje, Done);
  if Operatory<>nil then Dispose(Operatory, Done);
  inherited Done
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TAnalizator.Store(var S : TStream);
begin
  S.Put(Zmienne);
  S.Put(Stale);
  S.Put(Funkcje);
  S.Put(Operatory);
  S.Write(WskTabStalych, SizeOf(WskTabStalych));
  S.Write(TabStalych^, (WskTabStalych-1)*WlkLiczby);
  S.WriteStr(Wyrazenie);
  S.Write(WskKodWst, SizeOf(WskKodWst));
  S.Write(KodWstepny^, (WskKodWst-1)*WlkElKoduWst);
  S.Write(Pozycja, SizeOf(Pozycja));
  S.Write(Prior, SizeOf(Prior))
end;


{ Procedura inicjujaca analize lancucha znakowego. Przydzielenie pamieci }
{ dla niezbednych pol obiektu TAnalizator. Wstepny test poprawnosci      }
{ wykonanych operacji. Uruchomienie wlasciwej procedury analizujacej.    }

Procedure TAnalizator.Analizuj;
begin
  pozycja:=0;
  prior:=0;
  Zwolnij(TabStalych, WlkLiczby, WskTabStalych, RMEM);
  TabStalych:=Przydziel(TabStalych, WlkLiczby, WskTabStalych, RMEM);
  if TabStalych=nil then
    begin
      Blad(bsBrakPamieci, 1);
      Exit
    end;
  Zwolnij(KodWstepny, WlkElKoduWst, WskKodWst, RMEM);
  KodWstepny:=Przydziel(KodWstepny, WlkElKoduWst, WskKodWst, RMEM);
  if KodWstepny=nil then
    begin
      Blad(bsBrakPamieci, 1);
      Exit
    end;
  if Wyrazenie<>nil then CzytajWyrazenie;
  Dodaj(reKoniec, 0, 0)
end;


{ Wyciagniecie liczby z lancucha znakowego z danej pozycji. Wlaczenie stalej }
{ do dynamicznej tablicy stalych wartosci. Testowanie poprawnosci danych.    }

Procedure TAnalizator.CzytajLiczbe;
var
  liczba : Extended;
  p, p1  : Integer;
begin
  Val(Copy(Wyrazenie^, pozycja, 255), liczba, p);
  p1:=p;
  if p<>0 then
    begin
      Val(Copy(Wyrazenie^, pozycja, p-1), liczba, p);
      if p<>0 then
        Blad(bsBlednaLiczba, pozycja)
       else
        pozycja:=pozycja+p1-2
    end
   else
    pozycja:=Length(wyrazenie^);
  DodajStala(liczba)
end;


{ Wylaczenie ciagu znakow zgodnie z zasadami obowiazujacymi identyfi- }
{ katory. Funkcja nie operuje na wylaczonym lancuchu.                 }

Function TAnalizator.CzytajLancuch : string;
var
  s : string;
begin
  s:='';
  if Pos(Copy(Wyrazenie^, pozycja, 1), '0123456789')=0 then
    while (pozycja<=Length(Wyrazenie^)) and
      (Wyrazenie^[pozycja] in ['0'..'9', 'A'..'Z', '_']) do
        begin
          s:=s+Wyrazenie^[pozycja];
          Inc(pozycja)
        end;
  Dec(pozycja);
  CzytajLancuch:=s
end;


{ Osobna procedura odczytujaca operator, analizujaca jego skladnie, }
{ stwierdzajaca istnienie i dolaczajaca go do kodu wstepnego.       }
{ Przechwytywanie bledow.                                           }

Procedure TAnalizator.CzytajOperator;
var
  s      : string;
  P      : POperator;
begin
  s:=NastepnyZnak(True);
  Blad(bsOk, pozycja);
  Inc(pozycja);
  if pozycja<Length(Wyrazenie^) then
    if s[1] in ['0'..'9', 'A'..'Z', '_'] then
      while (pozycja<=Length(Wyrazenie^)) and (wyrazenie^[pozycja] in
        ['0'..'9', 'A'..'Z', '_']) do
          begin
            s:=s+Wyrazenie^[pozycja];
            Inc(pozycja)
          end
         else
          while (pozycja<=Length(Wyrazenie^)) and not (wyrazenie^[pozycja] in
            ['0'..'9', 'A'..'Z', '_', ' ', '(', ')', '+', '-']) do
              begin
                s:=s+Wyrazenie^[pozycja];
                Inc(pozycja)
              end;
  Dec(pozycja);
  if s=')' then
    begin
      Blad(bsZbednyNawiasZamyk, pozycja);
      Exit
    end;
  P:=POperator(AdresElementu(Operatory, s, True));
  if P=nil then
    Blad(bsBlednyOperator, PozBledu)
   else
    if pozycja>Length(Wyrazenie^) then
      Blad(bsNiespKoniecLinii, Length(Wyrazenie^)+1)
     else
      Dodaj(reOperator, P^.id, P^.priorytet+prior shl 8)
end;


{ Przeczytanie jednego skladnika, ktory moze byc: znakiem, liczba, }
{ identyfikatorem stalej, zmiennej lub funkcji, badz blokiem       }
{ skladnikow ograniczonych nawiasami. Obsluga bledow.              }

Procedure TAnalizator.CzytajSkladnik;
var
  P : POperator;
begin
  case NastepnyZnak(True) of
    #13 : ;
    '+' :
      begin
        if NastepnyZnak(False)=#13 then
          Blad(bsNiespKoniecLinii, pozycja+1)
         else
          CzytajSkladnik
      end;
    '-' :
      begin
        P:=POperator(AdrElWgId(Operatory, opMnozenie, True));
        if P=nil then
          begin
            Blad(bsBrakOperMnoz, pozycja);
            Exit
          end;
        DodajStala(-1);
        Dodaj(reOperator, opMnozenie, P^.priorytet+(prior+1) shl 8);
        if NastepnyZnak(False)=#13 then
          Blad(bsNiespKoniecLinii, pozycja+1)
         else
          CzytajSkladnik
      end;
    '(' :
      begin
        if prior>=$FE then
          begin
            Blad(bsZbytDuzoNawiasow, pozycja);
            Exit
          end;
        Inc(prior);
        CzytajWyrazenie;
        if KodBledu<>bsOk then Exit;
        Dec(prior);
        if NastepnyZnak(True)<>')' then Blad(bsBrakNawiasuZamyk, pozycja)
      end;
    '0'..'9' : CzytajLiczbe;
    else
      begin
        Blad(bsOk, pozycja);
        if not Identyfikator then Blad(bsBlednyId, PozBledu)
      end
  end
end;


{ Przeczytanie calego wyrazenia (a nie calego ciagu znakow). Stwierdzenie }
{ poprawnosci danych.                                                     }

Procedure TAnalizator.CzytajWyrazenie;
begin
  repeat
    if (NastepnyZnak(False)=#13) and (prior>0) then Break;
    CzytajSkladnik;
    if (NastepnyZnak(False)=')') and (prior>0) or
      (NastepnyZnak(False)=#13) or (KodBledu<>bsOk) then Break;
    CzytajOperator;
    if (NastepnyZnak(False)=#13) and (KodBledu=bsOk) then
      Blad(bsNiespKoniecLinii, pozycja+1)
  until KodBledu<>bsOk
end;


{ Procedura realizujaca zadanie dodania jednego elementu do kodu wstepnego }
{ (stala, zmienna, funkcja badz tez operator). Uwaga! Funkcja ma zawsze    }
{ najwiekszy priorytet rowny 255, dlatego zaden operator nie moze miec     }
{ takiego priorytetu!                                                      }

Procedure TAnalizator.Dodaj(Atyp : TKod; Anr, Apriorytet : Word);
begin
  with KodWstepny^[WskKodWst] do
    begin
      typ:=Atyp;
      nr:=Anr;
      priorytet:=Apriorytet
    end;
  KodWstepny:=Incr(KodWstepny, WlkElKoduWst, WskKodWst, RMEM)
end;


{ Procedura uozliwiajaca dodanie stalej do tablicy wartosci stalych }
{ oraz elementu stalej do kodu wstepnego.                           }

Procedure TAnalizator.DodajStala(liczba : Extended);
begin
  Dodaj(reStala, WskTabStalych, 0);
  TabStalych^[WskTabStalych]:=liczba;
  TabStalych:=Incr(TabStalych, WlkLiczby, WskTabStalych, RMEM);
end;


{ Stwierdzenie istnienia identyfikatora (stala, zmienna, funkcja /bez   }
{ operatora/). Zrealizowanie dodania danego elementu do kodu wstepnego. }
{ Obsluga sytuacji blednych. Funkcja zwraca True, jesli zostal rozpo-   }
{ znany chociaz jeden z wszystkich identyfikatorow w/w.                 }

Function TAnalizator.Identyfikator : Boolean;
var
  nazwa : string;
  P     : PElement;
begin
  Identyfikator:=True;
  nazwa:=CzytajLancuch;
  P:=AdresElementu(Zmienne, nazwa, True);
  if P<>nil then
    begin
      Dodaj(reZmienna, P^.id, 0);
      Exit
    end;
  P:=AdresElementu(Stale, nazwa, True);
  if P<>nil then
    begin
      DodajStala(PStala(P)^.wartosc);
      Exit
    end;
  P:=AdresElementu(Funkcje, nazwa, True);
  if P<>nil then
    begin
      if NastepnyZnak(False)<>#13 then
        begin
          Dodaj(reFunkcja, P^.id, $FF+prior shl 8);
          CzytajSkladnik;
          Exit
        end
       else
        Blad(bsNiespKoniecLinii, pozycja+1);
      Exit
    end;
  Identyfikator:=False
end;


{ Funkcja realizuje przesuniecie wskaznika na nastepny element w zaleznosci }
{ od paramatru PRZESUNIECIE i zwraca znak, ktory zostal odczytany na nowej  }
{ pozycji niezaleznie od parametru PRZESUNIECIE.                            }

Function TAnalizator.NastepnyZnak(przesuniecie : Boolean) : Char;
var
  ch : Char;
begin
  Inc(pozycja);
  if pozycja>Length(Wyrazenie^)then
    ch:=#13
   else
    if Wyrazenie^[pozycja]=' ' then
      ch:=NastepnyZnak(przesuniecie)
     else
      ch:=Wyrazenie^[pozycja];
  NastepnyZnak:=ch;
  if not przesuniecie then Dec(pozycja)
end;


{ TKoder ---------------------------------------------------------- }


{ Inicjacja obiektu typu TKoder. Przydzielenie pamieci dla pola ANALIZATOR }
{ oraz zainicjowanie go. Wywolanie wirtualnej procedury dodania elementow  }
{ takich jak stale, zmienne, funkcje oraz operatory. Podstawowy test.      }

Constructor TKoder.Init(wyrazenie : string);
begin
  inherited Init;
  WskKodu:=1;
  Analizator:=New(PAnalizator, Init(wyrazenie));
  if Analizator=nil then Blad(bsBrakPamieci, 1) else DodajElementy
end;


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TKoder.Load(var S : TStream);
begin
  Analizator:=PAnalizator(S.Get);
  S.Read(WskKodu, SizeOf(WskKodu));
  Kod:=Przydziel(Kod, WlkElKodu, WskKodu, RMEM);
  S.Read(Kod^, (WskKodu-1)*WlkElKodu)
end;


{ Zwolnienie pamieci dla pola ANALIZATOR oraz jego destrukcja, a takze }
{ dla tablicy kodu wynikowego oraz wywolanie przodka typu TKoder.      }

Destructor TKoder.Done;
begin
  if Analizator<>nil then Dispose(Analizator, Done);
  Kod:=Zwolnij(Kod, WlkElKodu, WskKodu, RMEM);
  inherited Done
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TKoder.Store(var S : TStream);
begin
  S.Put(Analizator);
  S.Write(WskKodu, SizeOf(WskKodu));
  S.Write(Kod^, (WskKodu-1)*WlkElKodu)
end;


{ Sprawdzenie znajdowania sie wartosci w zakresie zmiennych typu LongInt. }

Function TKoder.IsLongInt(var x : Extended) : Boolean;
var
  w : Boolean;
begin
  w:=(x<=MaxLongInt) and (x>=-MaxLongInt);
  if not w then Blad(bsBlednaLiczba, 0);
  IsLongInt:=w
end;


{ Procedura tworzy kod wynikowy na podstawie wczesniej przygotowanego   }
{ kodu wstepnego. Podczas tworzenia kodu sa dokonywane uproszczenia,    }
{ w celu osiagniecia mniejszego i bardziej efektywnego kodu wynikowego. }

Procedure TKoder.UtworzKod;


  { Dodanie do kodu wynikowego jednego elementu. }

  Procedure DodajDoKodu(Atyp : TKod; Anr : Word);
  begin
    with Kod^[WskKodu] do
      begin
        nr:=Anr; typ:=Atyp
      end;
    Kod:=Incr(Kod, WlkElKodu, WskKodu, RMEM)
  end;


  { Usuniecie z kodu wstepnego juz przeanalizowanego elementu. }

  Procedure UsunZKoduWst(nr : Word);
  begin
    with Analizator^ do
      begin
        repeat
          KodWstepny^[nr]:=KodWstepny^[nr+1];
          Inc(nr)
        until KodWstepny^[nr].typ=reKoniec;
        KodWstepny:=Decr(KodWstepny, WlkElKoduWst, WskKodWst, RMEM)
     end
  end;

var
  wsk, poz, ils : Word;
  koniec        : Boolean;
begin
  ils:=0;
  with Analizator^ do
    repeat
      wsk:=1; poz:=0;
      koniec:=True;
      while KodWstepny^[1].typ<>reKoniec do
        begin
          koniec:=False;

          with KodWstepny^[wsk] do
            case typ of
              reStala, reZmienna :
                begin
                  DodajDoKodu(typ, nr);
                  if typ=reStala then Inc(ils) else ils:=0;
                  UsunZKoduWst(wsk);
                  koniec:=True;
                  Continue
                end;
              reOperator, reFunkcja :
                if (poz=0) or (typ=reFunkcja) and
                  (priorytet>=KodWstepny^[poz].priorytet) or
                  (typ=reOperator) and
                  (priorytet>KodWstepny^[poz].priorytet) then
                  begin
                    poz:=wsk;
                    Inc(wsk);
                    koniec:=True;
                    Continue
                  end
            end;

          with KodWstepny^[poz] do
            begin
              if (ils>0) and (Kod^[WskKodu-1].typ=reStala) and
                (typ=reFunkcja) then
                begin
                  TabStalych^[Kod^[WskKodu-1].nr]:=WynikFunkcji(nr,
                    TabStalych^[Kod^[WskKodu-1].nr]);
                  UsunZKoduWst(poz);
                  Break
                end;
              if (ils>1) and (typ=reOperator) and
                (reStala in [Kod^[WskKodu-1].typ, Kod^[WskKodu-2].typ]) then
                begin
                  TabStalych^[Kod^[WskKodu-2].nr]:=WynikOperatora(nr,
                    TabStalych^[Kod^[WskKodu-2].nr],
                    TabStalych^[Kod^[WskKodu-1].nr]);
                  UsunZKoduWst(poz);
                  Kod:=Decr(Kod, WlkElKodu, WskKodu, RMEM);
                  Dec(ils);
                  Break
                end
            end;

          if poz>0 then
            begin
              with KodWstepny^[poz] do DodajDoKodu(typ, nr);
              UsunZKoduWst(poz)
            end;
          ils:=0;
          Break
        end
    until koniec or (KodBledu<>bsOk);
  DodajDoKodu(reKoniec, 0)
end;


{ Przydzielenie pamieci dla tablicy kodu wynikowego oraz wywolanie }
{ analizatora obiektu TAnalizator.                                 }

Procedure TKoder.Analizuj;
begin
  if Analizator<>nil then
    begin
      Analizator^.Analizuj;
      if KodBledu<>bsOk then Exit;
      Zwolnij(Kod, WlkElKodu, WskKodu, RMEM);
      Kod:=Przydziel(Kod, WlkElKodu, WskKodu, RMEM);
      if Kod<>nil then
        UtworzKod
       else
        Blad(bsBrakPamieci, 1)
    end
end;


{ Wprowadzenie nowego wyrazenia w postaci lancucha znakowego. Destrukcja }
{ oraz ponowne zainicjowanie potrzebnych obiektow.                       }

Procedure TKoder.NoweWyrazenie(wyrazenie : string);
begin
  if Analizator<>nil then
    begin
      if Analizator^.Wyrazenie<>nil then DisposeStr(Analizator^.Wyrazenie);
      Analizator^.Wyrazenie:=NewStr(UpStr(wyrazenie));
      Analizuj
    end
end;


{ Procedura majaca za zadanie dodanie elementow takich jak zmienne, }
{ funkcje, stale i operatory.                                       }

Procedure TKoder.DodajElementy;
begin
end;


{ Zwrocenie wyniku funkcji nr NR z argumentem ARG. }

Function TKoder.WynikFunkcji(nr : Word; arg : Extended) : Extended;
begin
  Blad(bsBrakRealizacjiFunk, 0)
end;


{ Zwrocenie wyniku operatora nr NR dla stalych ARG1 i ARG2. }

Function TKoder.WynikOperatora(nr : Word; arg1, arg2 : Extended) : Extended;
begin
  Blad(bsBrakRealizacjiOper, 0)
end;


{ Dodanie zmiennej do kolekcji zmiennych. Stwierdzenie poprawnosci. }

Function TKoder.DodajZmienna(nazwa : string; id : Word) : Boolean;
var
  P : PZmienna;
begin
  DodajZmienna:=False;
  if not PoprawnoscElementu(nazwa) then Exit;
  P:=New(PZmienna, Init(nazwa, id));
  if P<>nil then
    begin
      DodajZmienna:=True;
      Analizator^.Zmienne^.Insert(P)
    end
end;


{ Dodanie stalej do kolekcji zmiennych. Stwierdzenie poprawnosci. }

Function TKoder.DodajStala(nazwa : string; id : Word; wartosc : Extended) :
  Boolean;
var
  P : PStala;
begin
  DodajStala:=False;
  if not PoprawnoscElementu(nazwa) then Exit;
  P:=New(PStala, Init(nazwa, id, wartosc));
  if P<>nil then
    begin
      DodajStala:=True;
      Analizator^.Stale^.Insert(P)
    end
end;


{ Dodanie funkcji do kolekcji zmiennych. Stwierdzenie poprawnosci. }

Function TKoder.DodajFunkcje(nazwa : string; id : Word) : Boolean;
var
  P : PFunkcja;
begin
  DodajFunkcje:=False;
  if not PoprawnoscElementu(nazwa) then Exit;
  P:=New(PFunkcja, Init(nazwa, id));
  if P<>nil then
    begin
      DodajFunkcje:=True;
      Analizator^.Funkcje^.Insert(P)
    end
end;


{ Dodanie operatora do kolekcji zmiennych. Stwierdzenie poprawnosci. }

Function TKoder.DodajOperator(nazwa : string; id : Word; priorytet : Byte) : Boolean;
var
  P : POperator;
begin
  DodajOperator:=False;
  if not PoprawnyOperator(nazwa) then Exit;
  P:=New(POperator, Init(nazwa, id, priorytet));
  if P<>nil then
    begin
      DodajOperator:=True;
      Analizator^.Operatory^.Insert(P)
    end
end;


{ Funkcja zwracajaca wskaznik na kolekcje odpowiednich elementow }
{ w zaleznosci od parametru typ, ktory moze zawierac wartosci    }
{ takie jak: teZmienna, teStala, teFunkcja, teOperator.          }


Function TKoder.Rodzaj(typ : Byte) : PCollection;
begin
  Rodzaj:=nil;
  if Analizator<>nil then
    with Analizator^ do
      case typ of
        teZmienna  : Rodzaj:=Zmienne;
        teStala    : Rodzaj:=Stale;
        teFunkcja  : Rodzaj:=Funkcje;
        teOperator : Rodzaj:=Operatory;
        else         Rodzaj:=nil
      end
end;


{ Usuniecie danego elementu z kolekcji wg nazwy i zwrocenie wyniku tej }
{ operacji w postaci True - ok, False - brak takiego elementu.         }

Function TKoder.Usun(typ : Byte; nazwa : string) : Boolean;
var
  P  : PCollection;
  El : PElement;
begin
  Usun:=False;
  P:=Rodzaj(typ);
  El:=AdresElementu(P, nazwa, False);
  if El=nil then Exit;
  P^.Free(El);
  Usun:=True
end;


{ Zablokowanie danego elementu poszukowujac go wg nazwy i typu. Zwraca }
{ wynik popranosci wykonania tej operacji.                             }

Function TKoder.Blokuj(typ : Byte; nazwa : string) : Boolean;
var
  El : PElement;
begin
  Blokuj:=False;
  El:=AdresElementu(Rodzaj(typ), nazwa, False);
  if El=nil then Exit;
  El^.stan:=False;
  Blokuj:=True
end;


{ Odblokowanie danego elementu poszukowujac go wg nazwy i typu. Zwraca }
{ wynik popranosci wykonania tej operacji.                             }

Function TKoder.Odblokuj(typ : Byte; nazwa : string) : Boolean;
var
  El : PElement;
begin
  Odblokuj:=False;
  El:=AdresElementu(Rodzaj(typ), nazwa, False);
  if El=nil then Exit;
  El^.stan:=True;
  Odblokuj:=True
end;


{ Stwierdzenie istnienia danego elementu. }

Function TKoder.Istnieje(typ : Byte; nazwa : string) : Boolean;
begin
  Istnieje:=AdresElementu(Rodzaj(typ), UpStr(nazwa), False)<>nil
end;


{ Zmiana nazwy danego elementu. Analiza jej poprawnosci. }

Function TKoder.ZmienNazwe(typ : Byte; NazwaZrodlowa, NazwaDocelowa : string) :
  Boolean;
var
  El : PElement;
begin
  ZmienNazwe:=False;
  if typ=teOperator then
    begin
      if not PoprawnyOperator(NazwaDocelowa) then Exit
    end
   else
    if not PoprawnoscElementu(NazwaDocelowa) then Exit;
  El:=AdresElementu(Rodzaj(typ), NazwaZrodlowa, False);
  if El=nil then Exit;
  if El^.nazwa<>nil then DisposeStr(El^.nazwa);
  El^.nazwa:=NewStr(UpStr(NazwaDocelowa));
  ZmienNazwe:=True
end;


{ Zmiana identyfikatora danego elementu. Zwrocenie wyniku poprawnosci }
{ wykonania operacji.                                                 }

Function TKoder.ZmienId(typ : Byte; nazwa : string; id : Word) : Boolean;
var
  El : PElement;
begin
  ZmienId:=False;
  El:=AdresElementu(Rodzaj(typ), nazwa, False);
  if El=nil then Exit;
  El^.id:=id;
  ZmienId:=True
end;


{ Zmiana priorytetu operatora. Zwrocenie wyniku poprawnosci wykonanej     }
{ operacji. True - znaleziony operator, False - nie ma takiego operatora. }
{ UWAGA! Nie jest mozliwa zmiana priorytetow wszystkich funkcji bowiem    }
{        musza miec one najwyzszy priorytet rowny 255 i zaden operator    }
{        NIE MOZE miec tak wysokiego priorytetu.                          }

Function TKoder.ZmienPriorytet(nazwa : string; priorytet : Byte) : Boolean;
var
  P : POperator;
begin
  ZmienPriorytet:=False;
  if Analizator=nil then Exit;
  P:=POperator(AdresElementu(Analizator^.Operatory, nazwa, False));
  if P=nil then Exit;
  P^.priorytet:=priorytet;
  ZmienPriorytet:=True
end;


{ TWyrazenie ----------------------------------------------------- }


{ Konstruktor wczytujacy dane ze strumienia. }

Constructor TWyrazenie.Load(var S : TStream);
begin
  inherited Load(S);
  S.Read(IloscZmiennych, SizeOf(IloscZmiennych));
  GetMem(TablicaZmiennych, WlkLiczby*IloscZmiennych);
  S.Read(TablicaZmiennych^, IloscZmiennych*WlkLiczby)
end;


{ Zwolnienie pamieci dla pol obiektu TWyrazenie. }

Destructor TWyrazenie.Done;
begin
  if TablicaZmiennych<>nil then FreeMem(TablicaZmiennych, 10*IloscZmiennych);
  inherited Done
end;


{ Procedura zapisujaca obiekt w strumieniu. }

Procedure TWyrazenie.Store(var S : TStream);
begin
  inherited Store(S);
  S.Write(IloscZmiennych, SizeOf(IloscZmiennych));
  S.Write(TablicaZmiennych^, IloscZmiennych*WlkLiczby)
end;


{ Procedura tworzy tablice, w ktorej beda zapamietywane wartosci danych }
{ zmiennych. Inicjuje pola nr_pom kazdej zmiennej oraz modyfikuje kod   }
{ wynikowy tak, aby korzystanie ze zmiennych bylo duzo elastyczniejsze  }
{ i szybsze.                                                            }

Procedure TWyrazenie.UtworzKod;
var
  i : Integer;
begin
  if TablicaZmiennych<>nil then
    FreeMem(TablicaZmiennych, WlkLiczby*IloscZmiennych);
  IloscZmiennych:=0;
  TablicaZmiennych:=nil;
  if Analizator^.Zmienne<>nil then
    with Analizator^.Zmienne^ do
      begin
        IloscZmiennych:=Count;
        for i:=0 to IloscZmiennych-1 do PZmienna(At(i))^.nr_pom:=i+1
      end;
  inherited UtworzKod;
  GetMem(TablicaZmiennych, WlkLiczby*IloscZmiennych);
  if (IloscZmiennych>0) and (TablicaZmiennych=nil) then Blad(bsBrakPamieci, 1);
  if KodBledu<>bsOk then Exit;
  for i:=1 to WskKodu-1 do
    with Kod^[i] do
      if typ=reZmienna then nr:=PZmienna(AdrElWgId(Analizator^.Zmienne,
        nr, True))^.nr_pom
end;


{ Funkcja realizuje obliczenie danego, juz przeanalizowanego wyrazenia. }
{ Uwaga. Tuz po inicjacji obiekty nie jest mozliwe od razu obliczenie   }
{ jakiegokolwiek wyrazenia. Przed ta operacja uzytkownik MUSI wywolac   }
{ procedure analizujaca tekst zawarty w lancuchu znakowym. Takiej       }
{ koniecznosci nie ma, jezeli chcemy obliczyc dane wyrazenie po wywo-   }
{ laniu metody NoweWyrazenie, gdyz ona sama dokonuje automatycznie      }
{ analizy danego wyrazenia.                                             }

Function TWyrazenie.Oblicz : Extended;
var
  StosWartosci    : PTablicaLiczb;
  WskStosWartosci : Word;


  { Glowna petla. }

  Function Wykonaj : Extended;
  var
    liczba : Extended;
    i : Word;
  begin
    WskStosWartosci:=1; i:=1; liczba:=0;
    while Kod^[i].typ<>reKoniec do
      begin
        with Kod^[i] do
          case typ of
            reZmienna  : liczba:=TablicaZmiennych^[nr];
            reStala    : liczba:=Analizator^.TabStalych^[nr];
            reOperator :
              begin
                liczba:=WynikOperatora(nr, StosWartosci^[WskStosWartosci-2],
                  StosWartosci^[WskStosWartosci-1]);
                StosWartosci:=Decr(StosWartosci, WlkLiczby, WskStosWartosci,
                  RMEM);
                StosWartosci:=Decr(StosWartosci, WlkLiczby, WskStosWartosci,
                  RMEM)
              end;
            reFunkcja  :
              begin
                liczba:=WynikFunkcji(nr, StosWartosci^[WskStosWartosci-1]);
                StosWartosci:=Decr(StosWartosci, WlkLiczby, WskStosWartosci,
                  RMEM)
              end
          end;
        Inc(i);
        StosWartosci^[WskStosWartosci]:=liczba;
        StosWartosci:=Incr(StosWartosci, WlkLiczby, WskStosWartosci, RMEM)
      end;
    Wykonaj:=liczba
  end;

begin
  Oblicz:=0;
  WskStosWartosci:=1;
  StosWartosci:=Przydziel(StosWartosci, WlkLiczby, WskStosWartosci, RMEM);
  if KodBledu<>bsOk then Exit;
  if (StosWartosci<>nil) and (Kod<>nil) then
    begin
      Oblicz:=Wykonaj;
      Zwolnij(StosWartosci, WlkLiczby, WskStosWartosci, RMEM)
    end
   else
    Blad(bsBrakPamieci, 1)
end;


{ Funkcja pozwala na przypisanie danej zmiennej nowej wartosci. Jest to }
{ rownoznaczne z instrukcja: nazwa_zmiennej:=wartosc. Funkcja zwraca    }
{ True jezeli operacja sie powiodla.                                    }

Function TWyrazenie.NowaWartosc(id : Word; wartosc : Extended) : Boolean;
var
  zmienna : PZmienna;
begin
  if Analizator<>nil then
    begin
      zmienna:=PZmienna(AdrElWgId(Analizator^.Zmienne, id, False));
      if (zmienna<>nil) and (zmienna^.nr_pom>0) then
        begin
          TablicaZmiennych^[zmienna^.nr_pom]:=wartosc;
          NowaWartosc:=True
        end
       else
        NowaWartosc:=False
    end
   else
    NowaWartosc:=False
end;


{ TWyrPodst ------------------------------------------------------- }


{ Procedura powoduje dodanie elementow wyrazenia podstawowego. }

Procedure TWyrPodst.DodajElementy;
begin
  inherited DodajElementy;
  DodajFunkcje('Abs', fnAbs);
  DodajFunkcje('Int', fnInt);
  DodajFunkcje('Frac', fnFrac);
  DodajFunkcje('Trunc', fnTrunc);
  DodajFunkcje('Odd', fnOdd);
  DodajFunkcje('Sgn', fnSgn);
  DodajFunkcje('Round', fnRound);
  DodajOperator('-', opMinus, 20);
  DodajOperator('+', opPlus, 20);
  DodajOperator('Mod', opMod, 30);
  DodajOperator('Div', opDiv, 30);
  DodajOperator('/', opDzielenie, 40);
  DodajOperator('*', opMnozenie, 40);
  DodajOperator('^', opPotegowanie, 50)
end;


{ Wynikiem funkcji jest obliczenie funkcji nr NR z argumentem ARG. }

Function TWyrPodst.WynikFunkcji(nr : Word; arg : Extended) : Extended;
var
  liczba : Extended;
begin
  case nr of
    fnAbs   : liczba:=Abs(arg);
    fnInt   : liczba:=Int(arg);
    fnFrac  : liczba:=Frac(arg);
    fnTrunc :
      if IsLongInt(arg) then liczba:=Trunc(arg);
    fnOdd   :
      if IsLongInt(arg) then liczba:=Ord(Odd(Round(arg)));
    fnSgn   : liczba:=Sgn(arg);
    fnRound :
      if IsLongInt(arg) then liczba:=Round(arg);
    else
      liczba:=inherited WynikFunkcji(nr, arg)
  end;
  WynikFunkcji:=liczba
end;


{ Rezultaten funkcji jest wynik operatora dla argumentow ARG1 i ARG2. }

Function TWyrPodst.WynikOperatora(nr : Word; arg1, arg2 : Extended) : Extended;
var
  liczba, w : Extended;
begin
  case nr of
    opPlus        : liczba:=arg1+arg2;
    opMinus       : liczba:=arg1-arg2;
    opMod         :
      if IsLongInt(arg1) and IsLongInt(arg2) then
        liczba:=Round(arg1) mod Round(arg2);
    opDiv         :
      if IsLongInt(arg1) and IsLongInt(arg2) then
        liczba:=Round(arg1) div Round(arg2);
    opDzielenie   :
      begin
        if arg2=0 then
          Blad(bsDzieleniePrzezZero, 0)
         else
          liczba:=arg1/arg2
      end;
    opMnozenie    : liczba:=arg1*arg2;
    opPotegowanie :
      begin
        w:=arg2*Ln(Abs(arg1));
        if Abs(w)>1.1356e+4 then
          Blad(bsBlednaLiczba, 0)
         else
          begin
            if arg1=0 then
              liczba:=0
             else
              if arg2=0 then
                liczba:=1
               else
                liczba:=Sgn(arg1)*Exp(w)
          end
      end;
    else liczba:=inherited WynikOperatora(nr, arg1, arg2)
  end;
  WynikOperatora:=liczba
end;


{ TWyrLog --------------------------------------------------------- }


{ Procedura powoduje dodanie elementow wyrazenia logicznego. }

Procedure TWyrLog.DodajElementy;
begin
  inherited DodajElementy;
  DodajOperator('<=', opMniejszelubRowne, 15);
  DodajOperator('>=', opWiekszeLubRowne, 15);
  DodajOperator('<', opMniejsze, 15);
  DodajOperator('>', opWieksze, 15);
  DodajOperator('=', opRowne, 14);
  DodajOperator('Or', opOr, 5);
  DodajOperator('And', opAnd, 10);
  DodajOperator('<>', opRozne, 14);
  DodajOperator('#', opRozne, 14);
  DodajOperator('!=', opRozne, 14)
end;


{ Rezultaten funkcji jest wynik operatora dla argumentow ARG1 i ARG2. }

Function TWyrLog.WynikOperatora(nr : Word; arg1, arg2 : Extended) : Extended;
var
  liczba : Extended;
begin
  case nr of
    opMniejszeLubRowne : liczba:=Ord(arg1<=arg2);
    opWiekszeLubRowne  : liczba:=Ord(arg1>=arg2);
    opMniejsze         : liczba:=Ord(arg1<arg2);
    opWieksze          : liczba:=Ord(arg1>arg2);
    opRowne            : liczba:=Ord(arg1=arg2);
    opOr               : liczba:=Ord((arg1<>0) or (arg2<>0));
    opAnd              : liczba:=Ord((arg1<>0) and (arg2<>0));
    else                 liczba:=inherited WynikOperatora(nr, arg1, arg2)
  end;
  WynikOperatora:=liczba
end;


{ TFunkcjaTryg ------------------------------------------------------ }


{ Procedura powoduje dodanie elementow funkcji trygonometrycznych. }

Procedure TFunkcjeTryg.DodajElementy;
begin
  inherited DodajElementy;
  DodajFunkcje('Sec', fnSec);
  DodajFunkcje('CSC', fnCSC);
  DodajFunkcje('Sqrt', fnSqrt);
  DodajFunkcje('Tan', fnTan);
  DodajFunkcje('CTg', fnCTg);
  DodajFunkcje('ArcCos', fnArcCos);
  DodajFunkcje('ArcTan', fnArcTan);
  DodajFunkcje('Sin', fnSin);
  DodajFunkcje('Cos', fnCos);
  DodajFunkcje('ArcSin', fnArcCos);
  DodajFunkcje('ArcCTg', fnArcTan);
  DodajFunkcje('SinH', fnSinH);
  DodajFunkcje('CosH', fnCosH);
  DodajFunkcje('TanH', fnTanH);
  DodajFunkcje('Exp', fnExp);
  DodajFunkcje('LN', fnLN);
  DodajStala('Pi', stPi, Pi);
  DodajStala('e', stE, Exp(1))
end;


{ Wynikiem funkcji jest obliczenie funkcji nr NR z argumentem ARG. }

Function TFunkcjeTryg.WynikFunkcji(nr : Word; arg : Extended) : Extended;
var
  liczba, w : Extended;
begin
  case nr of
    fnSec    :
      if Abs(arg)>1.8446e+19 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        begin
          w:=Cos(arg);
          if w=0 then
            Blad(bsBlednyArgFunkcji, 0)
           else
            liczba:=1/w
        end;
    fnCSC    :
      if Abs(arg)>1.8446e+19 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        begin
          w:=Sin(arg);
          if w=0 then
            Blad(bsBlednyArgFunkcji, 0)
           else
            liczba:=1/w
        end;
    fnSqrt   :
      if arg<0 then
        Blad(bsLiczbaUjemna, 0)
       else
        liczba:=Sqrt(arg);
    fnTan    :
      if Abs(arg)>1.8446e+19 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        begin
          w:=Cos(arg);
          if w=0 then
            Blad(bsBlednyArgFunkcji, 0)
           else
            liczba:=Sin(arg)/w
        end;
    fnCTg    :
      if Abs(arg)>1.8446e+19 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        begin
          w:=Sin(arg);
          if w=0 then
            Blad(bsBlednyArgFunkcji, 0)
           else
            liczba:=Cos(arg)/w
        end;
    fnArcSin :
      if (arg<=-1) or (arg>=1) or (arg=0) then
        Blad(bsBlednyArgFunkcji, 0)
       else
        begin
          w:=1-Sqr(arg);
          if w<=0 then
            Blad(bsBlednyArgFunkcji, 0)
           else
            liczba:=ArcTan (arg / Sqrt (w))
        end;
    fnArcCos :
      if (arg<=-1) or (arg>=1) or (arg=0) then
        Blad(bsBlednyArgFunkcji, 0)
       else
         liczba:=ArcTan (sqrt ( 1-Sqr(arg) )) /arg; {-ArcTan(arg/Sqrt( 1- sqr (arg) ))+Pi/2}
    fnArcTan : liczba:=ArcTan(arg);
    fnSin    :
      if Abs(arg)>1.8446e+19 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        liczba:=Sin(arg);
    fnCos    :
      if Abs(arg)>1.8446e+19 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        liczba:=Cos(arg);
    fnArcCTg : liczba:=ArcTan(arg)+Pi/2;
    fnSinH   :
      if Abs(arg)>1.1356e+4 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        liczba:=(Exp(arg)-Exp(-arg))/2;
    fnCosH   :
      if Abs(arg)>1.1356e+4 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        liczba:=(Exp(arg)+Exp(-arg))/2;
    fnTanH   :
      if Abs(arg)>1.1356e+4 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        begin
          w:=Exp(arg)+Exp(-arg);
          if w=0 then
            Blad(bsBlednyArgFunkcji, 0)
           else
            liczba:=(Exp(arg)-Exp(-arg))/w
        end;
    fnExp    :
      if Abs(arg)>1.1356e+4 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        liczba:=Exp(arg);
    fnLN     :
      if arg<=0 then
        Blad(bsBlednyArgFunkcji, 0)
       else
        liczba:=Ln(arg);
    else
     liczba:=inherited WynikFunkcji(nr, arg)
  end;
  WynikFunkcji:=liczba
end;

end.