{******************************************************}
{*                                                    *}
{*  Nazwa modulu  : Image256                          *}
{*  Wersja        : 1.4                               *}
{*  Kompilowane w : Turbo Pascal v. 7.0               *}
{*  Procesor      : 80x86                             *}
{*  Co-procesor   : niepotrzebny                      *}
{*  Karta graf.   : VGA, MCGA lub lepsza w 13h trybie *}
{*  Autor         : Buk Mariusz                       *}
{*  Data          : 6 XI 1994                         *}
{*  Miejsce       : Polska                            *}
{*                                                    *}
{*  Opis:                                             *}
{*                                                    *}
{*  Modul umozliwia szybsza od standardowej ob-       *}
{*  sluge grafiki w rozdzielczosci 320x200 w 256      *}
{*  kolorach. Mozliwe jest rowniez zapisywanie        *}
{*  rysunkow i ich identyfikowanie w pliku.           *}
{*  Dzieki specjalnym funkcjom modul ma rowniez       *}
{*  mozliwosc zapisywania i odczytywania palety       *}
{*  z pliku wraz z przypisaniem jej danego id.        *}
{*  Identyfikator danego obiektu ma postac ciagu      *}
{*  znakow (string).                                  *}
{*  Modul moze dzialac rownolegle z modulem GRAPH.    *}
{*  W celu przyspieszenia dzialania nie jest          *}
{*  sprawdzana poprawnosc zadnych danych.             *}
{*  Modul zawiera takze procedury umozliwiajace       *}
{*  tworzenie efektu scrollingu w postaci przesuwu    *}
{*  okienka graficznego. Wszystkie procedury          *}
{*  przeznaczone sa do dzialania w trybie VGA         *}
{*  320 x 200 w 256 kolorach i nie beda dzialaly      *}
{*  w wyzszych rozdzielczosciach o tej samej          *}
{*  liczbie kolorow.                                  *}
{*                                                    *}
{******************************************************}

{$F+,X+,D-,O-,I-,S+}

Unit Image256;

interface

type
  Scr_RGB = record
    Red, Green, Blue : Byte;
  end;

  PScr_RGB = ^TScr_RGB;
  TScr_RGB = array [0..255] of Scr_RGB;


{ proste operacje na ekranie }

procedure _PutPixel(x, y : Word; c : Byte);
function _GetPixel(x, y : Word) : Byte;

{ inne operacje graficzne }

function _ImageSize(x1, y1, x2, y2 : Word) : Word;
procedure _GetImage(x1, y1, x2, y2 : Word; var Buf);
procedure _PutImage(x, y : Word; var buf; oper : Byte);
procedure SpecialPut(x, y, dx, dy : Word; width, height : Integer;
  var buf; oper : Byte);

{ operacje dyskowe na rysunku zwiazane z buforem }

function AddImage(var handle : file; id : string; var buf) : Boolean;
function LoadImage(var handle : file; var buf) : Boolean;

function DeleteImage(var handle : file; const name : string) : Boolean;
function FindNextPic(var handle : file) : Boolean;
function SearchPicById(var handle : file; id : string) : Boolean;
function FileImageSize(var handle : file; var Size : Word) : Boolean;
function ExistPicId(var handle : file; id : string) : Boolean;
function ImageInfo(var handle : file; var id : string;
  var szer, wys : Word) : Boolean;

{ operacje dyskowe na rysunku zwiazane bezposrednio z ekranem }

function AddScrImage(var handle : file; id : string; x1, y1, x2, y2 : Word) : Boolean;
function LoadScrImage(var handle : file; x, y : Word) : Boolean;

{ operacje dyskowe na palecie zwiazane z buforem }

function AddPalette(var handle : file; id : string; var pal : TScr_RGB;
  first, cnt : Word) : Boolean;
function LoadPalette(var handle : file; var pal : TScr_RGB) : Boolean;

function DeletePalette(var handle : file; const name : string) : Boolean;
function FindNextPal(var handle : file) : Boolean;
function SearchPalById(var handle : file; id : string) : Boolean;
function ExistPalId(var handle : file; id : string) : Boolean;
function PaletteInfo(var handle : file; var id : string;
  var first, cnt : Word) : Boolean;

procedure UsePalette(const pal : TScr_RGB);
procedure ReadPalette(var pal : TScr_RGB);

{ operacje dyskowe na palecie zwiazane bezposrednio z ekranem }

function AddScrPalette(var handle : file; id : string; first, cnt : Word) : Boolean;
function LoadScrPalette(var handle : file) : Boolean;

{ przesuwanie okna graficznego }

procedure MoveWindow(x1, y1, x2, y2 : Word; sx, sy : Integer);

{ obsluga bledow }

var
  IOStatus : Integer;


{ Stala LayOnPut okresla sposob polozenia zawartosci bufora na ekran. }
{ Wynikiem operacji jest polozenie wszystkich punktow, oprocz punktow }
{ o kolorze tla (=0).                                                 }

const
  NormalPut = 0;  { MOV }
  CopyPut   = 0;  { MOV }
  XORPut    = 1;  { XOR }
  OrPut     = 2;  { OR  }
  AndPut    = 3;  { AND }
  NotPut    = 4;  { NOT }
  LayOnPut  = 5;  { MOV if not 0 }


{ Elementy nie objasnione w ksiazce. }

procedure Proc_Del_Image(position : LongInt; const Id : string;
  arg1, arg2 : Word);
procedure Proc_Del_Palette(position : LongInt; const Id : string;
  arg1, arg2 : Word);

type
  TDel = procedure (position : LongInt; const Id : string; arg1, arg2 : Word);

const
  Del_Image   : TDel = Proc_Del_Image;
  Del_Palette : TDel = Proc_Del_Palette;


implementation


uses DOS;

type
  THeader = array [1..300] of Byte;

const
  MaxSpace = 65520;

{ Konwersja liczby calkowitej do tekstu. }

Function IntStr(n : LongInt) : string;
var
  S : string;
begin
  Str(n, S);
  IntStr:=S
end;


{ Narysowanie punktu w danym kolorze o danych wspolrzednych. }

Procedure _PutPixel(x, y : Word; c : Byte); assembler;
asm
  MOV   AX, SegA000
  MOV   ES, AX
  MOV   AX, [y]
  MOV   BX, [x]
  XCHG  AH, AL          { AX <- y * 256           }
  ADD   BX, AX          { BX <- y *256 + x        }
  SHR   AX, 1           { AX <- AX div 4 = 64 * y }
  SHR   AX, 1
  ADD   BX, AX          { BX <- 320 * y + x       }
  MOV   DI, BX
  MOV   AL, [c]
  STOSB
end;


{ Pobranie koloru punktu o danych wspolrzednych. }

Function _GetPixel(x, y : Word) : Byte; assembler;
asm
  MOV   AX, SegA000
  MOV   ES, AX
  MOV   AX, [y]
  MOV   BX, [x]
  XCHG  AH, AL          { AX <- y * 256           }
  ADD   BX, AX          { BX <- y *256 + x        }
  SHR   AX, 1           { AX <- AX div 4 = 64 * y }
  SHR   AX, 1
  ADD   BX, AX          { BX <- 320 * y + x       }
  MOV   DI, BX
  MOV   AL, ES:[DI]
end;


{ Obliczenie wielkosci bufora dla zapamietywanego obszaru. }

Function _ImageSize(x1, y1, x2, y2 : Word) : Word; assembler;
asm
  MOV   AX, [y2]
  SUB   AX, [y1]
  INC   AX
  MOV   BX, [x2]
  SUB   BX, [x1]
  INC   BX
  MUL   BX
  ADD   AX, 04h
  JNC   @d1
  INC   DX
@d1:
end;


{ Pobranie rysunku i zapamietanie go w buforze. }

Procedure _GetImage(x1, y1, x2, y2 : Word; var Buf); assembler;
asm
  PUSH  DS

  MOV   BX, [x2]
  SUB   BX, [x1]
  INC   BX                              { obliczenie szerokosci }
  MOV   DX, [Y2]
  SUB   DX, [Y1]
  INC   DX                              { obliczenie wysokosci }

  MOV   ES, word ptr [buf+2]
  MOV   DI, word ptr [buf]              { ustalenie adresu bufora }
  MOV   AX, BX
  STOSW                                 { zapisanie szerokosci }
  MOV   AX, DX
  STOSW                                 { zapisanie wysokosci }

  MOV   AX, SegA000
  MOV   DS, AX
  MOV   AX, [y1]
  MOV   CX, [x1]
  XCHG  AH, AL          { AX <- y * 256           }
  ADD   CX, AX          { BX <- y *256 + x        }
  SHR   AX, 1           { AX <- AX div 4 = 64 * y }
  SHR   AX, 1
  ADD   CX, AX          { BX <- 320 * y + x       }
  MOV   SI, CX          { ustalenie poczatkowego adresu punktu }

@for:
  MOV   CX, BX                          { skopiowanie szerokosci }
  SHR   CX, 01h
  REP   MOVSW                           { jedna linia do bufora }
  TEST  BL, 01h
  JP    @p1
  MOVSB
@p1:
  ADD   SI, 320d
  SUB   SI, BX                          { nastepna linia }
  DEC   DX
  JNZ   @for

  POP   DS
end;


{ Odtworzenie rysunku z bufora i narysowanie go na ekranie.    }
{ Obsluguje wszystkie dotychczasowe operacji bitowe z dodaniem }
{ operacji LayOnPut (znaczenie wyzej).                         }

Procedure _PutImage(x, y : Word; var buf; oper : Byte); assembler;
asm
  PUSH  DS

  MOV   AX, SegA000
  MOV   ES, AX
  MOV   AX, [y]
  MOV   CX, [x]
  XCHG  AH, AL          { AX <- y * 256           }
  ADD   CX, AX          { BX <- y *256 + x        }
  SHR   AX, 1           { AX <- AX div 4 = 64 * y }
  SHR   AX, 1
  ADD   CX, AX          { BX <- 320 * y + x       }
  MOV   DI, CX          { obliczenie adresu poczatkowego punktu }

  MOV   DS, word ptr [buf+2]
  MOV   SI, word ptr [buf]
  LODSW
  MOV   BX, AX                          { pobranie szerokosci }
  LODSW
  MOV   DX, AX                          { pobranie wysokosci }

  MOV   AL, [oper]
  CMP   AL, LayOnPut
  JE    @LayOnPut
  CMP   AL, NotPut
  JE    @NotPut
  CMP   AL, AndPut
  MOV   AH, 22h                         { code AND byte ptr }
  JE    @XXXPut
  CMP   AL, OrPut
  MOV   AH, 0Ah                         { code OR byte ptr }
  JE    @XXXPut
  CMP   AL, XOrPut
  MOV   AH, 32h                         { code XOR byte ptr }
  JE    @XXXPut
  CMP   AL, CopyPut
  JNE   @koniec

@CopyPut:
  MOV   CX, BX
  SHR   CX, 01h
  REP   MOVSW                           { jedna linia }
  TEST  BL, 01h
  JP    @p1
  MOVSB
@p1:
  ADD   DI, 320d
  SUB   DI, BX
  DEC   DX                              { nastepna linia }
  JNZ   @CopyPut
  JMP   @koniec

@XXXPut:
  MOV   byte ptr [@ins+1], AH
@for1:
  MOV   CX, BX
@for2:
  LODSB
@ins:
  XOR   AL, ES:[DI]
  STOSB
  LOOP  @for2                           { skopiowanie jednej linii }
  ADD   DI, 320d
  SUB   DI, BX
  DEC   DX                              { nastepna linia }
  JNZ   @for1
  JMP   @koniec

@LayOnPut:
  MOV   CX, BX
  SHR   CX, 01h
@for3:
  LODSB
  AND   AL, AL
  JZ    @hop
  MOV   byte ptr ES:[DI], AL
@hop:
  INC   DI
  LOOP  @for3                           { skopiowanie jednej linii }
  ADD   DI, 320d
  SUB   DI, BX
  DEC   DX                              { nastepna linia }
  JNZ   @LayOnPut
  JMP   @koniec

@NotPut:
  MOV   CX, BX
  SHR   CX, 01h
@for4:
  LODSW
  NOT   AX
  STOSW
  LOOP  @for2                           { skopiowanie jednej linii }
  TEST  BL, 01h
  JP    @p2
  LODSB
  NOT   AL
  STOSB
@p2:
  ADD   DI, 320d
  SUB   DI, BX
  DEC   DX                              { nastepna linia }
  JNZ   @NotPut

@koniec:
  POP DS
end;


{ Specjalne odtworzenie bufora BUF na ekran.                        }
{ x, y   - wspolrzedne punktu, od ktorego bedzie odtwarzany rysunek }
{ dx, dy - dodatnie przesuniecie w kazdej z osi w buforze           }
{ width  - szerokosc wyswietlanego rysunku                          }
{ height - wysokosc wyswietlanego rysunku                           }

Procedure SpecialPut(x, y, dx, dy : Word; width, height : Integer;
  var buf; oper : Byte); assembler;
var
  szer : Word;
asm
  PUSH  DS

  MOV   AX, SegA000
  MOV   ES, AX
  MOV   AX, [y]
  MOV   CX, [x]
  XCHG  AH, AL          { AX <- y * 256           }
  ADD   CX, AX          { BX <- y * 256 + x       }
  SHR   AX, 01h         { AX <- AX div 4 = 64 * y }
  SHR   AX, 01h
  ADD   CX, AX          { BX <- 320 * y + x       }
  MOV   DI, CX          { obliczenie adresu poczatkowego punktu }

  MOV   DS, word ptr [buf+2]
  MOV   SI, word ptr [buf]

  LODSW
  MOV   BX, AX                          { pobranie szerokosci }
  CMP   AX, [width]
  JNC   @dn_1
  MOV   [width], AX
@dn_1:
  CMP   AX, [&dx]
  JC    @koniec

  LODSW
  MOV   CX, [height]
  CMP   AX, CX
  JC    @dn_2
  MOV   AX, CX
@dn_2:
  SUB   AX, [dy]
  JC    @koniec
  JE    @koniec
  PUSH  AX                              { pobranie wysokosci }

  MOV   AX, [dy]
  MUL   BX
  ADD   AX, [&dx]
  ADD   SI, AX          { przesuniecie w buforze o wektor <dx, dy> }
  POP   DX

  MOV   AX, [width]
  SUB   AX, [&dx]
  MOV   [szer], AX

  MOV   AL, [oper]
  CMP   AL, LayOnPut
  JE    @LayOnPut
  CMP   AL, NotPut
  JE    @NotPut
  CMP   AL, AndPut
  MOV   AH, 22h                         { code AND byte ptr }
  JE    @XXXPut
  CMP   AL, OrPut
  MOV   AH, 0Ah                         { code OR byte ptr }
  JE    @XXXPut
  CMP   AL, XOrPut
  MOV   AH, 32h                         { code XOR byte ptr }
  JE    @XXXPut
  CMP   AL, CopyPut
  JNE   @koniec

@CopyPut:
  MOV   AX, [szer]
@loop_1:
  MOV   CX, AX
  SHR   CX, 01h
  REPNE MOVSW                           { jedna linia }
  TEST  AL, 01h
  JP    @p1
  MOVSB
@p1:
  SUB   SI, AX
  ADD   SI, BX
  SUB   DI, AX
  ADD   DI, 320d
  DEC   DX                              { nastepna linia }
  JNZ   @loop_1
  JMP   @koniec

@XXXPut:
  MOV   byte ptr [@ins+1], AH
@for1:
  MOV   CX, [szer]
@for2:
  LODSB
@ins:
  XOR   AL, ES:[DI]
  STOSB
  LOOP  @for2                           { skopiowanie jednej linii }
  MOV   AX, [szer]
  SUB   SI, AX
  ADD   SI, BX
  SUB   DI, AX
  ADD   DI, 320d
  DEC   DX                              { nastepna linia }
  JNZ   @for1
  JMP   @koniec

@LayOnPut:
  MOV   CX, [szer]
@for3:
  LODSB
  AND   AL, AL
  JZ    @hop
  MOV   byte ptr ES:[DI], AL
@hop:
  INC   DI
  LOOP  @for3                           { skopiowanie jednej linii }
  MOV   AX, [szer]
  SUB   SI, AX
  ADD   SI, BX
  SUB   DI, AX
  ADD   DI, 320d
  DEC   DX                              { nastepna linia }
  JNZ   @LayOnPut
  JMP   @koniec

@NotPut:
  MOV   CX, [szer]
@for4:
  LODSB
  NOT   AL
  STOSB
  LOOP  @for2                           { skopiowanie jednej linii }
  MOV   AX, [szer]
  SUB   SI, AX
  ADD   SI, BX
  SUB   DI, AX
  ADD   DI, 320d
  DEC   DX                              { nastepna linia }
  JNZ   @NotPut

@koniec:
  POP DS
end;


{ Procedura MoveWindow sluzy do przesuwania okienka graficznego w pionie }
{ jak i poziomie.                                                        }
{  sx - ilosc piksli (punktow) o jaka okienko jest przesuwane w poziomie }
{  sy - ilosc piksli (punktow) o jaka okienko jest przesuwane w pionie   }

Procedure MoveWindow(x1, y1, x2, y2 : Word; sx, sy : Integer); assembler;
asm
        { INICJACJA REJESTOW SEGMENTOWYCH                                }

        PUSH    DS
        MOV     AX, SegA000
        MOV     ES, AX
        MOV     DS, AX          { wszystkie operacje na pamieci obrazu   }

        { INICJACJA DANYCH                                               }

        MOV     CX, [x2]        { ilosc bajtow w poziomie => CX          }
        SUB     CX, [x1]        { CX = x2 - x1 + 1                       }
        INC     CX

        { PROCEDURY PRZESUWU ------------------------------------------- }

        MOV     AX, [sy]
        TEST    AX, AX          { jezeli sx lub sy posiada ujemny znak   }
        JS      @do_gory        { wowczas przesuwanie w dol pamieci,     }
        JNZ     @w_dol
        MOV     AX, [sx]        { czyli w gore ekranu                    }
        TEST    AX, AX
        JS      @do_gory


        { SCROLLING W DOL                                                }

@w_dol:
        STD                     { ustalenie kierunku przesylania danych  }

                                { obliczamy adres miejsca, z ktrego     }
        MOV     AX, [y2]        { zacznie sie przesuwanie                }
        MOV     SI, [x2]        { (x2, y2)  [DS:SI]                      }
        XCHG    AH, AL          { AX <- y * 256                          }
        ADD     SI, AX          { SI <- y * 256 + x                      }
        SHR     AX, 1           { AX <- AX div 4 = 64 * y                }
        SHR     AX, 1
        ADD     SI, AX          { SI <- 320 * y + x                      }

        MOV     AX, [sy]        { oraz adres miejsca, dokad przesuwamy   }
        MOV     DI, [sx]        { (x2 + sx, y2 + sy)   [ES:DI]           }
        XCHG    AH, AL
        ADD     DI, AX
        SHR     AX, 1
        SHR     AX, 1
        ADD     DI, AX
        ADD     DI, SI

        MOV     DX, [y2]        { liczba linii do przesuniecia => DX     }
        SUB     DX, [y1]        { DX = y2 - y1 + 1                       }
        INC     DX

        MOV     BX, CX          { do BX - szerokosc                      }
@petla1:
        MOV     CX, BX
        SHR     CX, 01h
        DEC     SI
        DEC     DI
        REP     MOVSW           { przesuwamy jedna linie                 }
        INC     SI
        INC     DI
        TEST    BL, 01h
        JP      @p1
        MOVSB
@p1:
        SUB     DI, 320d        { obliczamy adresy nastepnych linii      }
        ADD     DI, BX
        SUB     SI, 320d        { (zrodlowej i docelowej)                }
        ADD     SI, BX
        DEC     DX              { zmniejszamy licznik linii              }
        JNZ     @petla1         { jesli nie koniec to nastepna linia     }
        JMP     @end            { koniec                                 }

        { SCROLLING DO GORY                                              }

@do_gory:

        CLD                     { ustalenie kierunku przesylania danych  }

                                { obliczamy adres miejsca, z ktrego     }
        MOV     AX, [y1]        { zacznie sie przesuwanie                }
        MOV     SI, [x1]        { (x1, y1)  [DS:SI]                      }
        XCHG    AH, AL          { AX <- y * 256                          }
        ADD     SI, AX          { SI <- y * 256 + x                      }
        SHR     AX, 1           { AX <- AX div 4 = 64 * y                }
        SHR     AX, 1
        ADD     SI, AX          { SI <- 320 * y + x                      }

        NEG     [sy]
        NEG     [sx]

        MOV     AX, [sy]        { oraz adres miejsca, dokad przesuwamy   }
        MOV     DI, [sx]        { (x1 + sx, y1 + sy)   [ES:DI]           }
        XCHG    AH, AL          { AX <- y * 256                          }
        ADD     DI, AX          { AX <- y * 256 + x                      }
        SHR     AX, 1           { AX <- AX div 4 = 64 * y                }
        SHR     AX, 1
        ADD     DI, AX          { DI <- 320 * y + x                      }
        MOV     BX, SI
        SUB     SI, DI
        MOV     DI, SI
        MOV     SI, BX          { DI <- SI - DI                          }

        MOV     DX, [y2]
        SUB     DX, [y1]        { liczba linii do przesuniecia => DX     }
        INC     DX              { DX = y2 - y1 + 1                       }

        MOV     BX, CX          { do BX - szerokosc                      }
@petla2:
        MOV     CX, BX
        SHR     CX, 01h
        REP     MOVSW           { przesuwamy jedna linie                 }
        TEST    BL, 01h
        JP      @p2
        MOVSB
@p2:
        ADD     DI, 320d        { obliczamy adresy nastepnych linii      }
        SUB     DI, BX
        ADD     SI, 320d        { (zrodlowej i docelowej)                }
        SUB     SI, BX
        DEC     DX              { zmniejszamy licznik linii              }
        JNZ     @petla2         { jesli nie koniec to nastepna linia     }

@end:   { ODTWORZENIE REJESTRU SEGMENTOWEGO DS                           }

        POP     DS
end;


{ Funkcja spowoduje zapisanie naglowka do pliku. }

Function WriteHeader(var handle : file; var header : THeader) : Boolean;
begin
  BlockWrite(handle, header, header[1]+6-1);
  IOStatus:=IOResult;
  WriteHeader:=IOStatus=0
end;


{ Funkcja spowoduje odczytanie naglowka z pliku. }
{ shift = True  - bez ruszania wskaznika         }
{ shift = False - przesuniecie na kod danych     }

Function ReadHeader(var handle : file; var header : THeader;
  var arg1, arg2 : Word; shift : Boolean) : Boolean;
var
  OldPos : LongInt;
  test   : Word;
begin
  OldPos:=FilePos(handle);
  BlockRead(handle, header, SizeOf(header), test);
  if shift then
    Seek(handle, OldPos)
   else
    Seek(handle, OldPos+header[1]+6-1);
  Move(header[header[1]+2], arg1, SizeOf(arg1));
  Move(header[header[1]+4], arg2, SizeOf(arg2));
  IOStatus:=IOResult;
  if IOStatus=0 then if header[header[1]+1]<>not header[1] then IOStatus:=13;
  ReadHeader:=IOStatus=0
end;


{ Dodanie rysunku do pliku. W razie niepowodzenia zwrocenie wawrtosci }
{ FALSE i umieszczenie w zmiennej IOStatus kodu bledu.                  }
{ Aktualna pozycja wskaznika pliku nie zmienia sie.                   }

Function AddImage(var handle : file; id : string; var buf) : Boolean;
var
  rozmiar    : Word;
  oldpos, dl : LongInt;
begin
  IOStatus:=0;
  AddImage:=False;
  OldPos:=FilePos(handle);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  dl:=FileSize(handle);
  Seek(handle, dl);
  asm
    MOV     ES, word ptr [buf+2]
    MOV     DI, word ptr [buf]
    MOV     AX, ES:[DI]
    MOV     BX, ES:[DI+2]
    MUL     BX
    ADD     AX, 04h
    MOV     [rozmiar], AX
  end;
  if Length(id)>254 then Dec(id[0]);
  id:=id+Chr(not (Ord(id[0])+1));
  BlockWrite(handle, id, Length(id)+1);
  BlockWrite(handle, buf, rozmiar);
  IOStatus:=IOResult;
  if IOStatus<>0 then
    begin
      Seek(handle, dl);
      Truncate(handle)
    end
   else
     begin
      AddImage:=True;
      Seek(handle, OldPos);
      IOStatus:=IOResult
    end
end;


{ Dodanie rysunku do pliku wraz z identyfikatorem. Zwraca FALSE, }
{ jezeli operacja sie nie powiodla. Funkcja ta zapisuje rysunek  }
{ bezposrednio z ekranu (bez uywania bufora).                   }

Function AddScrImage(var handle : file; id : string; x1, y1, x2, y2 : Word) : Boolean;
var
  i, tmp     : Word;
  buf        : ^Byte;
  oldpos, dl : LongInt;
  header     : THeader;
  id1        : string absolute header;
begin
  IOStatus:=0;
  AddScrImage:=False;
  OldPos:=FilePos(handle);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  dl:=FileSize(handle);
  Seek(handle, dl);

  if Length(id)>254 then Dec(id[0]);
  id1:=id+Chr(not (Ord(id[0])+1));
  header[header[1]+2]:=Lo(x2-x1+1);
  header[header[1]+3]:=Hi(x2-x1+1);
  header[header[1]+4]:=Lo(y2-y1+1);
  header[header[1]+5]:=Hi(y2-y1+1);
  WriteHeader(handle, header);
  for i:=y1 to y2 do
    begin
      buf:=Ptr(SegA000, i*320+x1);
      BlockWrite(handle, buf^, x2-x1+1);
      IOStatus:=IOResult;
      if IOStatus<>0 then
        begin
          Seek(handle, dl);
          Truncate(handle);
          Exit
        end
    end;
  AddScrImage:=True;
  Seek(handle, OldPos);
  IOStatus:=IOResult
end;


{ Pusta procedura wykonywana podczas usuwania rysunkw. }

Procedure Proc_Del_Image(position : LongInt; const Id : string;
  arg1, arg2 : Word);
begin
end;


{ Skopiowanie pliku grafiki lecz bez rysunku o danym id. }
{ Praktycznie usuniecie danego rysunku z pliku.          }

Function CopyFile(var handle, f : file) : Boolean;
var
  szer, wys      : Word;
  ilosc          : LongInt;
  rozmiar        : Word;
  x, test        : Word;
  header         : THeader;
  P              : Pointer;
  OldPos, npos   : LongInt;
  ident          : string absolute header;
begin
  IOStatus:=0;
  CopyFile:=False;
  OldPos:=FilePos(handle);
  Seek(handle, 0);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  ilosc:=MaxAvail;
  if ilosc>MaxSpace then ilosc:=MaxSpace;
  GetMem(P, ilosc);
  if P=nil then Exit;
  while not EOF(handle) do
    begin
      if not ReadHeader(handle, header, szer, wys, True) then
        begin
          FreeMem(P, ilosc);
          CopyFile:=IOStatus=0;
          Exit
        end;
      rozmiar:=szer*wys;
      if FilePos(handle)=OldPos then
        begin
          Seek(handle, FilePos(handle)+header[1]+rozmiar+6-1);
          Continue
        end;
      Seek(handle, FilePos(handle)+header[1]+6-1);
      npos:=FilePos(f);
      if not WriteHeader(f, header) then
        begin
          FreeMem(P, ilosc);
          Exit
        end;
      Dec(ident[0]);
      Del_Image(npos, ident, szer, wys);
      x:=ilosc;
      if x>rozmiar then x:=rozmiar;
      while rozmiar>0 do
        begin
          BlockRead(handle, P^, x, test);
          BlockWrite(f, P^, test);
          IOStatus:=IOResult;
          if IOStatus<>0 then
            begin
              FreeMem(P, ilosc);
              Exit
            end;
          Dec(rozmiar, test)
        end
    end;
  FreeMem(P, ilosc);
  CopyFile:=True
end;


{ Usuniecie wszystkich rysunkow id z pliku. Zwraca FALSE, jezeli operacja }
{ zapisu sie nie powiodla.                                                }

Function DeleteImage(var handle : file; const name : string) : Boolean;
var
  f      : file;
  oldpos : LongInt;
  SDIR   : DirStr;
  SNAME  : NameStr;
  SEXT   : ExtSTr;
begin
  IOStatus:=0;
  DeleteImage:=False;
  if Eof(handle) then
    begin
      Del_Image:=Proc_Del_Image;
      Exit;
    end;
  OldPos:=FilePos(handle);
  IOStatus:=IOResult;
  if IOStatus<>0 then
    begin
      Del_Image:=Proc_Del_Image;
      Exit
    end;
  FSplit(name, SDIR, SNAME, SEXT);
  Assign(f, SDIR+'TMP.$$$');
  ReWrite(f, 1);
  IOStatus:=IOResult;
  if IOStatus<>0 then
    begin
      Del_Image:=Proc_Del_Image;
      Exit
    end;
  if CopyFile(handle, f)=False then
    begin
      Del_Image:=Proc_Del_Image;
      Close(f);
      Erase(f);
      if IOStatus=0 then IOStatus:=IOResult;
      Exit
    end;
  Close(handle);
  Close(f);
  Erase(handle);
  Assign(handle, SDIR+'TMP.$$$');
  Rename(handle, name);
  Reset(handle, 1);
  Seek(handle, OldPos);
  IOStatus:=IOResult;
  Del_Image:=Proc_Del_Image;
  DeleteImage:=IOStatus=0
end;


{ Odczytanie rysunku id z pliku. Zwraca FALSE w przypadku nieistnienia }
{ rysunku badz bledu odczytu pliku.                                    }

Function LoadImage(var handle : file; var buf) : Boolean;
var
  szer, wys : Word;
  header    : THeader;
begin
  IOStatus:=0;
  LoadImage:=False;
  if Eof(handle) or not ReadHeader(handle, header, szer, wys, True) then Exit;
  Seek(handle, FilePos(handle)+header[1]+2-1);
  BlockRead(handle, buf, szer*wys+4);
  IOStatus:=IOResult;
  if IOStatus=0 then LoadImage:=True
end;


{ Odczytanie rysunku id z pliku. Zwraca FALSE w przypadku nieistnienia  }
{ rysunku badz bledu odczytu pliku. Odczytany rysunek jest bezporednio }
{ umieszczany na ekranie.                                               }

Function LoadScrImage(var handle : file; x, y : Word) : Boolean;
var
  szer, wys : Word;
  i         : Word;
  buf, wsk  : PChar;
  header    : THeader;
  llen      : Word;
  wielk     : LongInt;
  test      : Word;
  Dest      : PChar;
  toread    : Word;
begin
  IOStatus:=0;
  LoadScrImage:=False;
  if Eof(handle) or not ReadHeader(handle, header, szer, wys, False) then Exit;
  llen:=szer;
  wielk:=MaxAvail;
  if wielk>MaxSpace-szer then wielk:=MaxSpace-szer;
  wielk:=(wielk div szer)*szer;
  GetMem(buf, wielk);
  if szer+x>320 then szer:=320-x;
  if wys+y>200 then wys:=200-y;
  Dest:=Ptr(SegA000, y*320+x);
  wsk:=buf+wielk; toread:=szer*wys;
  for i:=0 to wys-1 do
    begin
      wsk:=wsk+llen;
      if wsk-buf>=wielk then
        begin
          wsk:=buf;
          if toread<wielk then
            BlockRead(Handle, wsk^, toread, test)
           else
            begin
              BlockRead(Handle, wsk^, wielk, test);
              Dec(toread, test)
            end
        end;
      Move(wsk^, Dest^, szer);
      asm
        mov ax, 20d                     { 320 shr 4}
        add [word ptr Dest+2], ax
      end
    end;
  FreeMem(buf, wielk);

  IOStatus:=IOResult;
  if IOStatus=0 then LoadScrImage:=True
end;


{ Pobranie wielkosci rysunku id z pliku. Zwraca FALSE w przypadku braku }
{ rysunku o danym identyfikatorze lub w przypadku bledu odczytu pliku.  }

Function FileImageSize(var handle : file; var Size : Word) : Boolean;
var
  szer, wys : Word;
  header    : THeader;
begin
  IOStatus:=0;
  FileImageSize:=False;
  Size:=0;
  if not Eof(handle) and ReadHeader(handle, header, szer, wys, True) then
    begin
      Size:=szer*wys+4;
      FileImageSize:=True
    end
end;


{ Sprawdzenie istnienia rysunku o danym identyfikatorze w pliku. }

Function ExistPicId(var handle : file; id : string) : Boolean;
var
  szer, wys : Word;
  header    : THeader;
  id1       : string absolute header;
  oldpos    : LongInt;
begin
  IOStatus:=0;
  ExistPicId:=False;
  OldPos:=FilePos(handle);
  Seek(handle, 0);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  if Length(id)>254 then Dec(id[0]);
  id:=id+Chr(not (Ord(id[0])+1));
  while not Eof(handle) do
    begin
      if not ReadHeader(handle, header, szer, wys, True) then Exit;
      if id1<>id then
        begin
          Seek(handle, FilePos(handle)+header[1]+6+szer*wys-1);
          Continue
        end;
      ExistPicId:=True;
      Break
    end;
  Seek(handle, OldPos);
  IOStatus:=IOResult
end;


{ Pobranie informacji o rysunku o danym numerze w pliku. }

Function ImageInfo(var handle : file; var id : string;
  var szer, wys : Word) : Boolean;
var
  header : THeader;
  id1    : string absolute header;
begin
  IOStatus:=0;
  ImageInfo:=False;
  if Eof(handle) then Exit;
  if ReadHeader(handle, header, szer, wys, True) then ImageInfo:=True;
  id:=id1;
  Dec(id[0])
end;


{ Odszukanie nastepnego obrazka. Praktycznie przesuniecie wskaznika pliku. }

Function FindNextPic(var handle : file) : Boolean;
var
  szer, wys : Word;
  header    : THeader;
begin
  IOStatus:=0;
  FindNextPic:=False;
  if Eof(handle) then Exit;
  if not ReadHeader(handle, header, szer, wys, False) then Exit;
  Seek(handle, FilePos(handle)+szer*wys);
  IOStatus:=IOResult;
  FindNextPic:=True
end;


{ Wyszukanie rysunku o danym identyfikatorze. }

Function SearchPicById(var handle : file; id : string) : Boolean;
var
  szer, wys : Word;
  header    : THeader;
  id1       : string absolute header;
begin
  IOStatus:=0;
  SearchPicById:=False;
  if Length(id)>254 then Dec(id[0]);
  id:=id+Chr(not (Ord(id[0])+1));
  Seek(handle, 0);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  while not Eof(handle) do
    begin
      if not ReadHeader(handle, header, szer, wys, True) then Exit;
      if id1<>id then
        begin
          Seek(handle, FilePos(handle)+header[1]+6+szer*wys-1);
          Continue
        end;
      SearchPicById:=True;
      Break
    end
end;


{ Odczytanie palety z dysku. }

Function CzytajPalete(var handle : file; var pal : TScr_RGB;
  var first, cnt : Word) : Boolean;
var
  header : THeader;
begin
  IOStatus:=0;
  CzytajPalete:=False;
  if Eof(handle) then Exit;
  if not ReadHeader(handle, header, first, cnt, False) then Exit;
  BlockRead(handle, pal, cnt*3);
  IOStatus:=IOResult;
  if IOStatus=0 then CzytajPalete:=True
end;


{ Odczytanie palety z biezacej pozycji pliku i zapisanie jej do bufora. }

Function LoadPalette(var handle : file; var pal : TScr_RGB) : Boolean;
var
  first, cnt : Word;
begin
  LoadPalette:=CzytajPalete(handle, pal, first, cnt)
end;


{ Zaladowanie palety kolorow z dysku. Automatyczne uaktualnienie kolorow. }

Function LoadScrPalette(var handle : file) : Boolean;
var
  palette    : TScr_RGB;
  stan       : Boolean;
  first, cnt : Word;
begin
  stan:=CzytajPalete(handle, palette, first, cnt);
  LoadScrPalette:=stan;
  if stan then UsePalette(palette)
end;


{ Dopisanie palety kolorow na dysk. }

Function AddPalette(var handle : file; id : string; var pal : TScr_RGB;
  first, cnt : Word) : Boolean;
var
  dl     : LongInt;
  oldpos : LongInt;
  header : THeader;
  id1    : string absolute header;
begin
  IOStatus:=0;
  AddPalette:=False;
  OldPos:=FilePos(handle);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  dl:=FileSize(handle);
  Seek(handle, dl);
  if Length(id)>254 then Dec(id[0]);
  id1:=id+Chr(not (Ord(id[0])+1));
  header[header[1]+2]:=Lo(first);
  header[header[1]+3]:=Hi(first);
  header[header[1]+4]:=Lo(cnt);
  header[header[1]+5]:=Hi(cnt);
  if WriteHeader(handle, header) then
    begin
      BlockWrite(handle, pal, cnt*3);
      IOStatus:=IOResult;
    end;
  if IOStatus<>0 then
    begin
      Seek(handle, dl);
      Truncate(handle);
      Exit
    end
   else
    begin
      AddPalette:=True;
      Seek(handle, OldPos);
      IOStatus:=IOResult
    end
end;


{ Zapisanie aktualnej palety kolorow na dysk. }

Function AddScrPalette(var handle : file; id : string; first, cnt : Word) : Boolean;
var
  palette : TScr_RGB;
begin
  ReadPalette(palette);
  AddScrPalette:=AddPalette(handle, id, palette, first, cnt)
end;


{ Pobranie informacie o palecie o danym identyfikatorze. Brak palety    }
{ sygnalizowany jest przez funkcje, ktora zwroci wowczas wartosc FALSE. }

Function PaletteInfo(var handle : file; var id : string;
  var first, cnt : Word) : Boolean;
var
  header : THeader;
  id1    : string absolute header;
begin
  IOStatus:=0;
  PaletteInfo:=False;
  if Eof(handle) then Exit;
  first:=0; cnt:=0;
  if not ReadHeader(handle, header, first, cnt, True) then Exit;
  id:=id1;
  Dec(id[0]);
  PaletteInfo:=True
end;


{ Sprawdzenie istnienia palety o danym identyfikatorze w pliku. }

Function ExistPalId(var handle : file; id : string) : Boolean;
var
  first, cnt : Word;
  header     : THeader;
  id1        : string absolute header;
  oldpos     : LongInt;
begin
  IOStatus:=0;
  ExistPalId:=False;
  OldPos:=FilePos(handle);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  id:=id+Chr(not (Ord(id[0])+1));
  while not Eof(handle) do
    begin
      if not ReadHeader(handle, header, first, cnt, True) then Exit;
      if id1<>id then
        begin
          Seek(handle, FilePos(handle)+6+header[1]+cnt*3-1);
          Continue
        end;
      ExistPalId:=True;
      Break
    end;
  Seek(handle, OldPos);
  IOStatus:=IOResult
end;


{ Pusta procedura wykonywana podczas usuwania palet. }

Procedure Proc_Del_Palette(position : LongInt; const Id : string;
  arg1, arg2 : Word);
begin
end;


{ Usuniecie palety na pozycji pliku. }

Function DelPal(var handle, f : file) : Boolean;
var
  first, cnt     : Word;
  ilosc          : LongInt;
  rozmiar        : Word;
  x, test        : Word;
  header         : THeader;
  P              : Pointer;
  ident          : string absolute header;
  OldPos, npos   : LongInt;
begin
  IOStatus:=0;
  DelPal:=False;
  OldPos:=FilePos(handle);
  Seek(handle, 0);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  ilosc:=MaxAvail;
  if ilosc>MaxSpace then ilosc:=MaxSpace;
  GetMem(P, ilosc);
  if P=nil then Exit;
  while not EOF(handle) do
    begin
      if not ReadHeader(handle, header, first, cnt, True) then
        begin
          FreeMem(P, Ilosc);
          DelPal:=IOStatus=0;
          Exit
        end;
      rozmiar:=cnt*3;
      if FilePos(handle)=OldPos then
        begin
          Seek(handle, FilePos(handle)+header[1]+6+rozmiar-1);
          Continue
        end;
      Seek(handle, FilePos(handle)+header[1]+6-1);
      npos:=FilePos(f);
      if not WriteHeader(f, header) then
        begin
          FreeMem(P, ilosc);
          Exit
        end;
      Dec(ident[0]);
      Del_Image(npos, ident, first, cnt);
      x:=ilosc;
      if x>rozmiar then x:=rozmiar;
      while rozmiar>0 do
        begin
          BlockRead(handle, P^, x, test);
          BlockWrite(f, P^, test);
          IOStatus:=IOResult;
          if IOStatus<>0 then
            begin
              FreeMem(P, Ilosc);
              Exit
            end;
          Dec(rozmiar, test)
        end
    end;
  FreeMem(P, ilosc);
  DelPal:=True
end;


{ Usuniecie aktualnej palety z dysku. Zwraca FALSE, jezeli operacja }
{ zapisu sie nie powiodla.                                          }

Function DeletePalette(var handle : file; const name : string) : Boolean;
var
  f      : file;
  oldpos : LongInt;
  SDIR   : DirStr;
  SNAME  : NameStr;
  SEXT   : ExtSTr;
begin
  IOStatus:=0;
  DeletePalette:=False;
  if Eof(handle) then
    begin
      Del_Palette:=Proc_Del_Palette;
      Exit
    end;
  OldPos:=FilePos(handle);
  IOStatus:=IOResult;
  if IOStatus<>0 then
    begin
      Del_Palette:=Proc_Del_Palette;
      Exit
    end;
  FSplit(name, SDIR, SNAME, SEXT);
  Assign(f, SDIR+'TMP.$$$');
  ReWrite(f, 1);
  IOStatus:=IOResult;
  if IOStatus<>0 then
    begin
      Del_Palette:=Proc_Del_Palette;
      Exit
    end;
  if DelPal(handle, f)=False then
    begin
      Del_Palette:=Proc_Del_Palette;
      Close(f);
      Erase(f);
      if IOStatus=0 then IOStatus:=IOResult;
      Exit
    end;
  Close(handle);
  Close(f);
  Erase(handle);
  Assign(handle, SDIR+'TMP.$$$');
  Rename(handle, name);
  Reset(handle, 1);
  Seek(handle, OldPos);
  IOStatus:=IOResult;
  Del_Palette:=Proc_Del_Palette;
  DeletePalette:=IOStatus=0
end;


{ Ustawienie wskaznika na nastepna palete. }

function FindNextPal(var handle : file) : Boolean;
var
  first, cnt : Word;
  header     : THeader;
begin
  IOStatus:=0;
  FindNextPal:=False;
  if Eof(handle) then Exit;
  if not ReadHeader(handle, header, first, cnt, False) then Exit;
  Seek(handle, FilePos(handle)+cnt*3);
  IOStatus:=IOResult;
  if IOStatus=0 then FindNextPal:=True
end;


{ Ustawienie wskaznika pliku na palecie o danym id. }

Function SearchPalById(var handle : file; id : string) : Boolean;
var
  first, cnt : Word;
  header     : THeader;
  id1        : string absolute header;
begin
  IOStatus:=0;
  SearchPalById:=False;
  if Length(id)>254 then Dec(id[0]);
  id:=id+Chr(not (Ord(id[0])+1));
  Seek(handle, 0);
  IOStatus:=IOResult;
  if IOStatus<>0 then Exit;
  while not Eof(handle) do
    begin
      if not ReadHeader(handle, header, first, cnt, True) then Exit;
      if id1<>id then
        begin
          Seek(handle, FilePos(handle)+header[1]+6+cnt*3-1);
          Continue
        end;
      SearchPalById:=True;
      Break
    end
end;


{ Wykorzystanie palety uzytkownika. }

Procedure UsePalette(const pal : TScr_RGB);
var
  Reg : Registers;
begin
  with Reg do
    begin
      ah:=$10;
      al:=$12;
      bx:=0;
      cx:=256;
      es:=Seg(pal);
      dx:=Ofs(pal);
      Intr($10, reg)
    end
end;


{ Pobranie aktualnej palety kolorow. }

Procedure ReadPalette(var pal : TScr_RGB);
var
  reg : Registers;
begin
  with Reg do
    begin
      ah:=$10;
      al:=$17;
      bx:=0;
      cx:=256;
      es:=Seg(pal);
      dx:=Ofs(pal);
      Intr($10, reg)
    end
end;


end.