class User < ActiveRecord::Base
  has_one :spec
  has_one :faq
  has_one :blog
  has_many :friendships
  has_many :friends,
           :through => :friendships,
           :conditions => "status = 'accepted'",
           :order => :screen_name
  has_many :requested_friends,
             :through => :friendships,
             :source => :friend,
             :conditions => "status = 'requested'",
             :order => :created_at
    
  has_many :pending_friends,
           :through => :friendships,
           :source => :friend,
           :conditions => "status = 'pending'",
           :order => :created_at
  has_many :comments, :order => "created_at", :dependent => :destroy


  attr_accessor :remember_me
  attr_accessor :current_password
  acts_as_ferret :Fields => ['screen_name', 'email'] # ale NIE password

  # Maksymalne i minimalne długości wszystkich pól
  SCREEN_NAME_MIN_LENGTH = 4
  SCREEN_NAME_MAX_LENGTH = 20
  PASSWORD_MIN_LENGTH = 4
  PASSWORD_MAX_LENGTH = 40
  EMAIL_MAX_LENGTH = 50
  SCREEN_NAME_RANGE = SCREEN_NAME_MIN_LENGTH..SCREEN_NAME_MAX_LENGTH
  PASSWORD_RANGE = PASSWORD_MIN_LENGTH..PASSWORD_MAX_LENGTH
  
  # Rozmiary pól tekstowych wyświetlanych w widokach
  SCREEN_NAME_SIZE = 20
  PASSWORD_SIZE = 10
  EMAIL_SIZE = 30


  # Przetłumaczone nazwy pól
  HUMANIZED_ATTRIBUTES = {
  :email => "Adres e-mail",
  :screen_name => "Pseudonim",
  :password => "Hasło",
  :current_password => "Bieżące hasło",
  :password_confirmation => "Potwierdzenie hasła"
  }

  # Tłumaczy nazwy pól
  def self.human_attribute_name(attr)
    HUMANIZED_ATTRIBUTES[attr.to_sym] || super
  end

  validates_uniqueness_of :screen_name, :email, :message => "jest już zajęty."
  validates_length_of     :screen_name, :within => 4..20, :too_short => "jest zbyt krótki (minimum to 4 znaki).", :too_long => "jest zbyt długi (maksimum to 20 znaków)."
  validates_length_of     :password,    :within => 4..40, :too_short => "jest zbyt krótkie (minimum to 4 znaki).", :too_long => "jest zbyt długie (maksimum to 40 znaków)."
  validates_length_of     :email,  :maximum => 50, :message => "jest zbyt długi."
  validates_presence_of   :email, :message => "nie może być pusty."
  validates_length_of :email, :maximum => EMAIL_MAX_LENGTH , :message => "jest zbyt długi."
  validates_format_of :screen_name,
                      :with => /^[A-ZąćęłńóśźżĄĆĘŁŃÓŚŹŻ0-9_]*$/i,
                      :message => "może zawierać tylko litery, " +
                      "cyfry i znaki podkreślenia."
  validates_format_of :email,
                      :with => /^[A-Z0-9._%-]+@([A-Z0-9-]+\.)+[A-Z]{2,4}$/i,
                      :message => "musi być prawidłowym adresem e-mail."
  validates_confirmation_of :password, :message => "i jego potwierdzenie są różne"

  # Loguje użytkownika
  def login!(session)
    session[:user_id] = self.id
  end
  
  # Wylogowuje użytkownika.
  # Wylogowuje użytkownika.
def self.logout!(session, cookies)
  session[:user_id] = nil
  cookies.delete(:authorization_token)
end

  
  # Wyczyść hasło (zwykle w celu niewyświetlania go w widoku)
  def clear_password!
    self.password = nil
    self.password_confirmation = nil
    self.current_password = nil

  end

# Zapamiętuje użytkownika, aby zalogować go w przyszłości.
def remember!(cookies)
  cookie_expiration = 10.years.from_now
  cookies[:remember_me] = { :value => "1",
                            :expires => cookie_expiration }
        self.authorization_token = unique_identifier 
        save!
        cookies[:authorization_token] = {
            :value => authorization_token,
            :expires => cookie_expiration }
end

# Zapomina status zalogowania użytkownika
def forget!(cookies)
  cookies.delete(:remember_me)
  cookies.delete(:authorization_token)    
end

# Zwraca true, jeżeli użytkownika chce aby status zalogowania został zapamiętany
def remember_me?
  remember_me == "1"
end

# Zwraca true jeżeli hasło z params jest poprawne.
def correct_password?(params)
  current_password = params[:user][:current_password]
  password == current_password
end 

# Generuje komunikaty dla błędów hasła.
  def password_errors(params)
    # Użyj metody valid? modelu User, aby wygenerować komunikaty o błędach
    # w przypadku niezgodności haseł.
    self.password = params[:user][:password]
    self.password_confirmation = params[:user][:password_confirmation]
    valid?
    # Bieżące hasło jest niepoprawne, więc dodaj komunikat.
    errors.add(:current_password, "jest nieprawidłowe")
  end
  
  # Zwraca rozsądną nazwę użytkownika
  def name
    spec.full_name.or_else(screen_name)
  end
  
  def avatar
    Avatar.new(self)
  end





private

# Generuje unikatowy identyfikator użytkownika.
def unique_identifier
  Digest::SHA1.hexdigest("#{self.screen_name}:#{self.password}")
end





end
