require File.dirname(__FILE__) + '/../test_helper'
require 'user_controller'
include ApplicationHelper

# Ponownie zgłoś błędy wykryte przez kontroler
class UserController; def rescue_action(e) raise e end; end

class UserControllerTest < Test::Unit::TestCase
  def setup
    @controller = UserController.new
    @request = ActionController::TestRequest.new
    @response = ActionController::TestResponse.new
    # Ten użytkownik początkowo jest prawidłowy, ale możemy zmieniać jego atrybuty.
    @valid_user = users(:valid_user)
  end
  
  # Sprawdź, czy strona rejestracji zwraca odpowiedni formularz
# Sprawdź, czy strona rejestracji zwraca odpowiedni formularz
def test_registration_page
  get :register
  title = assigns(:title)
  assert_equal "Rejestracja", title
  assert_response :success
  assert_template "register"
  assert_form_tag "/user/register"
  assert_screen_name_field
  assert_email_field
  assert_password_field
  assert_password_field "password_confirmation"
  assert_submit_button "Zarejestruj!"
end

  # Test poprawnej rejestracji
  def test_registration_success
    post :register, :user => { :screen_name => "new_screen_name",
                               :email       => "valid@example.com",
                               :password    => "long_enough_password" }
    # Testuj przypisanie użytkownika
    user = assigns(:user)
    assert_not_nil user
    # Testuj nowego użytkownika w bazie danych
    new_user = User.find_by_screen_name_and_password(user.screen_name,
                                                     user.password)
    assert_equal new_user, user
    # Testuj komunikat flash i przekierowanie
    assert_equal "Utworzono konto użytkownika #{new_user.screen_name}!", flash[:notice]
    assert_redirected_to :action => "index"
    
    # Sprawdź, czy użytkownik został poprawnie zalogowany.
    assert logged_in?
    assert_equal user.id, session[:user_id]
  end

  # Test niepowodzenia rejestracji
  def test_registration_failure
    post :register, :user => { :screen_name => "aa/noyes",
                               :email => "anoyes@example,com",
                               :password => "sun"}
    assert_response :success
    assert_template "register"
    # Test wyświetlania komunikatów o błędach
    assert_tag "div", :attributes => { :id => "errorExplanation",
                                       :class => "errorExplanation" }
    # Sprawdza, czy dla każdego z pól zostal wyświetlony choć jeden komunikat o błędzie
    assert_tag "li", :content => /Pseudonim/
    assert_tag "li", :content => /Adres e-mail/
    assert_tag "li", :content => /Hasło/
  
    # Sprawdza, czy pola wprowadzania są umieszczane w odpowiednim <div>
    error_div = { :tag => "div", :attributes => { :class => "fieldWithErrors"}}
  
    assert_tag "input",
               :attributes => {:name => "user[screen_name]",
                               :value => "aa/noyes" },
                               :parent => error_div
    assert_tag "input",
               :attributes => { :name => "user[email]", 
                                :value => "anoyes@example,com" },
                           :parent => error_div
    assert_tag "input",
               :attributes => { :name => "user[password]",
                                :value => nil },
               :parent => error_div
  end
  
  # Sprawdź, czy strona logowania działa i zawiera odpowiednie pola.
  def test_login_page
    get :login
    title = assigns(:title)
    assert_equal "Logowanie do RailsSpace", title
    assert_response :success
    assert_template "login"
    assert_tag "form", :attributes => { :action => "/user/login",
                       :method => "post" }
    assert_tag "input",
               :attributes => { :name => "user[screen_name]",
               :type => "text",
               :size => User::SCREEN_NAME_SIZE,
               :maxlength => User::SCREEN_NAME_MAX_LENGTH }
    assert_tag "input",
               :attributes => { :name => "user[password]",
               :type => "password",
               :size => User::PASSWORD_SIZE,
               :maxlength => User::PASSWORD_MAX_LENGTH }
    assert_tag "input", :attributes => { :name => "user[remember_me]",
                                         :type => "checkbox" }
    assert_tag "input", :attributes => { :type => "submit",
                        :value => "Zaloguj!" }
  end
  
  # Test poprawnego logowania.
  def test_login_success
    try_to_login @valid_user, :remember_me => "0"
    assert logged_in?
    assert_equal @valid_user.id, session[:user_id]
    assert_equal "Użytkownik #{@valid_user.screen_name} zalogowany!", flash[:notice]
    assert_response :redirect
    assert_redirected_to :action => "index"
  
    # Upewnij się, że nie pamiętamy użytkownika
    user = assigns(:user)
    assert user.remember_me != "1"
    # Nie powinny być ustawione żadne cookie.
    assert_nil cookie_value(:remember_me)
    assert_nil cookie_value(:authorization_token)
  end

  # Test poprawnego logowania przy zaznaczonej opcji "pamiętaj mnie".
  def test_login_success_with_remember_me
    try_to_login @valid_user, :remember_me => "1"
    test_time = Time.now
    assert logged_in?
    assert_equal @valid_user.id, session[:user_id]
    assert_equal "Użytkownik #{@valid_user.screen_name} zalogowany!", flash[:notice]
    assert_response :redirect
    assert_redirected_to :action => "index"
  
    # Sprawdza cookie i okresy ważności
    user = User.find(@valid_user.id)
    time_range = 100 # zakres zgodności czasu w mikrosekundach
  
    # Cookie "pamiętaj mnie".
    assert_equal "1", cookie_value(:remember_me)    
    assert_in_delta 10.years.from_now(test_time),
                 cookie_expires(:remember_me),
                 time_range
  
    # Cookie autoryzacji.
    assert_equal user.authorization_token, cookie_value(:authorization_token)
    assert_in_delta 10.years.from_now(test_time),
                 cookie_expires(:authorization_token),
                 time_range
  end



  
  # Testowanie logowania przy podaniu nieprawidłowego pseudonimu.
  def test_login_failure_with_nonexistent_screen_name
    invalid_user = @valid_user
    invalid_user.screen_name = "no such user"
    try_to_login invalid_user
    assert_template "login"
    assert_equal "Nieprawidłowa kombinacja użytkownika/hasła", flash[:notice]
    # Upewnij się, że wyświetlone zostanie screen_name. ale nie hasło.
    user = assigns(:user)
    assert_equal invalid_user.screen_name, user.screen_name
    assert_nil user.password
  end

  # Testowanie logowania przy podaniu nieprawidłowego hasła.
  def test_login_failure_with_wrong_password
    invalid_user = @valid_user
    # Konstruuje nieprawidłowe hasło.
    invalid_user.password += "baz"
    try_to_login invalid_user
    assert_template "login"
    assert_equal "Nieprawidłowa kombinacja użytkownika/hasła", flash[:notice]
    # Upewnij się, że wyśwetlone zostanie screen_name. ale nie hasło.
    user = assigns(:user)
    assert_equal invalid_user.screen_name, user.screen_name
    assert_nil user.password
  end
  
  # Test funkcji wylogowania.
  def test_logout
    try_to_login @valid_user
    assert logged_in?
    get :logout
    assert_response :redirect
    assert_redirected_to :action => "index", :controller => "site"
    assert_equal "Wylogowano", flash[:notice]
    assert_nil session[:user_id]
  end
  
  # Test paska nawigacji po zalogowaniu użytkownika.
  def test_navigation_logged_in
    authorize @valid_user
    get :index
    assert_tag "a", :content => /Wyloguj/,
                    :attributes => { :href => "/user/logout" }
    assert_no_tag "a", :content => /Zarejestruj się/
    assert_no_tag "a", :content => /Zaloguj/
  end

  # Test strony głównej użytkownika przy braku autoryzacji.
  def test_index_unauthorized
    # Sprawdza, czy działa before_filter.
    get :index
    assert_response :redirect
    assert_redirected_to :action => "login"
    assert_equal "Proszę się zalogować!", flash[:notice]
  end

  #Test strony głównej uwierzytelnionego użytkownika.
  def test_index_authorized
    authorize @valid_user
    get :index
    assert_response :success
    assert_template "index"
  end

# Test przekierowania do chronionej strony po zalogowaniu.
def test_login_friendly_url_forwarding
  user = { :screen_name => @valid_user.screen_name,
           :password => @valid_user.password }
  friendly_url_forwarding_aux(:login, :index, user)
end

# Test powrotnego przekierowania do strony chronionej po zarejestrowaniu
def test_register_friendly_url_forwarding
  user = { :screen_name => "new_screen_name",
           :email => "valid@example.com",
           :password => "long_enough_password" }
  friendly_url_forwarding_aux(:register, :index, user)
end

# Test funkcji wylogowania.
def test_logout
  try_to_login @valid_user, :remember_me => "1"    
  assert logged_in?
  assert_not_nil cookie_value(:authorization_token)
  get :logout
  assert_response :redirect
  assert_redirected_to :action => "index", :controller => "site"
  assert_equal "Wylogowano", flash[:notice]
  assert !logged_in?
  assert_nil cookie_value(:authorization_token)
end

# Sprawdza, czy pole email ma poprawny HTML.
def assert_email_field(email = nil, options = {})
  assert_input_field("user[email]", email, "text",
                   User::EMAIL_SIZE, User::EMAIL_MAX_LENGTH,
                   options)
end
# Sprawdza, czy pole hasła (password) ma poprawny HTML.
def assert_password_field(password_field_name = "password", options = {})
  # Nie chcemy, aby hasło było ponownie wyświetlane w formularzu.
  blank = nil
  assert_input_field("user[#{password_field_name}]", blank, "password",
                      User::PASSWORD_SIZE, User::PASSWORD_MAX_LENGTH,
                      options)
end

# Sprawdza, czy pole pseudonimu (screen_name) ma poprawny HTML.
def assert_screen_name_field(screen_name = nil, options = {})
  assert_input_field("user[screen_name]", screen_name, "text",
                      User::SCREEN_NAME_SIZE, User::SCREEN_NAME_MAX_LENGTH,
                      options)
end

# Test strony edycji.
def test_edit_page
  authorize @valid_user
  get :edit
  title = assigns(:title)
  assert_equal "Edycja podstawowych informacji", title
  assert_response :success
  assert_template "edit"
  # Testuj formularz i jego wszystkie znaczniki.
  assert_form_tag "/user/edit"
  assert_email_field @valid_user.email
  assert_password_field "current_password"
  assert_password_field
  assert_password_field "password_confirmation"
  assert_submit_button "Aktualizuj"    
end



private

  # Próbuje zalogować użytkownika za pomocą akcji logowania.
  # Przesyła w opcjach :remember => "0" lub :remember_me => "1" 
  # w celu wywołania mechanizmu "pamiętaj mnie"
  def try_to_login(user, options = {})
    user_hash = { :screen_name => user.screen_name,
                  :password => user.password }
    user_hash.merge!(options)
    post :login, :user => user_hash
  end


  # Uwierzytelnienie użytkownika
  def authorize(user)
    @request.session[:user_id] = user.id
  end

  def friendly_url_forwarding_aux(test_page, protected_page, user)
    get protected_page
    assert_response :redirect
    assert_redirected_to :action => "login"
    post test_page, :user => user
    assert_response :redirect
    assert_redirected_to :action => protected_page
    # Upewnij się, że url przekierowujący został wyczyszczony.
    assert_nil session[:protected__page]
  end

  # Zwraca wartość cookie określanego przez symbol.
  def cookie_value(symbol)
    cookies[symbol.to_s].value.first
  end
  
  # Zwraca okes ważności cookie określonego przez symbol.
  def cookie_expires(symbol)
    cookies[symbol.to_s].expires
  end


end
