"""
Przeprowadza symulację tranzytu egzoplanety, rysuje krzywą blasku 
i szacuje promień egzoplanety.
"""
import cv2 as cv
import matplotlib.pyplot as plt

IMG_HT = 400
IMG_WIDTH = 500
BLACK_IMG = cv.imread('limb_darkening.png', cv.IMREAD_GRAYSCALE)
EXO_RADIUS = 7
EXO_DX = 3
EXO_START_X = 40
EXO_START_Y = 230
NUM_FRAMES = 145


def main():
    intensity_samples = record_transit(EXO_START_X, EXO_START_Y)
    relative_brightness = calc_rel_brightness(intensity_samples)
    plot_light_curve(relative_brightness)


def record_transit(exo_x, exo_y):
    """
    Rysuje tranzyt planety na tle gwiazdy
    i zwraca listę zmian intensywności.
    """
    intensity_samples = []
    for _ in range(NUM_FRAMES):
        temp_img = BLACK_IMG.copy()
        # Rysuje egzoplanetę.
        cv.circle(temp_img, (exo_x, exo_y), EXO_RADIUS, 0, -1)
        intensity = temp_img.mean()
        cv.putText(temp_img, 'Srednia intensywnosc = {}'.format(intensity),
                   (5, 390), cv.FONT_HERSHEY_PLAIN, 1, 255)
        cv.imshow('Tranzyt', temp_img)
        cv.waitKey(30)
        intensity_samples.append(intensity)
        exo_x += EXO_DX
    return intensity_samples


def calc_rel_brightness(intensity_samples):
    """Zwraca listę wyników względnej jasności na podstawie intensywności."""
    rel_brightness = []
    max_brightness = max(intensity_samples)
    for intensity in intensity_samples:
        rel_brightness.append(intensity / max_brightness)
    return rel_brightness


def plot_light_curve(rel_brightness):
    """Rysuje wykres zmian względnej jasności w czasie."""
    plt.plot(rel_brightness, color='red', linestyle='dashed',
             linewidth=2, label='Względna jasność')
    plt.legend(loc='upper center')
    plt.title('Względna jasność w czasie')
    ## plt.ylim(0.995, 1.001)
    plt.show()


if __name__ == '__main__':
    main()
