import numpy as np

def sigmoid(input):
    return 1.0 / (1 + np.exp(-input))

def compute_prediction(X, weights):
    """
    Funkcja wyliczająca prognozę y_hat z wykorzystaniem bieżących wag
    """
    z = np.dot(X, weights)
    predictions = sigmoid(z)
    return predictions

def update_weights_gd(X_train, y_train, weights, learning_rate):
    """
    Funkcja modyfikująca wagi w bieżącym kroku
    """
    predictions = compute_prediction(X_train, weights)
    weights_delta = np.dot(X_train.T, y_train - predictions)
    m = y_train.shape[0]
    weights += learning_rate / float(m) * weights_delta
    return weights

def compute_cost(X, y, weights):
    """
    Funkcja wyliczająca wagi J(w)
    """
    predictions = compute_prediction(X, weights)
    cost = np.mean(-y * np.log(predictions) - (1 - y) * np.log(1 - predictions))
    return cost

def train_logistic_regression(X_train, y_train, max_iter, learning_rate, fit_intercept=False):
    """ Funkcja trenująca model regresji logistycznej
    Args:
        X_train, y_train (numpy.ndarray, treningowy zbiór danych)
        max_iter (int, liczba iteracji)
        learning_rate (float)
        fit_intercept (bool, flaga: z przechwyceniem w0, czy bez niego)
    Wynik:
        numpy.ndarray, wyliczone wagi
    """
    if fit_intercept:
        intercept = np.ones((X_train.shape[0], 1))
        X_train = np.hstack((intercept, X_train))
    weights = np.zeros(X_train.shape[1])
    for iteration in range(max_iter):
        weights = update_weights_gd(X_train, y_train, weights, learning_rate)
        # Wyświetlenie kosztu co 100 iteracji
        if iteration % 100 == 0:
            print(compute_cost(X_train, y_train, weights))
    return weights

def predict(X, weights):
    if X.shape[1] == weights.shape[0] - 1:
        intercept = np.ones((X.shape[0], 1))
        X = np.hstack((intercept, X))
    return compute_prediction(X, weights)

X_train = np.array([[6, 7],
                    [2, 4],
                    [3, 6],
                    [4, 7],
                    [1, 6],
                    [5, 2],
                    [2, 0],
                    [6, 3],
                    [4, 1],
                    [7, 2]])

y_train = np.array([0,
                    0,
                    0,
                    0,
                    0,
                    1,
                    1,
                    1,
                    1,
                    1])

weights = train_logistic_regression(X_train, y_train, max_iter=1000, learning_rate=0.1, fit_intercept=True)

X_test = np.array([[6, 1],
                   [1, 3],
                   [3, 1],
                   [4, 5]])

predictions = predict(X_test, weights)

import matplotlib.pyplot as plt
plt.scatter(X_train[:,0], X_train[:,1], c=['b']*5+['k']*5, marker='o')
colours = ['k' if prediction >= 0.5 else 'b' for prediction in predictions]
plt.scatter(X_test[:,0], X_test[:,1], marker='*', c=colours)
plt.xlabel('x1')
plt.ylabel('x2')
plt.show()

import pandas as pd
n_rows = 300000
df = pd.read_csv("train", nrows=n_rows)

X = df.drop(['click', 'id', 'hour', 'device_id', 'device_ip'], axis=1).values
Y = df['click'].values

n_train = 100000
X_train = X[:n_train]
Y_train = Y[:n_train]
X_test = X[n_train:]
Y_test = Y[n_train:]

from sklearn.preprocessing import OneHotEncoder
enc = OneHotEncoder(handle_unknown='ignore')
X_train_enc = enc.fit_transform(X_train)

X_test_enc = enc.transform(X_test)

import timeit
start_time = timeit.default_timer()
weights = train_logistic_regression(X_train_enc.toarray(), Y_train, max_iter=100000, learning_rate=0.01,
                                    fit_intercept=True)
print(f"--- {(timeit.default_timer() - start_time)}.3f s ---")

pred = predict(X_test_enc.toarray(), weights)
from sklearn.metrics import roc_auc_score
print(f'Liczba próbek treningowych: {n_train}, pole pod krzywą ROC dla zbioru treningowego: {roc_auc_score(Y_test, pred):.3f}')

def update_weights_sgd(X_train, y_train, weights, learning_rate):
    """ Pojedyncza iteracja modyfikująca wagi: jeden krok na bazie pojedynczej próbki
    Argumenty:
        X_train, y_train (numpy.ndarray, zbiór treningowy)
        weights (numpy.ndarray)
        learning_rate (float)
    Wynik:
        numpy.ndarray, zmodyfikowane wagi
    """
    for X_each, y_each in zip(X_train, y_train):
        prediction = compute_prediction(X_each, weights)
        weights_delta = X_each.T * (y_each - prediction)
        weights += learning_rate * weights_delta
    return weights

def train_logistic_regression_sgd(X_train, y_train, max_iter, learning_rate, fit_intercept=False):
    """ Trening modelu wykorzystującego stochastyczny gradient prosty
    Argumenty:
        X_train, y_train (numpy.ndarray, zbiór treningowy)
        max_iter (int, liczba iteracji)
        learning_rate (float)
        fit_intercept (bool, flaga: z przechwyceniem w0, czy bez niego)
    Wynik:
        numpy.ndarray, wyliczone wagi
    """
    if fit_intercept:
        intercept = np.ones((X_train.shape[0], 1))
        X_train = np.hstack((intercept, X_train))
    weights = np.zeros(X_train.shape[1])
    for iteration in range(max_iter):
        weights = update_weights_sgd(X_train, y_train, weights, learning_rate)
        # Wyświetlenie kosztu co 2 iteracje
        if iteration % 2 == 0:
            print(compute_cost(X_train, y_train, weights))
    return weights

start_time = timeit.default_timer()
weights = train_logistic_regression_sgd(X_train_enc.toarray(), Y_train, max_iter=10, learning_rate=0.01,
                                        fit_intercept=True)
print(f"--- {(timeit.default_timer() - start_time)}.3f s ---")
pred = predict(X_test_enc.toarray(), weights)
print(f'Liczba próbek treningowych: {n_train}, pole pod krzywą ROC dla zbioru treningowego: {roc_auc_score(Y_test, pred):.3f}')