#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Przybliżony licznik Morrisa obsługujący wiele liczników"""
from __future__ import division  # 1/2 == 0.5 jak w języku Python 3
# Uniknięcie ukrywania globalnych modułów z elementami lokalnymi
from __future__ import absolute_import
from __future__ import print_function  # Wymuszenie użycia instrukcji print("Witaj")
# Wymuszenie prostych łańcuchów "" w formacie Unicode bez przedrostka u""
from __future__ import unicode_literals
import math
import random
import array

SMALLEST_UNSIGNED_INTEGER = b'B'  # zwykle 1 bajt w systemach 64-bitowych
# b'I' liczba całkowita bez znaku; 4 bajty w systemach 64-bitowych
# b'L' liczba całkowita bez znaku; 8 bajtów w systemach 64-bitowych


class MorrisCounter(object):

    """Przybliżony licznik, przechowuje wykładnik i oblicza przybliżenie 2^wykładnik

    https://en.wikipedia.org/wiki/Approximate_counting_algorithm"""

    def __init__(self, type_code=SMALLEST_UNSIGNED_INTEGER, nbr_counters=1):
        self.exponents = array.array(type_code, [0] * nbr_counters)

    def __len__(self):
        return len(self.exponents)

    def add_counter(self):
        """Dodanie nowego wyzerowanego licznika"""
        self.exponents.append(0)

    def get(self, counter=0):
        """Obliczenie przybliżonej wartości reprezentowanej przez licznik"""
        return math.pow(2, self.exponents[counter])

    def add(self, counter=0):
        """Probabilistyczne dodanie 1 do licznika"""
        value = self.get(counter)
        probability = 1.0 / value
        if random.uniform(0, 1) < probability:
            self.exponents[counter] += 1

if __name__ == "__main__":
    mc = MorrisCounter()
    print("MorrisCounter ma {} liczników.".format(len(mc)))
    for n in range(10):
        print("Iteracja %d, MorrisCounter ma: %d" % (n, mc.get()))
        mc.add()

    for n in xrange(990):
        mc.add()
    print("Iteracja 1000, MorrisCounter ma: %d" % (mc.get()))
