import timeit

import numpy as np
import pylab as py
from scipy import sparse


def benchmark(size=2048):
    densities = np.asarray([0.001, 0.005, 0.01, 0.05, 0.1, 0.15, 0.2])
    results = {"sparse": [], "dense": []}
    for density in densities:
        m_sparse = sparse.random(size, size, density).tocsr()
        m_dense = m_sparse.todense()
        N_sparse, total_time_sparse = timeit.Timer(
            "m_sparse * m_sparse", globals=locals()
        ).autorange()
        N_dense, total_time_dense = timeit.Timer(
            "m_dense * m_dense", globals=locals()
        ).autorange()
        time_sparse = total_time_sparse / N_sparse
        time_dense = total_time_dense / N_dense
        results["sparse"].append(time_sparse)
        results["dense"].append(time_dense)
        print(time_sparse, time_dense)

    fig = py.figure()
    py.plot(
        densities * 100, results["sparse"], "--", marker="s", label="Macierz rzadka CSR"
    )
    py.plot(densities * 100, results["dense"], "-", marker="o", label="Tablica Numpy")
    py.title(f"Czas trwania operacji mnożenia {size}x{size} w wypadku różnej gęstości macierzy")
    py.ylabel("Czas działania (sekundy)")
    py.xlabel("Gęstość (procent elementów niezerowych)")
    py.legend()
    ax = py.gca()
    py.tight_layout()
    py.savefig("../../images/sparse_runtime.png")
    py.close(fig)


def matrix_size(size=2028):
    densities = np.linspace(0, 100, 10)
    sparse_size = []
    dense_size = []
    for density in densities:
        m_sparse = sparse.random(size, size, density / 100).tocsr()
        m_dense = m_sparse.todense()

        sparse_size.append(
            (m_sparse.data.nbytes + m_sparse.indices.nbytes + m_sparse.indptr.nbytes)
            / 1e6
        )
        dense_size.append(m_dense.data.nbytes / 1e6)

    fig = py.figure()
    py.plot(densities, sparse_size, "--", marker="s", label="Macierz rzadka CSR")
    py.plot(densities, dense_size, "-", marker="o", label="Tablica Numpy")
    py.title(f"Wykorzystanie pamięci dla macierzy {size}x{size}")
    py.ylabel("Rozmiar (MB)")
    py.xlabel("Gęstość (procent elementów niezerowych)")
    py.legend()
    ax = py.gca()
    py.tight_layout()
    py.savefig("../../images/sparse_footprint.png")
    py.close(fig)


if __name__ == "__main__":
    matrix_size()
    benchmark()
