﻿# SimpleLanguageTranslator.py

   
""" Aplikacja ułatwiająca porozumiewanie się dwóch rozmówców,
    z których jeden posługuje się językiem angielskim, a drugi 
    hiszpańskim. Wykorzystuje API następujących usług platformy
    IBM Watson:
       
       Speech to Text
       Text to Speech
       Language Translator
"""   
   
from watson_developer_cloud import SpeechToTextV1
from watson_developer_cloud import LanguageTranslatorV3
from watson_developer_cloud import TextToSpeechV1
import keys  # zawiera klucze API dla poszczególnych usług
import pyaudio  # wykorzystywany do nagrywania przez mikrofon
import pydub  # wykorzystywany do ładowania pliku WAV
import pydub.playback  # wykorzystywany do odtwarzania pliku WAV
import wave  # wykorzystywany do zapisywania w pliku WAV

def prepare_input(aMsg1, aMsg2, aFilenameRecord, aFileNameDefault):
    print()
    print('Wprowadź:')
    print('  n - by nagrać ' + aMsg1)
    print('  g - by wykorzystać ' + aMsg2 + ' (' + aFileNameDefault + ')')
    print('i naciśnij Enter')
    print()
    ready = False
    while not ready:
      k = input(' ?')
      
      ready = (k == 'n') or (k == 'g')
      
      if not ready:
          print('Nieprawidłowy klawisz - wprowadź ponownie')
    
    if k == 'n':   # nagrywanie
        print('Zacznij mówić - masz 5 sekund')
        record_audio(aFilenameRecord)
        play_audio(aFilenameRecord)
        result_filename = aFilenameRecord
    else:
        print('Wykorzystuję zawartość pliku ' + aFileNameDefault) 
        result_filename = aFileNameDefault    
        play_audio(file_name=aFileNameDefault)
        
    return result_filename        
        
def run_translator():
    """Wywołuje funkcje realizujące interakcję 
       z usługami IBM Watson"""
   
    # Krok 1. Sygnał do zadania pytania w języku angielskim
    #         i nagrywanie głosu do pliku audio
    english_question_file_name = prepare_input(
                                         'pytanie po angielsku',
                                         'nagrane pytanie',
                                         'english_prepared.wav',
                                         'english.wav'
                                              )

    # Krok 2. Konwersja mowy angielskiej na tekst 
    english = speech_to_text(
        file_name=english_question_file_name, model_id='en-US_BroadbandModel')
    print('Angielski: ', english)

    # Krok 3. Tłumaczenie tekstu angielskiego na język hiszpański
    spanish = translate(text_to_translate=english, model='en-es')
    print('Hiszpański: ', spanish)

    # Krok 4. Synteza głosu na podstawie tekstu hiszpańskiego
    text_to_speech(text_to_speak=spanish, voice_to_use='es-US_SofiaVoice',
        file_name='spanish.wav')

    # Krok 5. Odtworzenie pliku audio ze zsyntetyzowaną 
    # mową hiszpańską
    play_audio(file_name='spanish.wav')

    # Krok 6. Sygnał do udzielenia odpowiedzi w języku hiszpańskim
    #         i nagrywanie głosu do pliku audio
    spanish_response_file_name = prepare_input(
                                         'odpowiedź po hiszpańsku',
                                         'nagraną odpowiedź',
                                         'spanishresponse_prepared.wav',
                                         'spanishresponse.wav'
                                              )
    
    # Krok 7. Konwersja mowy hiszpańskiej na tekst
    spanish = speech_to_text(
        file_name=spanish_response_file_name, model_id='es-ES_BroadbandModel')
    print('Odpowiedź po hiszpańsku: ', spanish)

    # Krok 8. Tłumaczenie tekstu hiszpańskiego na język angielski
    english = translate(text_to_translate=spanish, model='es-en')
    print('Odpowiedź po angielsku: ', english)

    # Krok 9. Synteza głosu na podstawie tekstu angielskiego
    text_to_speech(text_to_speak=english,
        voice_to_use='en-US_AllisonVoice',
        file_name='englishresponse.wav')

    # Krok 10. Odtworzenie pliku audio ze zsyntetyzowaną 
    # mową angielską
    play_audio(file_name='englishresponse.wav')


def speech_to_text(file_name, model_id):
    """Wywołanie usługi Watson Speech to Text w celu
       skonwertowania głosu na tekst pisany""" 
    
    # utworzenie klienta usługi
    stt = SpeechToTextV1(iam_apikey=keys.speech_to_text_key)

    # otwarcie pliku audio
    with open(file_name, 'rb') as audio_file:
        # przekazanie zawartości pliku do konwersji
        result = stt.recognize(audio=audio_file,
            content_type='audio/wav', model=model_id).get_result()
    
    # Pobranie listy wyników. Może ona zawierać, oprócz wyniku
    # końcowego, także wyniki pośrednie, zależnie od żądanej metody
    # rozpoznawania głosu. W tym przypadku żądamy tylko końcowego
    # wyniku, więc lista zawiera tylko jeden element. 
    
    results_list = result['results'] 

    # Pobierz końcowy wynik rozpoznawania
    speech_recognition_result  = results_list[0]
    
    # Pobranie listy wariantów transkrypcji.Jej liczebność
    # zależna jest od żądanej metody rozpoznawania głosu.
    # W tym przypadku lista zawiera tylko jeden element.
    alternatives_list = speech_recognition_result['alternatives']
    
    first_alternative = alternatives_list[0]

    # Pobranie tekstu stanowiącego wynik transkrypcji
    # i zwrócenie go jako wynik
    transcript = first_alternative['transcript']
    return transcript


def translate(text_to_translate, model):
    """Wywołanie usługi Watson Language Translator w celu
       przetłumaczenia tekstu angielskiego na hiszpański
       (model en-es) albo hiszpańskiego na angielski 
       (model es-en)"""   
   
    # utworzenie klienta usługi
    language_translator = LanguageTranslatorV3(version='2018-05-31',
        iam_apikey=keys.translate_key)

    # wykonanie tłumaczenia
    translated_text = language_translator.translate(
        text=text_to_translate, model_id=model).get_result()
    
    # Pobranie listy wynikowych łańcuchów - każdy z nich 
    # odpowiada jednemu łańcuchowi źródłowemu. W tym przypadku
    # przekazujemy tylko jeden łańcuch źródłowy, więc lista 
    # wyników zawiera tylko jeden element.
    translations_list = translated_text['translations']
    first_translation = translations_list[0]

    # pobranie przetłumaczonego tekstu i zwrócenie go 
    # jako wynik
    translation = first_translation['translation']
    return translation

def text_to_speech(text_to_speak, voice_to_use, file_name):
    """Wywołanie usługi Watson Text to Speech w celu
       zsyntetyzowania mowy, z żądaną barwą głosu i zapisanie
       wyniku syntezy w pliku WAV o żądanej nazwie."""
       
    # utworzenie klienta usługi
    tts = TextToSpeechV1(iam_apikey=keys.text_to_speech_key)

    # utworzenie pliku, do którego zapisany zostanie 
    # wynik syntezy
    with open(file_name, 'wb') as audio_file:
        audio_file.write(tts.synthesize(text_to_speak, 
            accept='audio/wav', voice=voice_to_use).get_result().content)

def record_audio(file_name):
    """Zapisanie krótkiego (do 5 sekund) nagrania
       w pliku WAV o podanej nazwie, z wykorzystaniem 
       modułu pyaudio"""
    
    FRAME_RATE = 44100  # liczba ramek na sekundę
    CHUNK = 1024  # liczba ramek odczytywanych jednorazowo
    FORMAT = pyaudio.paInt16  # każda ramka jest 16-bitową 
                              # liczbą całkowitą
    CHANNELS = 2  # 2 sample na ramkę
    SECONDS = 5  # całkowity czas nagrania
 
    # otwarcie i zamknięcie strumieni audio
    recorder = pyaudio.PyAudio() 

    # skonfigurowanie i otwarcie strumienia audio
    # do nagrywania (input=True)
    audio_stream = recorder.open(format=FORMAT, channels=CHANNELS, 
        rate=FRAME_RATE, input=True, frames_per_buffer=CHUNK)
   
    # Zapisywanie surowych bajtów z wejścia mikrofonowego
    audio_frames = []  
    # print('Nagrywanie 5-sekundowe.')

    # odczytywanie nagrania w porcjach określonych przez CHUNK
    for i in range(0, int(FRAME_RATE * SECONDS / CHUNK)):
        audio_frames.append(audio_stream.read(CHUNK))

    print('Nagrywanie zakończone.')
    audio_stream.stop_stream()  # zatrzymanie nagrywania
    audio_stream.close()  
    recorder.terminate()  # zwolnienie zasobów używanych przez pyaudio

    # zapisanie ramek audio w pliku WAV
    with wave.open(file_name, 'wb') as output_file:
        output_file.setnchannels(CHANNELS)
        output_file.setsampwidth(recorder.get_sample_size(FORMAT))
        output_file.setframerate(FRAME_RATE)
        output_file.writeframes(b''.join(audio_frames))

def play_audio(file_name):
    """Odtworzenie zawartości pliku WAV o podanej nazwie
       za pomocą modułu pydub"""
    sound = pydub.AudioSegment.from_wav(file_name)
    pydub.playback.play(sound)

if __name__ == '__main__':
    run_translator()


##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
