﻿# sentimentlisener.py

"""Script that searches for tweets that match a search string
and tallies the number of positive, neutral and negative tweets."""

import keys
import preprocessor as p 
import sys
from textblob import TextBlob
import tweepy

class SentimentListener(tweepy.StreamListener):
    """Obsługuje strumień nadchodzących tweetów."""

    def __init__(self, api, sentiment_dict, topic, limit=10):
        """Konfigurowanie obiektu-obserwatora"""
        self.sentiment_dict = sentiment_dict
        self.tweet_count = 0
        self.topic = topic
        self.TWEET_LIMIT = limit

        # konfigurowanie preprocesora w celu usuwania
        # URL-i i zarezerwowanych słów Twittera.
        p.set_options(p.OPT.URL, p.OPT.RESERVED) 
        
        super().__init__(api)  # wywołanie __init__ z nadklasy
  
    def on_status(self, status):
        """Metoda wywoływana, gdy Twitter dostarcza nowy tweet"""
        
        # pobranie treści tweetu
        try:  
            tweet_text = status.extended_tweet.full_text
        except: 
            tweet_text = status.text

        # ignorowanie retweetów
        if tweet_text.startswith('RT'):
            return

        tweet_text = p.clean(tweet_text)  # normalizacja tweetu
        
        # ignorowanie tweetów, w których temat nie pojawia się w treści
        if self.topic.lower() not in tweet_text.lower():
            return

        # aktualizacja słownika sentymentów pod kątem 
        # polaryzacji bieżącego tweetu
        blob = TextBlob(tweet_text)
        if blob.sentiment.polarity > 0:
            sentiment = '+'
            self.sentiment_dict['positive'] += 1 
        elif blob.sentiment.polarity == 0:
            sentiment = ' '
            self.sentiment_dict['neutral'] += 1 
        else:
            sentiment = '-'
            self.sentiment_dict['negative'] += 1 
            
        # wyświetlenie tweetu
        print(f'{sentiment} {status.user.screen_name}: {tweet_text}\n')
        
        self.tweet_count += 1  # licznik przetworzonych tweetów

        # gdy przekroczony zostanie limit przetworzonych tweetów
        # (TWEET_LIMIT) zwróć False w celu przerwania strumieniowania,
        # w przeciwnym razie zwróć True.
        return self.tweet_count <= self.TWEET_LIMIT

def main():

    # weryfikacja i pobranie parametrów uruchamiania
    if len(sys.argv) < 3:
        print('Wymagane parametry uruchamiania:')
        print('  1. Badany temat')
        print('  2. Limit uwzględnianych tweetów')
        sys.exit

    search_key = sys.argv[1]  # fraza-kryterium wyszukiwania
    limit = int(sys.argv[2])  # limit analizowanych tweetów


    # konfigurowanie obiektu uwierzytelniania
    auth = tweepy.OAuthHandler(keys.consumer_key, keys.consumer_secret)
    auth.set_access_token(keys.access_token, keys.access_token_secret)

    # utworzenie obiektu api
    api = tweepy.API(auth, wait_on_rate_limit=True, 
                     wait_on_rate_limit_notify=True)
                 
    # utworzenie obiektu-obserwatora
    sentiment_dict = {'positive': 0, 'neutral': 0, 'negative': 0}
    sentiment_listener = SentimentListener(api, 
        sentiment_dict, search_key, limit)

    # skonfigurowanie strumienia
    stream = tweepy.Stream(auth=api.auth, listener=sentiment_listener)

    # rozpoczęcie pobierania tweetów, 
    # wyłącznie w języku angielskim
    # zawierających żądaną frazę
    stream.filter(track=[search_key], languages=['en'], is_async=False)  

    print(f'Sentyment tweetów zawierających "{search_key}"')
    print('Pozytywne:', sentiment_dict['positive'])
    print(' Obojętne:', sentiment_dict['neutral'])
    print('Negatywne:', sentiment_dict['negative'])

# wywołanie metody "main", gdy niniejszy plik 
# uruchomiony zostanie jako skrypt 
if __name__ == '__main__':
    main()

##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
