# Rozdzia 16. Big Data: Hadoop, Spark, NoSQL i IoT
## 16.2. Relacyjne bazy danych i jzyk SQL
#### 16.2.4.1. Tworzenie bazy danych
# sqlite3 books.db < books.sql

import sqlite3
connection = sqlite3.connect('books.db')

#### 16.2.4.3. Tabela authors 
##### 16.2.4.3.1. Ogldanie zawartoci tabeli
import pandas as pd
pd.options.display.max_columns = 10
pd.read_sql('SELECT * FROM authors', connection,
            index_col=['id'])

#### 16.2.4.4. Tabela titles 
pd.read_sql('SELECT * FROM titles', connection)

#### 16.2.4.5. Tabela author_ISBN 
df = pd.read_sql('SELECT * FROM author_ISBN', connection)
df.head()

### 16.2.6. Zapytania SQL SELECT
pd.read_sql('SELECT first, last FROM authors', connection)

#### 16.2.6.1. Klauzula WHERE
pd.read_sql("""SELECT title, edition, copyright
               FROM titles
               WHERE copyright > '2016'""", connection)

##### 16.2.6.1.1. Dopasowywanie wzorca  zero lub wicej znakw
pd.read_sql("""SELECT id, first, last
               FROM authors
               WHERE last LIKE 'D%'""",
            connection, index_col=['id'])

##### 16.2.6.1.2. Dopasowywanie wzorca  dowolny znak
pd.read_sql("""SELECT id, first, last
               FROM authors
               WHERE first LIKE '_b%'""",
            connection, index_col=['id'])

####16.2.6.2. Klauzula ORDER BY
pd.read_sql('SELECT title FROM titles ORDER BY title ASC',
            connection)

##### 16.2.6.2.1. Sortowanie wzgldem wielu kolumn
pd.read_sql("""SELECT id, first, last
               FROM authors
               ORDER BY last, first""",
            connection, index_col=['id'])

pd.read_sql("""SELECT id, first, last
               FROM authors
               ORDER BY last DESC, first ASC""",
            connection, index_col=['id'])

#### 16.2.6.3. Kombinacja klauzul WHERE i ORDER BY
pd.read_sql("""SELECT isbn, title, edition, copyright
               FROM titles
               WHERE title LIKE '%How to Program'
               ORDER BY title""", connection)

#### 16.2.6.4. Zczenia tabel  INNER JOIN
pd.read_sql("""SELECT first, last, isbn
               FROM authors
               INNER JOIN author_ISBN
               ON authors.id = author_ISBN.id
               ORDER BY last, first""", connection).head()

### 16.2.7. Instrukcja INSERT INTO
cursor = connection.cursor()
cursor = cursor.execute("""INSERT INTO authors (first, last)
                                   VALUES ('Sue', 'Red')""")

pd.read_sql('SELECT id, first, last FROM authors',
             connection, index_col=['id'])

### 16.2.8. Instrukcja UPDATE
cursor = cursor.execute("""UPDATE authors SET last='Black'
                           WHERE last='Red' AND first='Sue'""")
cursor.rowcount
pd.read_sql('SELECT id, first, last FROM authors',
            connection, index_col=['id'])

### 16.2.9. Instrukcja DELETE FROM
cursor = cursor.execute('DELETE FROM authors WHERE id=6')
cursor.rowcount
pd.read_sql('SELECT id, first, last FROM authors',
            connection, index_col=['id'])

### 16.2.10. Zamykanie bazy danych
connection.close()

## 16.4. Analiza przypadku: dokumenty JSON w bazie MongoDB
# https://www.mongodb.com/download-center/community

### 16.4.1. Instalowanie bibliotek Pythona dla interakcji z MongoDB
# conda install -c conda-forge pymongo
# conda install -c conda-forge dnspython

### 16.4.2. Plik keys.py

### 16.4.3. Tworzenie klastera MongoDB Atlas
# https://mongodb.com/

#### 16.4.4.1. Uwierzytelnianie w Twitterze za pomoc biblioteki Tweepy
import tweepy, keys

auth = tweepy.OAuthHandler(
   keys.consumer_key, keys.consumer_secret)
auth.set_access_token(keys.access_token,
   keys.access_token_secret)

api = tweepy.API(auth, wait_on_rate_limit=True,
                 wait_on_rate_limit_notify=True)
                 
#### 16.4.4.2. Zaadowanie danych o senatorach
import pandas as pd
senators_df = pd.read_csv('senators.csv')
senators_df['ID_Twittera'] = senators_df['ID_Twittera'].astype(str)
pd.options.display.max_columns = 6
senators_df.head()

#### 16.4.4.3. Konfigurowanie klienta MongoClient
from pymongo import MongoClient
atlas_client = MongoClient(keys.mongo_connection_string)
db = atlas_client.senators

#### 16.4.4.4. Konfigurowanie strumienia tweetw
from tweetlistener import TweetListener
tweet_limit = 10000
twitter_stream = tweepy.Stream(api.auth,
          TweetListener(api, db, tweet_limit))

#### 16.4.4.5. Uruchomienie strumienia tweetw
twitter_stream.filter(track=senators_df.Konto_Twittera.tolist(),
    follow=senators_df.ID_Twittera.tolist())

#### 16.4.4.7. Zliczanie tweetw dla kadego senatora
db.tweets.create_index([('$**', 'text')])
tweet_counts = []
for senator in senators_df.Konto_Twittera:
    tweet_counts.append(db.tweets.count_documents(
           {"$text": {"$search": senator}}))

#### 16.4.4.8. Statystyka tweetw dla kadego senatora
tweet_counts_df = senators_df.assign(Tweety=tweet_counts)
tweet_counts_df.sort_values(by='Tweety',
                    ascending=False).head(10)

#### 16.4.4.9. Geokodowanie stanw USA
from geopy import OpenMapQuest
import time
from state_codes import state_codes
geo = OpenMapQuest(api_key=keys.mapquest_key)
states = tweet_counts_df.Stan.unique()
states.sort()

locations = []
for state in states:
    processed = False
    delay = .1

    while not processed:
        try:
            state_name = state_codes[state]
            locations.append(
                geo.geocode(state_name + ', USA'))
            print(state_name, locations[-1])
            processed = True
        except: # przeterminowanie, odczekaj chwil i sprbuj ponownie
            print('OpenMapQuest:', state_name,
                '- upyn limit czasu oczekiwania.')
            time.sleep(delay)
            delay += .3

#### 16.4.4.10. Grupowanie tweetw dla poszczeglnych stanw
tweets_counts_by_state = tweet_counts_df.groupby(
                      'Stan', as_index=False).sum()

tweets_counts_by_state.head()

#### 16.4.4.11. Tworzenie mapy
import folium
usmap = folium.Map(location=[39.8283, -98.5795],
                   zoom_start=4, detect_retina=True,
                   tiles='Stamen Toner')
                       
#### 16.4.4.12. Kolorowanie mapy w technologii Choropleth
# https://raw.githubusercontent.com/python-visualization/folium/master/examples/data/us-states.json
# http://geojson.org/ .
# --------------------------------
choropleth = folium.Choropleth(
    geo_data='us-states.json',
    name='choropleth',
    data=tweets_counts_by_state,
    columns=['Stan', 'Tweety'],
    key_on='feature.id',
    fill_color='YlOrRd',
    fill_opacity=0.7,
    line_opacity=0.2,
    legend_name='Tweety wg stanw'
   ).add_to(usmap)    


layer = folium.LayerControl().add_to(usmap)
# --------------------------------
# http://python-visualization.github.io/folium/modules.html#folium.features.Choropleth

#### 16.4.4.13. Tworzenie markerw dla poszczeglnych stanw
sorted_df = tweet_counts_df.sort_values(
               by='Tweety', ascending=False)

sorted_df.groupby('Stan')

for index, (name, group) in enumerate(sorted_df.groupby('Stan')):
    strings = [state_codes[name]] # kompozycja tekstu do wywietlenia

    for s in group.itertuples():
        strings.append(
            f'{s.Nazwisko} ({s.Partia}); Tweetw: {s.Tweety}')

    text = '<br>'.join(strings)
    marker = folium.Marker(
        (locations[index].latitude, locations[index].longitude),
        popup=text)
    marker.add_to(usmap)

#### 16.4.4.14. Wywietlanie mapy
usmap.save('SenatorsTweets.html')

##16.5. Hadoop
#### 16.5.1.2. Ekosystem Hadoop
# https://ambari.apache.org/       
# https://drill.apache.org/        
# https://flume.apache.org/        
# https://hbase.apache.org/        
# https://hive.apache.org/          
# https://impala.apache.org        
# https://kafka.apache.org/        
# https://pig.apache.org/          
# https://storm.apache.org/        
# https://zookeeper.apache.org/    

### 16.5.3. Tworzenie klastera Hadoopa za pomoc Microsoft Azure HDInsight
# https://azure.microsoft.com/en-us/free/free-account-faq/
# https://azure.microsoft.com/en-us/free
# https://docs.microsoft.com/en-us/azure/azure-resource-manager/resource-group-portal

#### 16.5.3.1. Tworzenie klastera
# https://docs.microsoft.com/en-us/azure/hdinsight/hadoop/apache-hadoop-linux-create-cluster-get-started-portal
# https://azure.microsoft.com/en-us/resources/knowledge-center/

#### 16.5.5.2. Kopiowanie pliku RomeoAndJuliet.txt na klaster
hadoop fs -copyFromLocal RomeoAndJuliet.txt /example/data/RomeoAndJuliet.txt

### 16.5.6. Uruchomienie zadania MapReduce
yarn jar /usr/hdp/current/hadoop-mapreduce-client/hadoop-streaming.jar
    -D mapred.output.key.comparator.class=
       org.apache.hadoop.mapred.lib.KeyFieldBasedComparator
    -D mapred.text.key.comparator.options=-n
    -files length_mapper.py,length_reducer.py
    -mapper length_mapper.py
    -reducer length_reducer.py
    -input /example/data/RomeoAndJuliet.txt
    -output /example/wordlengthsoutput

#### 16.5.6.1. Ogldanie statystyki dugoci sw
hdfs dfs -text /example/wordlengthsoutput/part-00000

#### 16.5.6.2. Usunicie niepotrzebnego ju klastera
# https://docs.microsoft.com/en-us/azure/azure-resource-manager/resource-group-portal

## 16.6. Spark
#### 16.6.2.2. Instalowanie Dockera
# https://www.docker.com/products/docker-desktop
# https://docs.docker.com/machine/drivers/virtualbox/
# https://docs.docker.com/install/overview/
# https://docs.docker.com/get-started/

#### 16.6.2.3. Kontenery Jupytera
# https://jupyter-docker-stacks.readthedocs.io/en/latest/index.html

#### 16.6.2.4. Uruchamianie kontenera Jupytera
docker run -p 8888:8888 -p 4040:4040 -it --user root \
   -v <pena cieka do folderu>:/home/jovyan/work \
   jupyter/pyspark-notebook:14fdfbf9cfc1 start.sh jupyter lab

jupyter/pyspark-notebook:14fdfbf9cfc1

#### 16.6.2.5. Otwieranie JupyterLaba w przegldarce WWW
# http://(bb00eb337630 or 127.0.0.1):8888/?token=
#     9570295e90ee94ecef75568b95545b7910a8f5502e6f5680

# http://localhost:8888/lab

#### 16.6.2.6. Dostp do dockerowego kontenera z jego wiersza polece
# docker ps
# conda install -c conda-forge nltk textblob

#### 16.6.2.7. Zatrzymywanie i restartowanie dockerowego kontenera
docker stop <nazwa kontenera>
docker restart <nazwa kontenera>
# https://kitematic.com/
# https://docs.docker.com/kitematic/userguide/

#### 16.6.3.1. adowanie listy sw nieznaczcych NLTK
import nltk
nltk.download('stopwords')
@ --------------------------------
from nltk.corpus import stopwords
stop_words = stopwords.words('english')

####16.6.3.2. Konfigurowanie kontekstu
from pyspark import SparkConf
configuration = SparkConf().setAppName('RomeoAndJulietCounter')\
.setMaster('local[*]')
from pyspark import SparkContext
sc = SparkContext(conf=configuration)

#### 16.6.3.3. Odczytywanie pliku tekstowego i mapowanie jego treci na listy sw
from textblob.utils import strip_punc
tokenized = sc.textFile('RomeoAndJuliet.txt')\
             .map(lambda line: strip_punc(line, all=True).lower())\
             .flatMap(lambda line: line.split())

#### 16.6.3.4. Usuwanie sw nieznaczcych
filtered = tokenized.filter(lambda word: word not in stop_words)

#### 16.6.3.5. Zliczanie pozostaych sw
from operator import add
word_counts = filtered.map(lambda word: (word, 1)).reduceByKey(add)

#### 16.6.3.6. Uwzgldnianie sw, ktre wystpiy co najmniej 60 razy
filtered_counts = word_counts.filter(lambda item: item[1] >= 60)

#### 16.6.3.7. Sortowanie sw i wypisywanie wynikw
from operator import itemgetter
sorted_items = sorted(filtered_counts.collect(),
                           key=itemgetter(1), reverse=True)

max_len = max([len(word) for word, count in sorted_items])
for word, count in sorted_items:
    print(f'{word:>{max_len}}: {count}')

#### 16.6.4.1. Tworzenie klastera Apache Spark przy uyciu portalu Azure
# https://docs.microsoft.com/en-us/azure/hdinsight/spark/apache-spark-jupyter-spark-sql-use-portal

#### 16.6.4.2. Instalowanie bibliotek w klasterze
/usr/bin/anaconda/envs/py35/bin/conda list
# http://deitel.com/bookresources/IntroToPython/install_libraries.sh
#### 16.6.4.3. Kopiowanie pliku RomeoAndJuliet.txt do klastera HDInsight
scp RomeoAndJuliet.txt sshuser@<nazwa klastera>-ssh.azurehdinsight.net:
ssh sshuser@<nazwa klastera>-ssh.azurehdinsight.net
hadoop fs -copyFromLocal RomeoAndJuliet.txt
    /example/data/RomeoAndJuliet.txt

#### 16.6.4.4. Notatnik Jupytera w HDInsight
# https://jupyter-notebook.readthedocs.io/en/stable/examples/Notebook/Notebook%20Basics.html

#### 16.6.4.6. Przystosowanie notatnika do Azure
nltk.download('stopwords')
nltk.download('stopwords', download_dir='.')
nltk.data.path.append('.')
# ---------------------------------
wasb:///example/data/RomeoAndJuliet.txt
print('{:>{width}}: {}'.format(word, count, width=max_len))
# ---------------------------------
# https://docs.microsoft.com/en-us/azure/azure-resource-manager/resource-group-portal

## 16.7. Strumieniowanie Sparka: zliczanie hashtagw przy uyciu pyspark-notebook
### 16.7.1. Uruchamianie kontenera i instalowanie biblioteki Tweepy
# pip install tweepy

### 16.7.3. Uruchamianie skryptu w kontenerze
# http://localhost:8888/lab
cd work/SparkHashtagSummarizer
ipython starttweetstream.py 10000 football

#### 16.7.3.1. Importowanie moduw
#### 16.7.4.1. Importowanie bibliotek
from pyspark import SparkContext
from pyspark.streaming import StreamingContext
from pyspark.sql import Row, SparkSession
from IPython import display
import matplotlib.pyplot as plt
import seaborn as sns
%matplotlib inline
# --------------------------------
# https://ipython.readthedocs.io/en/stable/interactive/magics.html

####16.7.4.2. Sesja  obiekt SparkSession
def getSparkSessionInstance(sparkConf):

  """Spark Streaming Programming Guide rekomenduje
     prezentowan tu metod uzyskiwania referencji 
     do istniejcego obiektu SparkSession lub tworzenie
     nowego obiektu tej klasy, gdy jeszcze takowy nie istnieje
     (obiekt ten jest singletonem)."""


   if ("sparkSessionSingletonInstance" not in globals()):
       globals()["sparkSessionSingletonInstance"] = SparkSession \
           .builder \
           .config(conf=sparkConf) \
           .getOrCreate()
   return globals()["sparkSessionSingletonInstance"]

#### 16.7.4.3. Wykres supkowy na podstawie ramki DataFrame
def display_barplot(spark_df, x, y, time, scale=2.0, size=(16, 9)):
    """Wywietlenie zawartoci ramki Spark DataFrame 
       w postaci wykresu supkowego"""
    df = spark_df.toPandas()
    
    # usunicie poprzedniego wykresu w momencie, 
    # gdy nowy wykres stanie si gotowy do wywietlenia
    display.clear_output(wait=True)
    print(f'CZAS: {time}')
    
    # utworzenie i skonfigurowanie obiektu Figure zawierajcego 
    # nowy wykres
    plt.figure(figsize=size)
    sns.set(font_scale=scale)
    barplot = sns.barplot(data=df, x=x, y=y
                          palette=sns.color_palette('cool', 20))

    # obrt etykiet na osi x o kt prosty dla lepszej czytelnoci
    for item in barplot.get_xticklabels():
        item.set_rotation(90)
        
    plt.tight_layout()
    plt.show()
    
#### 16.7.4.4. Czowka 20 hashtagw
def count_tags(time, rdd):
    """Zliczanie hashtagw i wywietlanie 20 najliczniejszych
       w kolejnoci malejcej"""
    try:
        # dostp do obiektuSparkSession
        spark = getSparkSessionInstance(rdd.context.getConf())         

# mapowanie hashtagw postaci acuch-licznik na wiersze Row
rows = rdd.map(
    lambda tag: Row(hashtag=tag[0], total=tag[1]))

# utworzenie ramki DataFrame na podstawie kolekcji wierszy
hashtags_df = spark.createDataFrame(rows)

# utworzenie tymczasowego widoku ramki, do uytku przez Spark SQL
hashtags_df.createOrReplaceTempView('hashtags')

# wybr 20 najliczniejszych hashtagw posortowanych malejco
top20_df = spark.sql(
    """select hashtag, total
       from hashtags
       order by total, hashtag desc
       limit 20""")

    display_barplot(top20_df, x='hashtagi', y='licznik', time=time)
except Exception as e:
    print(f'Bd: {e}')

#### 16.7.4.5. Kontekst  obiekt SparkContext
sc = SparkContext()

#### 16.7.4.6. Strumieniowanie  obiekt StreamingContext
ssc = StreamingContext(sc, 10)
# --------------------------------
# https://spark.apache.org/docs/latest/streaming-programming-guide.html#performance-tuning

#### 16.7.4.7. Zarzdzanie stanem  punkty kontrolne
ssc.checkpoint('hashtagsummarizer_checkpoint')
# --------------------------------
# https://spark.apache.org/docs/latest/streaming-programming-guide.html#checkpointing

#### 16.7.4.8. Poczenie ze strumieniem przez gniazdo
stream = ssc.socketTextStream('localhost', 9876)

#### 16.7.4.9. Tokenizacja acucha hashtagw
tokenized = stream.flatMap(lambda line: line.split())

#### 16.7.4.10. Mapowanie hashtagw w krotki hashtag-licznik 
mapped = tokenized.map(lambda hashtag: (hashtag, 1))

#### 16.7.4.11. Sumowanie licznikw
hashtag_counts = tokenized.updateStateByKey(
    lambda counts, prior_total: sum(counts) + (prior_total or 0))

#### 16.7.4.12. Metoda wywoywana dla kadego RDD
hashtag_counts.foreachRDD(count_tags)

#### 16.7.4.13. Rozpoczcie strumieniowania
ssc.start() # rozpoczcie strumieniowania Sparka

## 16.8. Internet rzeczy (IoT) i dashboardy
##### 16.8.4.1. Logowanie do usugi Freeboard.io
# https://freeboard.io/signup

#### 16.8.4.3. Dodawanie rde danych
# https://www.pubnub.com/developers/realtime-data-streams/sensor-network/

#### 16.8.5.1. Instalowanie biblioteki Dweepy
# pip install dweepy
# --------------------------------
# https://github.com/paddycarey/dweepy

#### 16.8.5.2. Uruchomienie skryptu symulatora
ipython simulator.py 1000 1

#### 16.8.5.3. Wysyanie dweetw
# https://dweet.io/follow/<nazwa rzeczy>

#### 16.8.7. Subskrybent usugi PubNub w Pythonie
# https://www.pubnub.com/docs/python/pubnub-python-sdk.
# --------------------------------
# pip install "pubnub>=4.1.2"
