# sentimentlisener.py
#-*- coding: 1250 -*-
"""Script that searches for tweets that match a search string
and tallies the number of positive, neutral and negative tweets."""

import keys
import preprocessor as p 
import sys
from textblob import TextBlob
import tweepy

class SentimentListener(tweepy.StreamListener):
    """Obsuguje strumie nadchodzcych tweetw."""

    def __init__(self, api, sentiment_dict, topic, limit=10):
        """Konfigurowanie obiektu-obserwatora"""
        self.sentiment_dict = sentiment_dict
        self.tweet_count = 0
        self.topic = topic
        self.TWEET_LIMIT = limit

        # konfigurowanie preprocesora w celu usuwania
        # URL-i i zarezerwowanych sw Twittera.
        p.set_options(p.OPT.URL, p.OPT.RESERVED) 
        
        super().__init__(api)  # wywoanie __init__ z nadklasy
  
    def on_status(self, status):
        """Metoda wywoywana, gdy Twitter dostarcza nowy tweet"""
        
        # pobranie treci tweetu
        try:  
            tweet_text = status.extended_tweet.full_text
        except: 
            tweet_text = status.text

        # ignorowanie retweetw
        if tweet_text.startswith('RT'):
            return

        tweet_text = p.clean(tweet_text)  # normalizacja tweetu
        
        # ignorowanie tweetw, w ktrych temat nie pojawia si w treci
        if self.topic.lower() not in tweet_text.lower():
            return

        # aktualizacja sownika sentymentw pod ktem 
        # polaryzacji biecego tweetu
        blob = TextBlob(tweet_text)
        if blob.sentiment.polarity > 0:
            sentiment = '+'
            self.sentiment_dict['positive'] += 1 
        elif blob.sentiment.polarity == 0:
            sentiment = ' '
            self.sentiment_dict['neutral'] += 1 
        else:
            sentiment = '-'
            self.sentiment_dict['negative'] += 1 
            
        # wywietlenie tweetu
        print(f'{sentiment} {status.user.screen_name}: {tweet_text}\n')
        
        self.tweet_count += 1  # licznik przetworzonych tweetw

        # gdy przekroczony zostanie limit przetworzonych tweetw
        # (TWEET_LIMIT) zwr False w celu przerwania strumieniowania,
        # w przeciwnym razie zwr True.
        return self.tweet_count <= self.TWEET_LIMIT

def main():

    # weryfikacja i pobranie parametrw uruchamiania
    if len(sys.argv) < 3:
        print('Wymagane parametry uruchamiania:')
        print('  1. Badany temat')
        print('  2. Limit uwzgldnianych tweetw')
        sys.exit

    search_key = sys.argv[1]  # fraza-kryterium wyszukiwania
    limit = int(sys.argv[2])  # limit analizowanych tweetw


    # konfigurowanie obiektu uwierzytelniania
    auth = tweepy.OAuthHandler(keys.consumer_key, keys.consumer_secret)
    auth.set_access_token(keys.access_token, keys.access_token_secret)

    # utworzenie obiektu api
    api = tweepy.API(auth, wait_on_rate_limit=True, 
                     wait_on_rate_limit_notify=True)
                 
    # utworzenie obiektu-obserwatora
    sentiment_dict = {'positive': 0, 'neutral': 0, 'negative': 0}
    sentiment_listener = SentimentListener(api, 
        sentiment_dict, search_key, limit)

    # skonfigurowanie strumienia
    stream = tweepy.Stream(auth=api.auth, listener=sentiment_listener)

    # rozpoczcie pobierania tweetw, 
    # wycznie w jzyku angielskim
    # zawierajcych dan fraz
    stream.filter(track=[search_key], languages=['en'], is_async=False)  

    print(f'Sentyment tweetw zawierajcych "{search_key}"')
    print('Pozytywne:', sentiment_dict['positive'])
    print(' Obojtne:', sentiment_dict['neutral'])
    print('Negatywne:', sentiment_dict['negative'])

# wywoanie metody "main", gdy niniejszy plik 
# uruchomiony zostanie jako skrypt 
if __name__ == '__main__':
    main()

##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
