
# coding: utf-8

# *Python. Uczenie maszynowe. Wydanie drugie*, [Sebastian Raschka](https://sebastianraschka.com), Packt Publishing Ltd. 2017
# 
# Repozytorium kodu: https://github.com/rasbt/python-machine-learning-book-2nd-edition
# 
# Licencja: [MIT License](https://github.com/rasbt/python-machine-learning-book-2nd-edition/blob/master/LICENSE.txt)

# # Python. Uczenie maszynowe - kod źródłowy

# # Rozdział 10. Przewidywanie ciągłych zmiennych docelowych za pomocą analizy regresywnej

# Zwróć uwagę, że rozszerzenie zawierające nieobowiązkowy znak wodny stanowi niewielki plugin notatnika IPython / Jupyter, który zaprojektowałem w celu powielania kodu źródłowego. Wystarczy pominąć poniższe wiersze kodu:

# In[1]:


get_ipython().run_line_magic('load_ext', 'watermark')
get_ipython().run_line_magic('watermark', '-a "Sebastian Raschka" -u -d -v -p numpy,pandas,matplotlib,sklearn,seaborn')


# *Korzystanie z rozszerzenia `watermark` nie jest obowiązkowe. Możesz je zainstalować za pomocą polecenia "`pip install watermark`". Więcej informacji na jego temat znajdziesz pod adresem: https://github.com/rasbt/watermark.*

# Bibliotekę wizualizacji seaborn, bazującą na bibliotece matplotlib, zainstalujemy w następujący sposób: 
# 
#     conda install seaborn
# 
# albo 
# 
#     pip install seaborn

# <br>
# <br>

# ### Spis treści

# - [Wprowadzenie do regresji liniowej](#Wprowadzenie-do-regresji-liniowej)
#   - [Prosta regresja liniowa](#Prosta-regresja-liniowa)
#   - [Wielowymiarowa regresja liniowa](#Wielowymiarowa-regresja-liniowa)
# - [Zestaw danych Housing](#Zestaw-danych-Housing)
#   - [Wczytywanie zestawu danych Housing do obiektu DataFrame](#Wczytywanie-zestawu-danych-Housing-do-obiektu-DataFrame)
#   - [Wizualizowanie ważnych elementów zestawu danych](#Wizualizowanie-ważnych-elementów-zestawu-danych)
# - [Implementacja modelu regresji liniowej wykorzystującego zwykłą metodę najmniejszych kwadratów](#Implementacja-modelu-regresji-liniowej-wykorzystującego-zwykłą-metodę-najmniejszych-kwadratów)
#   - [Określanie parametrów regresywnych za pomocą metody gradientu prostego](#Określanie-parametrów-regresywnych-za-pomocą-metody-gradientu-prostego)
#   - [Szacowanie współczynnika modelu regresji za pomocą biblioteki scikit-learn](#Szacowanie-współczynnika-modelu-regresji-za-pomocą-biblioteki-scikit-learn)
# - [Uczenie odpornego modelu regresywnego za pomocą algorytmu RANSAC](#Uczenie-odpornego-modelu-regresywnego-za-pomocą-algorytmu-RANSAC)
# - [Ocenianie skuteczności modeli regresji liniowej](#Ocenianie-skuteczności-modeli-regresji-liniowej)
# - [Stosowanie regularyzowanych metod regresji](#Stosowanie-regularyzowanych-metod-regresji)
# - [Przekształcanie modelu regresji liniowej w krzywą — regresja wielomianowa](#Przekształcanie-modelu-regresji-liniowej-w-krzywą-—-regresja-wielomianowa)
#   - [Modelowanie nieliniowych zależności w zestawie danych Housing](#Modelowanie-nieliniowych-zależności-w-zestawie-danych-Housing)
#   - [Analiza nieliniowych relacji za pomocą algorytmu losowego lasu](#Analiza-nieliniowych-relacji-za-pomocą-algorytmu-losowego-lasu)
#     - [Regresja przy użyciu drzewa decyzyjnego](#Regresja-przy-użyciu-drzewa-decyzyjnego)
#     - [Regresja przy użyciu losowego lasu](#Regresja-przy-użyciu-losowego-lasu)
# - [Podsumowanie](#Podsumowanie)

# <br>
# <br>

# In[1]:


from IPython.display import Image
get_ipython().run_line_magic('matplotlib', 'inline')


# # Wprowadzenie do regresji liniowej

# ## Prosta regresja liniowa

# In[2]:


Image(filename='rysunki/10_01.png', width=500) 


# ## Wielowymiarowa regresja liniowa

# In[3]:


Image(filename='rysunki/10_15.png', width=500) 


# <br>
# <br>

# # Zestaw danych Housing

# ## Wczytywanie zestawu danych Housing do obiektu DataFrame

# Opis, dostępny uprzednio pod adresem [https://archive.ics.uci.edu/ml/datasets/Housing](https://archive.ics.uci.edu/ml/datasets/Housing)
# 
# Atrybuty:
#     
# <pre>
# 1. CRIM      współczynnik przestępczości per capita na każde miasteczko
# 2. ZN        odsetek działek przekraczających 25 000 stóp kwadratowych (≈ 2533 metrów kwadratowych).
# 3. INDUS     odsetek terenów przeznaczonych na przemysł niedetaliczny na każde miasteczko
# 4. CHAS      zmienna zerojedynkowa określająca rzekę Charles (przyjmuje wartość 1, gdy na danym terenie znajduje się koryto      
#              rzeki)
# 5. NOX       stężenie tlenków azotu (w częściach na 10 milionów)
# 6. RM        średnia liczba pomieszczeń na dom
# 7. AGE       odsetek zamieszkałych budynków wybudowanych przed 1940 rokiem
# 8. DIS       ważona odległość do pięciu bostońskich urzędów pracy
# 9. RAD       wskaźnik dostępności do głównych arterii komunikacyjnych
# 10. TAX      pełna wartość podatku od nieruchomości na każde 10 000 dolarów
# 11. PTRATIO  stosunek liczby uczniów do nauczycieli na każde miasteczko
# 12. B        parametr wyliczany ze wzoru 1000(Bk - 0.63)^2, gdzie Bk oznacza odsetek osób pochodzenia afroamerykańskiego 
#              zamieszkujących dane miasteczko 
# 13. LSTAT    odsetek ubogiej części społeczeństwa
# 14. MEDV     mediana wartości zamieszkanych domów wyrażona w tysiącach dolarów
# </pre>

# In[4]:


import pandas as pd

df = pd.read_csv('https://raw.githubusercontent.com/rasbt/'
                 'python-machine-learning-book-2nd-edition'
                 '/master/code/ch10/housing.data.txt',
                 header=None,
                 sep='\s+')

df.columns = ['CRIM', 'ZN', 'INDUS', 'CHAS', 
              'NOX', 'RM', 'AGE', 'DIS', 'RAD', 
              'TAX', 'PTRATIO', 'B', 'LSTAT', 'MEDV']
df.head()


# <hr>
# 
# ### Uwaga
# 
# Uwaga
# Kopię zestawu danych Housing (a także wszystkich pozostałych ze-stawów danych wykorzystywanych w tej książce) znajdziesz w przy-kładowym kodzie dołączonym do niniejszej książki, dzięki czemu możesz z niego korzystać będąc odłączonym od internetu lub jeśli adres https://archive.ics.uci.edu/ml/machine-learning-databases/housing/housing.data będzie w danym momencie niedostępny. Na przykład, aby wczytać zestaw danych Housing z katalogu lokalnego, wystarczy zastąpić wiersze
# 
# df = pd.read_csv('https://archive.ics.uci.edu/ml/'
#                  'machine-learning-databases'
#                  '/housing/housing.data',
#                  sep='\s+')
# 
# w powyższym przykładzie wierszem
# 
# df = pd.read_csv('./housing.data',
#                  sep='\s+')

# <br>
# <br>

# ## Wizualizowanie ważnych elementów zestawu danych

# In[5]:


import matplotlib.pyplot as plt
import seaborn as sns


# In[6]:


cols = ['LSTAT', 'INDUS', 'NOX', 'RM', 'MEDV']

sns.pairplot(df[cols], size=2.5)
plt.tight_layout()
# plt.savefig('rysunki/10_03.png', dpi=300)
plt.show()


# In[7]:


import numpy as np


cm = np.corrcoef(df[cols].values.T)
#sns.set(font_scale=1.5)
hm = sns.heatmap(cm,
                 cbar=True,
                 annot=True,
                 square=True,
                 fmt='.2f',
                 annot_kws={'size': 15},
                 yticklabels=cols,
                 xticklabels=cols)

plt.tight_layout()
# plt.savefig('rysunki/10_04.png', dpi=300)
plt.show()


# <br>
# <br>

# # Implementacja modelu regresji liniowej wykorzystującego zwykłą metodę najmniejszych kwadratów

# ...

# ## Określanie parametrów regresywnych za pomocą metody gradientu prostego

# In[8]:


class LinearRegressionGD(object):

    def __init__(self, eta=0.001, n_iter=20):
        self.eta = eta
        self.n_iter = n_iter

    def fit(self, X, y):
        self.w_ = np.zeros(1 + X.shape[1])
        self.cost_ = []

        for i in range(self.n_iter):
            output = self.net_input(X)
            errors = (y - output)
            self.w_[1:] += self.eta * X.T.dot(errors)
            self.w_[0] += self.eta * errors.sum()
            cost = (errors**2).sum() / 2.0
            self.cost_.append(cost)
        return self

    def net_input(self, X):
        return np.dot(X, self.w_[1:]) + self.w_[0]

    def predict(self, X):
        return self.net_input(X)


# In[9]:


X = df[['RM']].values
y = df['MEDV'].values


# In[10]:


from sklearn.preprocessing import StandardScaler


sc_x = StandardScaler()
sc_y = StandardScaler()
X_std = sc_x.fit_transform(X)
y_std = sc_y.fit_transform(y[:, np.newaxis]).flatten()


# In[11]:


lr = LinearRegressionGD()
lr.fit(X_std, y_std)


# In[12]:


plt.plot(range(1, lr.n_iter+1), lr.cost_)
plt.ylabel('Suma kwadratów błędów')
plt.xlabel('Epoka')
#plt.tight_layout()
#plt.savefig('rysunki/10_05.png', dpi=300)
plt.show()


# In[13]:


def lin_regplot(X, y, model):
    plt.scatter(X, y, c='steelblue', edgecolor='white', s=70)
    plt.plot(X, model.predict(X), color='black', lw=2)    
    return 


# In[14]:


lin_regplot(X_std, y_std, lr)
plt.xlabel('Uśredniona liczba pomieszczeń [RM] (standaryzowana)')
plt.ylabel('Cena w tysiącach dolarów [MEDV] (standaryzowana)')

#plt.savefig('rysunki/10_06.png', dpi=300)
plt.show()


# In[15]:


print('Nachylenie: %.3f' % lr.w_[1])
print('Punkt przecięcia: %.3f' % lr.w_[0])


# In[16]:


num_rooms_std = sc_x.transform(np.array([[5.0]]))
price_std = lr.predict(num_rooms_std)
print("Cena w tysiącach dolarów: %.3f" % sc_y.inverse_transform(price_std))


# <br>
# <br>

# ## Szacowanie współczynnika modelu regresji za pomocą biblioteki scikit-learn

# In[17]:


from sklearn.linear_model import LinearRegression


# In[18]:


slr = LinearRegression()
slr.fit(X, y)
y_pred = slr.predict(X)
print('Nachylenie: %.3f' % slr.coef_[0])
print('Punkt przecięcia: %.3f' % slr.intercept_)


# In[19]:


lin_regplot(X, y, slr)
plt.xlabel('Uśredniona liczba pomieszczeń [RM]')
plt.ylabel('Cena w tysiącach dolarów [MEDV]')

#plt.savefig('rysunki/10_07.png', dpi=300)
plt.show()


# **Równania normalne** wersja alternatywna:

# In[20]:


# Dodaje wektor kolumnowy zawierający "jedynki"
Xb = np.hstack((np.ones((X.shape[0], 1)), X))
w = np.zeros(X.shape[1])
z = np.linalg.inv(np.dot(Xb.T, Xb))
w = np.dot(z, np.dot(Xb.T, y))

print('Nachylenie: %.3f' % w[1])
print('Punkt przecięcia: %.3f' % w[0])


# <br>
# <br>

# # Uczenie odpornego modelu regresywnego za pomocą algorytmu RANSAC

# In[21]:


from sklearn.linear_model import RANSACRegressor

ransac = RANSACRegressor(LinearRegression(), 
                         max_trials=100, 
                         min_samples=50, 
                         loss='absolute_loss', 
                         residual_threshold=5.0, 
                         random_state=0)


ransac.fit(X, y)

inlier_mask = ransac.inlier_mask_
outlier_mask = np.logical_not(inlier_mask)

line_X = np.arange(3, 10, 1)
line_y_ransac = ransac.predict(line_X[:, np.newaxis])
plt.scatter(X[inlier_mask], y[inlier_mask],
            c='steelblue', edgecolor='white', 
            marker='o', label='Pr. nieodstające')
plt.scatter(X[outlier_mask], y[outlier_mask],
            c='limegreen', edgecolor='white', 
            marker='s', label='Pr. odstające')
plt.plot(line_X, line_y_ransac, color='black', lw=2)   
plt.xlabel('Uśredniona liczba pomieszczeń [RM]')
plt.ylabel('Cena w tysiącach dolarów [MEDV]')
plt.legend(loc='upper left')

#plt.savefig('rysunki/10_08.png', dpi=300)
plt.show()


# In[22]:


print('Nachylenie: %.3f' % ransac.estimator_.coef_[0])
print('Punkt przecięcia: %.3f' % ransac.estimator_.intercept_)


# <br>
# <br>

# # Ocenianie skuteczności modeli regresji liniowej

# In[23]:


from sklearn.model_selection import train_test_split

X = df.iloc[:, :-1].values
y = df['MEDV'].values

X_train, X_test, y_train, y_test = train_test_split(
    X, y, test_size=0.3, random_state=0)


# In[24]:


slr = LinearRegression()

slr.fit(X_train, y_train)
y_train_pred = slr.predict(X_train)
y_test_pred = slr.predict(X_test)


# In[25]:


import numpy as np
import scipy as sp

ary = np.array(range(100000))


# In[26]:


get_ipython().run_line_magic('timeit', 'np.linalg.norm(ary)')


# In[27]:


get_ipython().run_line_magic('timeit', 'sp.linalg.norm(ary)')


# In[28]:


get_ipython().run_line_magic('timeit', 'np.sqrt(np.sum(ary**2))')


# In[29]:


plt.scatter(y_train_pred,  y_train_pred - y_train,
            c='steelblue', marker='o', edgecolor='white',
            label='Dane uczące')
plt.scatter(y_test_pred,  y_test_pred - y_test,
            c='limegreen', marker='s', edgecolor='white',
            label='Dane testowe')
plt.xlabel('Przewidywane wartości')
plt.ylabel('Wartości resztowe')
plt.legend(loc='upper left')
plt.hlines(y=0, xmin=-10, xmax=50, color='black', lw=2)
plt.xlim([-10, 50])
plt.tight_layout()

#plt.savefig('rysunki/10_09.png', dpi=300)
plt.show()


# In[30]:


from sklearn.metrics import r2_score
from sklearn.metrics import mean_squared_error

print('MSE na próbkach uczących: %.3f, testowych: %.3f' % (
        mean_squared_error(y_train, y_train_pred),
        mean_squared_error(y_test, y_test_pred)))
print('Współczynnik R^2 dla danych uczących: %.3f, testowych: %.3f' % (
        r2_score(y_train, y_train_pred),
        r2_score(y_test, y_test_pred)))


# <br>
# <br>

# # Stosowanie regularyzowanych metod regresji

# In[31]:


from sklearn.linear_model import Lasso

lasso = Lasso(alpha=0.1)
lasso.fit(X_train, y_train)
y_train_pred = lasso.predict(X_train)
y_test_pred = lasso.predict(X_test)
print(lasso.coef_)


# In[32]:


print('MSE na próbkach uczących: %.3f, testowych: %.3f' % (
        mean_squared_error(y_train, y_train_pred),
        mean_squared_error(y_test, y_test_pred)))
print('Współczynnik R^2 dla danych uczących: %.3f, testowych: %.3f' % (
        r2_score(y_train, y_train_pred),
        r2_score(y_test, y_test_pred)))


# Regresja grzbietowa:

# In[33]:


from sklearn.linear_model import Ridge
ridge = Ridge(alpha=1.0)


# Regresja typu LASSO:

# In[34]:


from sklearn.linear_model import Lasso
lasso = Lasso(alpha=1.0)


# Regresja typu siatka elastyczna:

# In[35]:


from sklearn.linear_model import ElasticNet
elanet = ElasticNet(alpha=1.0, l1_ratio=0.5)


# <br>
# <br>

# # Przekształcanie modelu regresji liniowej w krzywą — regresja wielomianowa

# In[36]:


X = np.array([258.0, 270.0, 294.0, 
              320.0, 342.0, 368.0, 
              396.0, 446.0, 480.0, 586.0])\
             [:, np.newaxis]

y = np.array([236.4, 234.4, 252.8, 
              298.6, 314.2, 342.2, 
              360.8, 368.0, 391.2,
              390.8])


# In[37]:


from sklearn.preprocessing import PolynomialFeatures

lr = LinearRegression()
pr = LinearRegression()
quadratic = PolynomialFeatures(degree=2)
X_quad = quadratic.fit_transform(X)


# In[38]:


# dopasowuje cechy liniowe
lr.fit(X, y)
X_fit = np.arange(250, 600, 10)[:, np.newaxis]
y_lin_fit = lr.predict(X_fit)

# dopasowuje cechy kwadratowe
pr.fit(X_quad, y)
y_quad_fit = pr.predict(quadratic.fit_transform(X_fit))

# rysuje wykres wynikowy
plt.scatter(X, y, label='Punkty uczące')
plt.plot(X_fit, y_lin_fit, label='Dopasowanie liniowe', linestyle='--')
plt.plot(X_fit, y_quad_fit, label='Dopasowanie kwadratowe')
plt.legend(loc='upper left')

plt.tight_layout()
#plt.savefig('rysunki/10_10.png', dpi=300)
plt.show()


# In[39]:


y_lin_pred = lr.predict(X)
y_quad_pred = pr.predict(X_quad)


# In[40]:


print('Błąd MSE dla uczenia liniowego: %.3f, kwadratowego: %.3f' % (
        mean_squared_error(y, y_lin_pred),
        mean_squared_error(y, y_quad_pred)))
print('Parametr R^2 dla uczenia liniowego: %.3f, kwadratowego: %.3f' % (
        r2_score(y, y_lin_pred),
        r2_score(y, y_quad_pred)))


# <br>
# <br>

# ## Modelowanie nieliniowych zależności w zestawie danych Housing

# In[41]:


X = df[['LSTAT']].values
y = df['MEDV'].values

regr = LinearRegression()

# tworzy cechy kwadratowe
quadratic = PolynomialFeatures(degree=2)
cubic = PolynomialFeatures(degree=3)
X_quad = quadratic.fit_transform(X)
X_cubic = cubic.fit_transform(X)

# dopasowuje cechy
X_fit = np.arange(X.min(), X.max(), 1)[:, np.newaxis]

regr = regr.fit(X, y)
y_lin_fit = regr.predict(X_fit)
linear_r2 = r2_score(y, regr.predict(X))

regr = regr.fit(X_quad, y)
y_quad_fit = regr.predict(quadratic.fit_transform(X_fit))
quadratic_r2 = r2_score(y, regr.predict(X_quad))

regr = regr.fit(X_cubic, y)
y_cubic_fit = regr.predict(cubic.fit_transform(X_fit))
cubic_r2 = r2_score(y, regr.predict(X_cubic))


# tworzy wykres wynikowy
plt.scatter(X, y, label='Punkty uczące', color='lightgray')

plt.plot(X_fit, y_lin_fit, 
         label='Liniowe (d=1), $R^2=%.2f$' % linear_r2, 
         color='blue', 
         lw=2, 
         linestyle=':')

plt.plot(X_fit, y_quad_fit, 
         label='Kwadratowe (d=2), $R^2=%.2f$' % quadratic_r2,
         color='red', 
         lw=2,
         linestyle='-')

plt.plot(X_fit, y_cubic_fit, 
         label='Sześcienne (d=3), $R^2=%.2f$' % cubic_r2,
         color='green', 
         lw=2, 
         linestyle='--')

plt.xlabel('Odsetek uboższej części społeczeństwa  [LSTAT]')
plt.ylabel('Cena w tysiącach dolarów [MEDV]')
plt.legend(loc='upper right')

#plt.savefig('rysunki/10_11.png', dpi=300)
plt.show()


# Przekształcanie zestawu danych:

# In[42]:


X = df[['LSTAT']].values
y = df['MEDV'].values

# przekształca cechy
X_log = np.log(X)
y_sqrt = np.sqrt(y)

# dopasowuje cechy
X_fit = np.arange(X_log.min()-1, X_log.max()+1, 1)[:, np.newaxis]

regr = regr.fit(X_log, y_sqrt)
y_lin_fit = regr.predict(X_fit)
linear_r2 = r2_score(y_sqrt, regr.predict(X_log))

# tworzy wykres wynikowy
plt.scatter(X_log, y_sqrt, label='Punkty uczące', color='lightgray')

plt.plot(X_fit, y_lin_fit, 
         label='Liniowe (d=1), $R^2=%.2f$' % linear_r2, 
         color='blue', 
         lw=2)

plt.xlabel('log(odsetek uboższej części społeczeństwa  [LSTAT])')
plt.ylabel('$\sqrt{Cena \; w \; tysiącach \; dolarów \; [MEDV]}$')
plt.legend(loc='lower left')

plt.tight_layout()
#plt.savefig('rysunki/10_12.png', dpi=300)
plt.show()


# <br>
# <br>

# # Analiza nieliniowych relacji za pomocą algorytmu losowego lasu

# ...

# ## Regresja przy użyciu drzewa decyzyjnego

# In[43]:


from sklearn.tree import DecisionTreeRegressor

X = df[['LSTAT']].values
y = df['MEDV'].values

tree = DecisionTreeRegressor(max_depth=3)
tree.fit(X, y)

sort_idx = X.flatten().argsort()

lin_regplot(X[sort_idx], y[sort_idx], tree)
plt.xlabel('Odsetek uboższej części społeczeństwa [LSTAT]')
plt.ylabel('Cena w tysiącach dolarów [MEDV]')
#plt.savefig('rysunki/10_13.png', dpi=300)
plt.show()


# <br>
# <br>

# ## Regresja przy użyciu losowego lasu

# In[44]:


X = df.iloc[:, :-1].values
y = df['MEDV'].values

X_train, X_test, y_train, y_test = train_test_split(
    X, y, test_size=0.4, random_state=1)


# In[45]:


from sklearn.ensemble import RandomForestRegressor

forest = RandomForestRegressor(n_estimators=1000, 
                               criterion='mse', 
                               random_state=1, 
                               n_jobs=-1)
forest.fit(X_train, y_train)
y_train_pred = forest.predict(X_train)
y_test_pred = forest.predict(X_test)

print('MSE na danych uczących: %.3f, testowych: %.3f' % (
        mean_squared_error(y_train, y_train_pred),
        mean_squared_error(y_test, y_test_pred)))
print('Współczynnik R^2 dla danych uczących: %.3f, testowych: %.3f' % (
        r2_score(y_train, y_train_pred),
        r2_score(y_test, y_test_pred)))


# In[46]:


plt.scatter(y_train_pred,  
            y_train_pred - y_train, 
            c='steelblue',
            edgecolor='white',
            marker='o', 
            s=35,
            alpha=0.9,
            label='Dane uczące')
plt.scatter(y_test_pred,  
            y_test_pred - y_test, 
            c='limegreen',
            edgecolor='white',
            marker='s', 
            s=35,
            alpha=0.9,
            label='Dane testowe')

plt.xlabel('Przewidywane wartości')
plt.ylabel('Wartości resztowe')
plt.legend(loc='upper left')
plt.hlines(y=0, xmin=-10, xmax=50, lw=2, color='black')
plt.xlim([-10, 50])
plt.tight_layout()

#plt.savefig('rysunki/10_14.png', dpi=300)
plt.show()


# <br>
# <br>

# # Podsumowanie

# ...

# ---
# 
# Czytelnicy mogą zignorować poniższą komórkę.

# In[48]:


get_ipython().system(' python ../.convert_notebook_to_script.py --input ch10.ipynb --output ch10.py')

