
# coding: utf-8

# *Python. Uczenie maszynowe. Wydanie drugie*, [Sebastian Raschka](https://sebastianraschka.com), Packt Publishing Ltd. 2017
# 
# Repozytorium kodu: https://github.com/rasbt/python-machine-learning-book-2nd-edition
# 
# Licencja: [MIT License](https://github.com/rasbt/python-machine-learning-book-2nd-edition/blob/master/LICENSE.txt)

# # Python. Uczenie maszynowe - kod źródłowy

# # Rozdział 16. Modelowanie danych sekwencyjnych za pomocą rekurencyjnych sieci neuronowych
# 
# 

# Zwróć uwagę, że rozszerzenie zawierające nieobowiązkowy znak wodny stanowi niewielki plugin notatnika IPython / Jupyter, który zaprojektowałem w celu powielania kodu źródłowego. Wystarczy pominąć poniższe wiersze kodu:

# In[1]:


get_ipython().run_line_magic('load_ext', 'watermark')
get_ipython().run_line_magic('watermark', '-a "Sebastian Raschka & Vahid Mirjalili" -u -d -v -p numpy,scipy,pyprind,tensorflow')


# *Korzystanie z rozszerzenia `watermark` nie jest obowiązkowe. Możesz je zainstalować za pomocą polecenia "`pip install watermark`". Więcej informacji na jego temat znajdziesz pod adresem: https://github.com/rasbt/watermark.*

# - [Wprowadzenie do danych sekwencyjnych](#Wprowadzenie-do-danych-sekwencyjnych)
#   - [Modelowanie danych sekwencyjnych — kolejność ma znaczenie](#Modelowanie-danych-sekwencyjnych-—-kolejność-ma-znaczenie)
#   - [Różne kategorie modelowania sekwencji](#Różne-kategorie-modelowania-sekwencji)
# - [Sieci rekurencyjne służące do modelowania sekwencji](#Sieci-rekurencyjne-służące-do-modelowania-sekwencji)
#   - [Struktura sieci RNN i przepływ danych](#Struktura-sieci-RNN-i-przepływ-danych)
#   - [Obliczanie aktywacji w sieciach rekurencyjnych](#Obliczanie-aktywacji-w-sieciach-rekurencyjnych)
#   - [Problemy z uczeniem długofalowych oddziaływań](#Problemy-z-uczeniem-długofalowych-oddziaływań)
#   - [Jednostki LSTM](#Jednostki-LSTM)
# - [Implementowanie wielowarstwowej sieci rekurencyjnej przy użyciu biblioteki TensorFlow do modelowania sekwencji](#Implementowanie-wielowarstwowej-sieci-rekurencyjnej-przy-użyciu-biblioteki-TensorFlow-do-modelowania-sekwencji)
#   - [Analiza sentymentów na zestawie danych IMDb za pomocą wielowarstwowej sieci rekurencyjnej](#Analiza-sentymentów-na-zestawie-danych-IMDb-za-pomocą-wielowarstwowej-sieci-rekurencyjnej)
#     - [Przygotowanie danych](#Przygotowanie-danych)
#     - [Wektor właściwościowy](#Wektor-właściwościowy)
#     - [Budowanie modelu sieci rekurencyjnej](#Budowanie-modelu-sieci-rekurencyjnej)
#       - [Etap 1. Definiowanie komórek wielowarstwowej sieci RNN](#Etap-1.-Definiowanie-komórek-wielowarstwowej-sieci-RNN)
#       - [Etap 2. Definiowanie stanów początkowych dla komórek rekurencyjnych](#Etap-2.-Definiowanie-stanów-początkowych-dla-komórek-rekurencyjnych)
#       - [Etap 3. Tworzenie sieci rekurencyjnej za pomocą komórek RNN i ich stanów](#Etap-3.-Tworzenie-sieci-rekurencyjnej-za-pomocą-komórek-RNN-i-ich-stanów)
#   - [Modelowanie języka na poziomie znaków](#Modelowanie-języka-na-poziomie-znaków)
#     - [Wstępna obróbka danych](#Wstępna-obróbka-danych)
#     - [Tworzenie sieci RNN przetwarzającej znaki](#Tworzenie-sieci-RNN-przetwarzającej-znaki)
# - [Podsumowanie](#Podsumowanie)

# In[4]:


from IPython.display import Image


# In[5]:


import gzip


with gzip.open('movie_data.csv.gz') as f_in, open('filmy_dane.csv', 'wb') as f_out:
    f_out.writelines(f_in)


# # Wprowadzenie do danych sekwencyjnych

# ## Modelowanie danych sekwencyjnych — kolejność ma znaczenie

# ## Przedstawianie sekwencji

# In[3]:


Image(filename='rysunki/16_01.png', width=600) 


# ## Różne kategorie modelowania sekwencji

# In[4]:


Image(filename='rysunki/16_02.png', width=600) 


# # Sieci rekurencyjne służące do modelowania sekwencji

# ## Struktura sieci RNN i przepływ danych

# In[5]:


Image(filename='rysunki/16_03.png', width=600) 


# In[6]:


Image(filename='rysunki/16_04.png', width=600) 


# ##  Obliczanie aktywacji w sieciach rekurencyjnych

# In[7]:


Image(filename='rysunki/16_05.png', width=600) 


# In[8]:


Image(filename='rysunki/16_06.png', width=600) 


# ## Problemy z uczeniem długofalowych oddziaływań

# In[9]:


Image(filename='rysunki/16_07.png', width=600) 


# ## Jednostki LSTM

# In[10]:


Image(filename='rysunki/16_08.png', width=600) 


# # Implementowanie wielowarstwowej sieci rekurencyjnej przy użyciu biblioteki TensorFlow do modelowania sekwencji

# ## Analiza sentymentów na zestawie danych IMDb za pomocą wielowarstwowej sieci rekurencyjnej

# ### Przygotowanie danych

# In[11]:


Image(filename='rysunki/16_09.png', width=600) 


# In[6]:


import pyprind
import pandas as pd
from string import punctuation
import re
import numpy as np


df = pd.read_csv('filmy_dane.csv', encoding='utf-8')
print(df.head(3))


# In[7]:


## @Czytelniku: ZIGNORUJ TĘ KOMÓRKĘ
##
## Komórka ta ma na celu zmniejszenie zestawu danych
## w sytuacji, gdy notatnik ten 
## jest uruchamiany na platformie Travis Continuous Integration;
## służy to testowaniu kodu, jak również
## przyśpieszeniu obliczeń przy użyciu mniejszego
## zestawu danych.

import os


if 'TRAVIS' in os.environ:
    df = pd.read_csv('filmy_dane.csv', encoding='utf-8', nrows=500)


# In[8]:


## Wstępne przetwarzanie danych:
## Rozdziela wyrazy i
## zlicza wystąpienia każdego z nich


from collections import Counter


counts = Counter()
pbar = pyprind.ProgBar(len(df['review']),
                       title='Zliczanie wystąpień słów')
for i,review in enumerate(df['review']):
    text = ''.join([c if c not in punctuation else ' '+c+' '                     for c in review]).lower()
    df.loc[i,'review'] = text
    pbar.update()
    counts.update(text.split())


# In[9]:


## Tworzy odwzorowanie:
## Nadaje każdemu niepowtarzalnemu wyrazowi wartość liczbową

word_counts = sorted(counts, key=counts.get, reverse=True)
print(word_counts[:5])
word_to_int = {word: ii for ii, word in enumerate(word_counts, 1)}


mapped_reviews = []
pbar = pyprind.ProgBar(len(df['review']),
                       title='Przekształcanie recenzji do postaci liczbowej')
for review in df['review']:
    mapped_reviews.append([word_to_int[word] for word in review.split()])
    pbar.update()


# In[10]:


## Definiuje sekwencje o takiej samej długości:
## jeżeli długość sekwencji < 200: uzupełnianie lewej strony zerami
## jeżeli długość sekwencji > 200: wykorzystuje 200 ostatnich elementów

sequence_length = 200  ## długość sekwencji (czynnik T w naszych równaniach sieci rekurencyjnej)
sequences = np.zeros((len(mapped_reviews), sequence_length), dtype=int)
for i, row in enumerate(mapped_reviews):
    review_arr = np.array(row)
    sequences[i, -len(row):] = review_arr[-sequence_length:]

X_train = sequences[:25000, :]
y_train = df.loc[:25000, 'sentiment'].values
X_test = sequences[25000:, :]
y_test = df.loc[25000:, 'sentiment'].values


np.random.seed(123) #  w celu zachowania odtwarzalności

## Definiuje funkcję tworzącą mini-grupy:
def create_batch_generator(x, y=None, batch_size=64):
    n_batches = len(x)//batch_size
    x= x[:n_batches*batch_size]
    if y is not None:
        y = y[:n_batches*batch_size]
    for ii in range(0, len(x), batch_size):
        if y is not None:
            yield x[ii:ii+batch_size], y[ii:ii+batch_size]
        else:
            yield x[ii:ii+batch_size]


# In[11]:


## @Czytelniku: ZIGNORUJ TĘ KOMÓRKĘ
##
## Komórka ta ma na celu zmniejszenie zestawu danych
## w sytuacji, gdy notatnik ten 
## jest uruchamiany na platformie Travis Continuous Integration;
## służy to testowaniu kodu, jak również
## przyśpieszeniu obliczeń przy użyciu mniejszego
## zestawu danych.

if 'TRAVIS' in os.environ:
    X_train = sequences[:250, :]
    y_train = df.loc[:250, 'sentiment'].values
    X_test = sequences[250:500, :]
    y_test = df.loc[250:500, 'sentiment'].values


# ### Wektor właściwościowy

# In[9]:


Image(filename='rysunki/16_10.png', width=600) 


# ### Budowanie modelu sieci rekurencyjnej

# In[12]:


import tensorflow as tf


class SentimentRNN(object):
    def __init__(self, n_words, seq_len=200,
                 lstm_size=256, num_layers=1, batch_size=64,
                 learning_rate=0.0001, embed_size=200):
        self.n_words = n_words
        self.seq_len = seq_len
        self.lstm_size = lstm_size   ## liczba jednostek ukrytych
        self.num_layers = num_layers
        self.batch_size = batch_size
        self.learning_rate = learning_rate
        self.embed_size = embed_size

        self.g = tf.Graph()
        with self.g.as_default():
            tf.set_random_seed(123)
            self.build()
            self.saver = tf.train.Saver()
            self.init_op = tf.global_variables_initializer()

    def build(self):
        ## Definiuje węzły zastępcze
        tf_x = tf.placeholder(tf.int32,
                    shape=(self.batch_size, self.seq_len),
                    name='tf_x')
        tf_y = tf.placeholder(tf.float32,
                    shape=(self.batch_size),
                    name='tf_y')
        tf_keepprob = tf.placeholder(tf.float32,
                    name='tf_prawd_pozost')
        ## Tworzy warstwę wektora właściwościowego
        embedding = tf.Variable(
                    tf.random_uniform(
                        (self.n_words, self.embed_size),
                        minval=-1, maxval=1),
                    name='wekt_wlasc')
        embed_x = tf.nn.embedding_lookup(
                    embedding, tf_x, 
                    name='wlasc_x')

        ## Definiuje komórkę LSTM i tworzy ich grupę
        cells = tf.contrib.rnn.MultiRNNCell(
                [tf.contrib.rnn.DropoutWrapper(
                   tf.contrib.rnn.BasicLSTMCell(self.lstm_size),
                   output_keep_prob=tf_keepprob)
                 for i in range(self.num_layers)])

        ## Definiuje stan początkowy:
        self.initial_state = cells.zero_state(
                 self.batch_size, tf.float32)
        print('  << Stan początkowy >> ', self.initial_state)

        lstm_outputs, self.final_state = tf.nn.dynamic_rnn(
                 cells, embed_x,
                 initial_state=self.initial_state)
        ## Uwaga: wymiar obiektu lstm_outputs: 
        ##  [rozmiar_grupy, maks_czas, rozmiar.komorek_wyjsciowych]
        print('\n  << lstm_output   >> ', lstm_outputs)
        print('\n  << Stan końcowy   >> ', self.final_state)

        ## Umieszcza na wierzchu sieci w pełni połączoną warstwę:
        logits = tf.layers.dense(
                 inputs=lstm_outputs[:, -1],
                 units=1, activation=None,
                 name='logity')
        
        logits = tf.squeeze(logits, name='logity_splaszczone')
        print ('\n  << Logity        >> ', logits)
        
        y_proba = tf.nn.sigmoid(logits, name='prawdopodobienstwa')
        predictions = {
            'probabilities': y_proba,
            'labels' : tf.cast(tf.round(y_proba), tf.int32,
                 name='etykiety')
        }
        print('\n  << Prognozy   >> ', predictions)

        ## Definiuje funkcję kosztu
        cost = tf.reduce_mean(
                 tf.nn.sigmoid_cross_entropy_with_logits(
                 labels=tf_y, logits=logits),
                 name='koszt')
        
        ## Definiuje optymalizator
        optimizer = tf.train.AdamOptimizer(self.learning_rate)
        train_op = optimizer.minimize(cost, name='op_uczenia')

    def train(self, X_train, y_train, num_epochs):
        with tf.Session(graph=self.g) as sess:
            sess.run(self.init_op)
            iteration = 1
            for epoch in range(num_epochs):
                state = sess.run(self.initial_state)
                
                for batch_x, batch_y in create_batch_generator(
                            X_train, y_train, self.batch_size):
                    feed = {'tf_x:0': batch_x,
                            'tf_y:0': batch_y,
                            'tf_prawd_pozost:0': 0.5,
                            self.initial_state : state}
                    loss, _, state = sess.run(
                            ['koszt:0', 'op_uczenia', 
                             self.final_state],
                            feed_dict=feed)

                    if iteration % 20 == 0:
                        print("Epoka: %d/%d Przebieg: %d. "
                              "| Strata uczenia: %.5f" % (
                               epoch + 1, num_epochs,
                               iteration, loss))

                    iteration +=1
                if (epoch+1)%10 == 0:
                    self.saver.save(sess,
                        "model/sentyment-%d.ckpt" % epoch)

    def predict(self, X_data, return_proba=False):
        preds = []
        with tf.Session(graph = self.g) as sess:
            self.saver.restore(
                sess, tf.train.latest_checkpoint('model/'))
            test_state = sess.run(self.initial_state)
            for ii, batch_x in enumerate(
                create_batch_generator(
                    X_data, None, batch_size=self.batch_size), 1):
                feed = {'tf_x:0' : batch_x,
                        'tf_prawd_pozost:0': 1.0,
                        self.initial_state : test_state}
                if return_proba:
                    pred, test_state = sess.run(
                        ['prawdopodobienstwa:0', self.final_state],
                        feed_dict=feed)
                else:
                    pred, test_state = sess.run(
                        ['etykiety:0', self.final_state],
                        feed_dict=feed)
                    
                preds.append(pred)
                
        return np.concatenate(preds)


# #### Etap 1. Definiowanie komórek wielowarstwowej sieci RNN

# #### Etap 2. Definiowanie stanów początkowych dla komórek rekurencyjnych
# 

# #### Etap 3. Tworzenie sieci rekurencyjnej za pomocą komórek RNN i ich stanów
# 
# 

# In[13]:


## Uczenie:

n_words = max(list(word_to_int.values())) + 1

rnn = SentimentRNN(n_words=n_words, 
                   seq_len=sequence_length,
                   embed_size=256, 
                   lstm_size=128, 
                   num_layers=1, 
                   batch_size=100, 
                   learning_rate=0.001)


# In[14]:


rnn.train(X_train, y_train, num_epochs=40)


# In[16]:


## Testowanie: 
preds = rnn.predict(X_test)
y_true = y_test[:len(preds)]
print('Dokładność dla zbioru testowego.: %.3f' % (
      np.sum(preds == y_true) / len(y_true)))


# In[17]:


## Wylicza prawdopodobieństwa:
proba = rnn.predict(X_test, return_proba=True)


# ## Modelowanie języka na poziomie znaków

# In[15]:


Image(filename='rysunki/16_11.png', width=600) 


# ### Wstępna obróbka danych
# 

# In[14]:


Image(filename='rysunki/16_12.png', width=600) 


# In[15]:


Image(filename='rysunki/16_13.png', width=600) 


# In[16]:


Image(filename='rysunki/16_14.png', width=600) 


# In[18]:


import numpy as np


## Odczytywanie i przetwarzanie tekstu
with open('pg2265.txt', 'r', encoding='utf-8') as f: 

    text=f.read()

text = text[15858:]
chars = set(text)
char2int = {ch:i for i,ch in enumerate(chars)}
int2char = dict(enumerate(chars))
text_ints = np.array([char2int[ch] for ch in text], 
                     dtype=np.int32)


# In[19]:


## @Czytelniku: ZIGNORUJ TĘ KOMÓRKĘ
##
## Komórka ta ma na celu zmniejszenie zestawu danych
## w sytuacji, gdy notatnik ten 
## jest uruchamiany na platformie Travis Continuous Integration;
## służy to testowaniu kodu, jak również
## przyśpieszeniu obliczeń przy użyciu mniejszego
## zestawu danych.

if 'TRAVIS' in os.environ:
    text = text[:1000]
    chars = set(text)
    char2int = {ch:i for i,ch in enumerate(chars)}
    int2char = dict(enumerate(chars))
    text_ints = np.array([char2int[ch] for ch in text], 
                         dtype=np.int32)


# In[20]:


def reshape_data(sequence, batch_size, num_steps):
    mini_batch_length = batch_size * num_steps
    num_batches = int(len(sequence) / mini_batch_length)
    if num_batches*mini_batch_length + 1 > len(sequence):
        num_batches = num_batches - 1
    ## Przycina sekwencję na końcu, aby usunąć 
    ## znaki nietworzące pełnej grupy
    x = sequence[0 : num_batches*mini_batch_length]
    y = sequence[1 : num_batches*mini_batch_length + 1]
    ## Rozdziela tablice x i y na listy sekwencji: 
    x_batch_splits = np.split(x, batch_size)
    y_batch_splits = np.split(y, batch_size)
    ## Łączy grupy ze sobą
    ## rozmiar_grupy x całkowita_długość_grupy
    x = np.stack(x_batch_splits)
    y = np.stack(y_batch_splits)
    
    return x, y

## Testowanie:
train_x, train_y = reshape_data(text_ints, 64, 10)
print(train_x.shape)
print(train_x[0, :10])
print(train_y[0, :10])
print(''.join(int2char[i] for i in train_x[0, :50]))


# In[21]:


np.random.seed(123)

def create_batch_generator(data_x, data_y, num_steps):
    batch_size, tot_batch_length = data_x.shape    
    num_batches = int(tot_batch_length/num_steps)
    for b in range(num_batches):
        yield (data_x[:, b*num_steps: (b+1)*num_steps], 
               data_y[:, b*num_steps: (b+1)*num_steps])
        
bgen = create_batch_generator(train_x[:,:100], train_y[:,:100], 15)
for b in bgen:
    print(b[0].shape, b[1].shape, end='  ')
    print(''.join(int2char[i] for i in b[0][0,:]).replace('\n', '*'), '    ',
          ''.join(int2char[i] for i in b[1][0,:]).replace('\n', '*'))


# ### Tworzenie sieci RNN przetwarzającej znaki

# In[22]:


import tensorflow as tf
import os

class CharRNN(object):
    def __init__(self, num_classes, batch_size=64, 
                 num_steps=100, lstm_size=128, 
                 num_layers=1, learning_rate=0.001, 
                 keep_prob=0.5, grad_clip=5, 
                 sampling=False):
        self.num_classes = num_classes
        self.batch_size = batch_size
        self.num_steps = num_steps
        self.lstm_size = lstm_size
        self.num_layers = num_layers
        self.learning_rate = learning_rate
        self.keep_prob = keep_prob
        self.grad_clip = grad_clip
        
        self.g = tf.Graph()
        with self.g.as_default():
            tf.set_random_seed(123)

            self.build(sampling=sampling)
            self.saver = tf.train.Saver()
            self.init_op = tf.global_variables_initializer()
            
    def build(self, sampling):
        if sampling == True:
            batch_size, num_steps = 1, 1
        else:
            batch_size = self.batch_size
            num_steps = self.num_steps

        tf_x = tf.placeholder(tf.int32, 
                              shape=[batch_size, num_steps], 
                              name='tf_x')
        tf_y = tf.placeholder(tf.int32, 
                              shape=[batch_size, num_steps], 
                              name='tf_y')
        tf_keepprob = tf.placeholder(tf.float32, 
                              name='tf_prawd_pozost')

        # Kodowanie gorącojedynkowe:
        x_onehot = tf.one_hot(tf_x, depth=self.num_classes)
        y_onehot = tf.one_hot(tf_y, depth=self.num_classes)

        ### Buduje komórki wielowarstwowej sieci RNN
        cells = tf.contrib.rnn.MultiRNNCell(
            [tf.contrib.rnn.DropoutWrapper(
                tf.contrib.rnn.BasicLSTMCell(self.lstm_size), 
                output_keep_prob=tf_keepprob) 
            for _ in range(self.num_layers)])
        
        ## Definiuje stan początkowy
        self.initial_state = cells.zero_state(
                    batch_size, tf.float32)

        ## Realizuje każdy takt sekwencji poprzez sieć RNN 
        lstm_outputs, self.final_state = tf.nn.dynamic_rnn(
                    cells, x_onehot, 
                    initial_state=self.initial_state)
        
        print('  << lstm_outputs  >>', lstm_outputs)

        seq_output_reshaped = tf.reshape(
                    lstm_outputs, 
                    shape=[-1, self.lstm_size],
                    name='wynik_wyjscia_przekszt')

        logits = tf.layers.dense(
                    inputs=seq_output_reshaped, 
                    units=self.num_classes,
                    activation=None,
                    name='logity')

        proba = tf.nn.softmax(
                    logits, 
                    name='prawdopodobienstwa')
        print(proba)

        y_reshaped = tf.reshape(
                    y_onehot, 
                    shape=[-1, self.num_classes],
                    name='y_przekszt')
        cost = tf.reduce_mean(
                    tf.nn.softmax_cross_entropy_with_logits(
                        logits=logits, 
                        labels=y_reshaped),
                    name='koszt')

        # Obcinanie gradientów pozwalające uniknąć „eksplodujących gradientów"
        tvars = tf.trainable_variables()
        grads, _ = tf.clip_by_global_norm(
                    tf.gradients(cost, tvars), 
                    self.grad_clip)
        optimizer = tf.train.AdamOptimizer(self.learning_rate)
        train_op = optimizer.apply_gradients(
                    zip(grads, tvars),
                    name='op_uczenia')
        
    def train(self, train_x, train_y, 
              num_epochs, ckpt_dir='./model/'):
        ## Tworzy katalog punktów kontrolnych
        ## jeśli jeszcze nie istnieje
        if not os.path.exists(ckpt_dir):
            os.mkdir(ckpt_dir)
            
        with tf.Session(graph=self.g) as sess:
            sess.run(self.init_op)

            n_batches = int(train_x.shape[1]/self.num_steps)
            iterations = n_batches * num_epochs
            for epoch in range(num_epochs):

                # Uczenie sieci
                new_state = sess.run(self.initial_state)
                loss = 0
                ## Generator mini-grup:
                bgen = create_batch_generator(
                        train_x, train_y, self.num_steps)
                for b, (batch_x, batch_y) in enumerate(bgen, 1):
                    iteration = epoch*n_batches + b
                    
                    feed = {'tf_x:0': batch_x,
                            'tf_y:0': batch_y,
                            'tf_prawd_pozost:0': self.keep_prob,
                            self.initial_state : new_state}
                    batch_cost, _, new_state = sess.run(
                            ['koszt:0', 'op_uczenia', 
                                self.final_state],
                            feed_dict=feed)
                    if iteration % 10 == 0:
                        print('Epoka %d/%d. Przebieg %d.'
                              '| Strata uczenia: %.4f' % (
                              epoch + 1, num_epochs, 
                              iteration, batch_cost))

                ## Zapisuje wyuczony model    
                self.saver.save(
                        sess, os.path.join(
                            ckpt_dir, 'modelowanie_jezyka.ckpt'))
                              
                              
                
    def sample(self, output_length, 
               ckpt_dir, starter_seq="The "):
        observed_seq = [ch for ch in starter_seq]        
        with tf.Session(graph=self.g) as sess:
            self.saver.restore(
                sess, 
                tf.train.latest_checkpoint(ckpt_dir))
            ## 1: uruchamia model za pomocą sekwencji początkowej 
            new_state = sess.run(self.initial_state)
            for ch in starter_seq:
                x = np.zeros((1, 1))
                x[0,0] = char2int[ch]
                feed = {'tf_x:0': x,
                        'tf_prawd_pozost:0': 1.0,
                        self.initial_state: new_state}
                proba, new_state = sess.run(
                        ['prawdopodobienstwa:0', self.final_state], 
                        feed_dict=feed)

            ch_id = get_top_char(proba, len(chars))
            observed_seq.append(int2char[ch_id])
            
            ## 2: uruchamia model za pomocą zaktualizowanej zmiennej observed_seq
            for i in range(output_length):
                x[0,0] = ch_id
                feed = {'tf_x:0': x,
                        'tf_prawd_pozost:0': 1.0,
                        self.initial_state: new_state}
                proba, new_state = sess.run(
                        ['prawdopodobienstwa:0', self.final_state], 
                        feed_dict=feed)

                ch_id = get_top_char(proba, len(chars))
                observed_seq.append(int2char[ch_id])

        return ''.join(observed_seq)


# In[23]:


def get_top_char(probas, char_size, top_n=5):
    p = np.squeeze(probas)
    p[np.argsort(p)[:-top_n]] = 0.0
    p = p / np.sum(p)
    ch_id = np.random.choice(char_size, 1, p=p)[0]
    return ch_id


# In[24]:


batch_size = 64
num_steps = 100 
train_x, train_y = reshape_data(text_ints, 
                                batch_size, 
                                num_steps)

rnn = CharRNN(num_classes=len(chars), batch_size=batch_size)
rnn.train(train_x, train_y, 
          num_epochs=100,
          ckpt_dir='./model-100/')


# In[25]:


np.random.seed(123)
rnn = CharRNN(len(chars), sampling=True)

print(rnn.sample(ckpt_dir='./model-100/', 
                 output_length=500))


# In[26]:


## run for 200 epochs
batch_size = 64
num_steps = 100 

rnn = CharRNN(num_classes=len(chars), batch_size=batch_size)
rnn.train(train_x, train_y, 
          num_epochs=200,
          ckpt_dir='./model-200/')


# In[27]:


del rnn

np.random.seed(123)
rnn = CharRNN(len(chars), sampling=True)
print(rnn.sample(ckpt_dir='./model-200/', 
                 output_length=500))


# # Podsumowanie

# ...

# ---
# 
# Czytelnicy mogą zignorować poniższą komórkę.

# In[1]:


get_ipython().system(' python ../.convert_notebook_to_script.py --input ch16.ipynb --output ch16.py')

