"""
Programowanie obiektowe w Pythonie 3

Rozdział 3, Kiedy obiekty są do siebie podobne
"""
from __future__ import annotations
from pathlib import Path
from typing import Protocol


class Playable(Protocol):
    def play(self) -> None:
        ...


class AudioFile(Playable):
    ext: str

    def __init__(self, filepath: Path) -> None:
        if not filepath.suffix == self.ext:
            raise ValueError("Nieprawidłowy format pliku")
        self.filepath = filepath


class MP3File(AudioFile):
    ext = ".mp3"

    def play(self) -> None:
        print(f"odtwarzam {self.filepath} jako mp3")


class WavFile(AudioFile):
    ext = ".wav"

    def play(self) -> None:
        print(f"odtwarzam {self.filepath} jako wav")


class OggFile(AudioFile):
    ext = ".ogg"

    def play(self) -> None:
        print(f"odtwarzam {self.filepath} jako ogg")


class FlacFile(Playable):
    def __init__(self, filepath: Path) -> None:
        if not filepath.suffix == ".flac":
            raise ValueError("to nie jest plik .flac")
        self.filepath = filepath

    def play(self) -> None:
        print(f"odtwarzam {self.filepath} jako flac")


test_audio_1 = """
>>> p_1 = MP3File(Path("Autobiografia.mp3"))
>>> p_1.play()
odtwarzam Autobiografia.mp3 jako mp3
>>> p_2 = WavFile(Path("Baśka.wav"))
>>> p_2.play()
odtwarzam Baśka.wav jako wav
>>> p_3 = OggFile(Path("Bombonierka.ogg"))
>>> p_3.play()
odtwarzam Bombonierka.ogg jako ogg
>>> error = MP3File(Path("Ona jest ze snu.mov"))
Traceback (most recent call last):
...
ValueError: Nieprawidłowy format pliku

"""

test_flac = """
>>> f_1 = FlacFile(Path("Stairway to Heaven.flac"))
>>> f_1.play()
odtwarzam Stairway to Heaven.flac jako flac
>>> error = FlacFile(Path("Nothing Else Matters.mov"))
Traceback (most recent call last):
...
ValueError: to nie jest plik .flac

"""


class Entertainment:
    def __init__(self, play_list: list[Playable]) -> None:
        self.queue = play_list

    def play(self) -> None:
        for song in self.queue:
            song.play


def fragile() -> None:
    s1 = WavFile(Path("Roundabout.wav"))
    s2 = MP3File(Path("Cans and Brahms.mp3"))
    s3 = OggFile(Path("We Have Heaven.ogg"))
    s4 = FlacFile(Path("South Side of the Sky.flac"))
    side_1 = Entertainment([s1, s2, s3, s4])


__test__ = {name: case for name, case in globals().items() if name.startswith("test_")}
