require 'yaml'
require 'wordplay'

# Implementacja prostego bota
class Bot
  attr_reader :name
  
  # Inicjalizacja obiektu bota, zaadowanie danych YAML z pliku zewntrznego
  # i ustawienie nazwy bota. Jeeli proces adowania danych nie powiedzie si,
  # nastpuje rzucenie wyjtku.
  def initialize(options)
    @name = options[:name] || "Bot bez nazwy"
    begin
      @data = YAML.load(File.open(options[:data_file]).read)
    rescue
      raise "Nie mona zaadowa danych bota"
    end
  end
  
  # Zwrcenie losowego powitania z pliku danych bota
  def greeting
    random_response(:greeting)
  end
  
  # Zwrcenie losowego poegnania z pliku danych bota
  def farewell
    random_response(:farewell)
  end  
  
  # Odpowied na tekst wpisany przez uytkownika
  def response_to(input)
    prepared_input = preprocess(input.downcase)
    sentence = best_sentence(prepared_input)
    reversed_sentence = WordPlay.switch_pronouns(sentence)
    responses = possible_responses(sentence)
    responses[rand(responses.length)]
  end


  private
  
  # Wybr z tablicy asocjacyjnej :responses losowej frazy odpowiedzi
  # i wstawienie we frazie metadanych
  def random_response(key)
    random_index = rand(@data[:responses][key].length)
    @data[:responses][key][random_index].gsub(/\[name\]/, @name)
  end
  
  # Wykonanie zada przetwarzania wstpnego na danych wejciowych dla bota
  def preprocess(input)
    perform_substitutions(input)
  end
  
  # Zastpienie sw i fraz w danych wejciowych
  # zgodnie z zawartoci tablicy :presubs
  def perform_substitutions(input)
    @data[:presubs].each { |s| input.gsub!(s[0], s[1]) }
    input
  end
  
  # Odnalezienie zdania, w ktrym znajduje si najwiksza liczba
  # pojedynczych sw-kluczy z tablicy :responses;
  # prawdopodobnie jest to 'najlepsze' zdanie do parsowania
  def best_sentence(input)
    hot_words = @data[:responses].keys.select do |k|
      k.class == String && k =~ /^\w+$/
    end
    
    WordPlay.best_sentence(input.sentences, hot_words)
  end
  


  # Na podstawie zawartoci zdania przechodzenie przez tablic
  # :responses bota i zebranie wszystkich fraz, ktrych mona uy
  # w charakterze odpowiedzi
  def possible_responses(sentence)
    responses = []

    # Odnalezienie wszystkich wzorcw, ktre bd dopasowywane
    @data[:responses].keys.each do |pattern|
      next unless pattern.is_a?(String)
    
      # Sprawdzenie dla kadego wzorca, czy znajduje si on w podanym zdaniu.
      # Przed dokonaniem sprawdzenia usuwane s symbole zastpienia (*).
      # Wszystkie odpowiedzi s przenoszone do tablicy odpowiedzi.
      if sentence.match('\b' + pattern.gsub(/\*/, '') + '\b')
        # Jeeli wzorzec zawiera symbole do zastpienia,
        # naley wykona odpowiednie zastpienia.
        if pattern.include?('*')
          responses << @data[:responses][pattern].collect do |phrase|
            # Usunicie wszystkich sw przed symbolem do zastpienia
            # i pozostawienie wszystkich sw po tym symbolu
            matching_section = sentence.sub(/^.*#{pattern}\s+/, '')

            # Zastpienie tekstu po symbolu do zastpienia
            # tekstem ze zmienionymi zaimkami
            phrase.sub('*', WordPlay.switch_pronouns(matching_section))
          end
        else
          # Brak symboli do zastpienia? Wystarczy wic doda zdania do tablicy
          responses << @data[:responses][pattern]
        end
      end
    end
  
    # Jeeli nie znaleziono dopasowa, dodanie odpowiedzi domylnych
    responses << @data[:responses][:default] if responses.empty?
  
    # Kompresja blokw odpowiedzi do postaci paskiej tablicy
    responses.flatten
  end

end

