# Przykadowy kod z ksiki Ruby. Programowanie
#include "ruby.h"
#include "cdjukebox.h"

static VALUE cCDPlayer;

// Funkcja pomocnicza zwalniajc struktur CDJukebox producenta
static void cd_free(void *p) {
  free_jukebox(p);
}

// Przydzielanie nowego obiektu CDPlayer i opakowywanie
// struktury CDJukebox producenta

static VALUE cd_alloc(VALUE klass) {
  CDJukebox *jukebox;
  VALUE obj;

  // Biblioteka producenta pozwala utworzy obiekt jukebox
  jukebox = new_jukebox();

  // nastpnie mona opakowa go w obiekt CDPlayer jzyka Ruby
  obj = Data_Wrap_Struct(klass, 0, cd_free, jukebox);

  return obj;
}

// Przydzielanie nowego obiektu CDPlayer do
// konkretnej jednostki
static VALUE cd_initialize(VALUE self, VALUE unit) {
  int unit_id;
  CDJukebox *jb;

  Data_Get_Struct(self, CDJukebox, jb);

  unit_id = NUM2INT(unit);
  assign_jukebox(jb, unit_id);

  return self;
}

// Kopiowanie stanu (uywane przez metody clone i dup). W przypadku obiektu jukebox
// kod tworzy nowy obiekt producenta i ustawia jego numer jednostkowy na podstawie
// starego

static VALUE cd_init_copy(VALUE copy, VALUE orig) {

  CDJukebox *orig_jb;
  CDJukebox *copy_jb;

  if (copy == orig)
    return copy;

  // mona zainicjowa kopi jedynie na podstawie innego obiektu CDPlayer
  // lub obiektw klas pochodnych

  if (TYPE(orig) != T_DATA ||
      RDATA(orig)->dfree != (RUBY_DATA_FUNC)cd_free) {
    rb_raise(rb_eTypeError, "wrong argument type");
  }

  // kopiuje wszystkie pola ze struktury CDJukebox
  // oryginalnego obiektu do nowego obiektu 

  Data_Get_Struct(orig, CDJukebox, orig_jb);
  Data_Get_Struct(copy, CDJukebox, copy_jb);
  MEMCPY(copy_jb, orig_jb, CDJukebox, 1);

  return copy;
}

// Wywoanie zwrotne dotyczce postpu zwraca do jednostki wywoujce procent wykonania zadania
static void progress(CDJukebox *rec, int percent) {

  if (rb_block_given_p()) {
    if (percent > 100) percent = 100;
    if (percent < 0) percent = 0;
    rb_yield(INT2FIX(percent));
  }
}

// Przechodzenie do danej czci cieki i zgaszanie wywoania zwrotnego
// zwizanego z postpem w czasie tej czynnoci

static VALUE
cd_seek(VALUE self, VALUE disc, VALUE track) {

  CDJukebox *jb;
  Data_Get_Struct(self, CDJukebox, jb);

  jukebox_seek(jb,
               NUM2INT(disc),
               NUM2INT(track),
               progress);
  return Qnil;
}

// Zwraca redni czas wyszukiwania dla danej jednostki
static VALUE
cd_seek_time(VALUE self)
{
  double tm;
  CDJukebox *jb;
  Data_Get_Struct(self, CDJukebox, jb);
  tm = get_avg_seek_time(jb);
  return rb_float_new(tm);
}

// Zwraca numer jednostkowy odtwarzacza
static VALUE
cd_unit(VALUE self) {

  CDJukebox *jb;
  Data_Get_Struct(self, CDJukebox, jb);

  return INT2NUM(jb->unit_id);
}

void Init_CDPlayer() {
  cCDPlayer = rb_define_class("CDPlayer", rb_cObject);
  rb_define_alloc_func(cCDPlayer, cd_alloc);

  rb_define_method(cCDPlayer, "initialize", cd_initialize, 1);
  rb_define_method(cCDPlayer, "initialize_copy", cd_init_copy, 1);

  rb_define_method(cCDPlayer, "seek", cd_seek, 2);
  rb_define_method(cCDPlayer, "seek_time", cd_seek_time, 0);
  rb_define_method(cCDPlayer, "unit", cd_unit, 0);
}


