# Przykładowy kod z książki Ruby. Programowanie
# Ten moduł hermetyzuje funkcjonalność związaną
# z generowaniem ciągów Fibonacciego.
#--
# Copyright (c) 2004 Dave Thomas, The Pragmatic Programmers, LLC.
# Licensed under the same terms as Ruby. No warranty is provided.
module Fibonacci

  # Oblicz pierwsze _count_ liczb Fibonacciego, począwszy od 1,1.
  #
  # :call-seq:
  #   Fibonacci.sequence(count)                 -> array
  #   Fibonacci.sequence(count) {|val| ... }    -> nil
  #
  # Jeśli podany jest blok, przekazuj kolejne wartości do bloku
  # i zwróć +nil+, w przeciwnym wypadku zwróć wszystkie wartości jako tablicę.
  def Fibonacci.sequence(count, &block)
    result, block = setup_optional_block(block)
    generate do |val|
      break if count <= 0
      count -= 1
      block[val]
    end
    result
  end

  # Oblicz liczby Fibonacciego do i łącznie z wartością _max_.
  #
  # :call-seq:
  #   Fibonacci.upto(count)                -> array
  #   Fibonacci.upto(count) {|val ... }   -> nil
  #
  # Jeśli podany jest blok, dostarcz kolejne wartości do bloku
  # i zwróć +nil+, w przeciwnym wypadku zwróć wszystkie wartości jako tablicę.
  def Fibonacci.upto(max, &block)
    result, block = setup_optional_block(block)
    generate do |val|
      break if val > max
      block[val]
    end
    result
  end

  private

  # Zwracaj kolejne liczby ciągu Fibonaciego do bloku.
  def Fibonacci.generate
    f1, f2 = 1, 1
    loop do
      yield f1
      f1, f2 = f2, f1+f2
    end
  end

  # Jeśli jako parametr podany jest blok, użyj go, w przeciwnym wypadku
  # zwróć tablicę do gromadzenia wyników. Zwracaj wartość wyniku i blok.
  def Fibonacci.setup_optional_block(block)
    if block.nil?
      [ result = [], lambda {|val| result << val } ]
    else
      [ nil, block ]
    end
  end
end
