/* Rozdzia 10. ThreeStageCS.c */
/* Trzyetapowy system z producentem i konsumentem. */
/* Inne pliki potrzebne w tym projekcie - bezporednio lub		*/
/* w postaci bibliotek (preferowane s biblioteki DLL):			*/
/*		QueueObj.c												*/
/*		Messages.c												*/

/* Stosowanie: ThreeStageCS liczba_par cel */
/* Program uruchamia liczba_par par wtkw producent-konsument. */
/* Kady producent musi utworzy w sumie */
/* cel komunikatw. Wszystkie komunikaty maj doczon */
/* nazw konsumenta, ktry powinien go otrzyma. */
/* Komunikaty s wysyane do wtku przekanikowego, ktry wykonuje */
/* dodatkowe operacje przed przesaniem grup komunikatw do */
/* wtku odbiorczego. Na zakoczenie wtek odbiorczy przekazuje */
/* komunikaty do wtkw konsumentw. */
/*																*/

/* Przekanik: odbiera komunikaty po jednym od producentw i    */
/* tworzy komunikat zawierajcy do "TBLOCK_SIZE" komunikatw	*/
/* wysyany do odbiorcy. 	                                    */
/* Odbiorca: przyjmuje bloki komunikatw wysane przez przekanik	*/
/* i wysya pojedyncze komunikaty do okrelonego konsumenta.	*/

#include "Everything.h"
#include "SynchObjCS.h" /* To jedyna zmiana wzgldem programu ThreeStage.c. */
#include "messages.h"
#include <time.h>

#define DELAY_COUNT 1000
#define MAX_THREADS 1024

/* Dugo kolejek i czynnik zwizany z ich blokowaniem. S to arbitralne wartoci i */
/* mona je dostosowa pod ktem wydajnoci. Obecne wartoci nie s */
/* dobrze zrwnowaone.							*/

#define TBLOCK_SIZE 5  	/* Przekanik czy po pi komunikatw. */
#define TBLOCK_TIMEOUT 50 /* Limit czasu oczekiwania na komunikaty. */
#define P2T_QLEN 10 	/* Dugo kolejki midzy producentem i przekanikiem. */
#define T2R_QLEN 4	/* Dugo kolejki midzy przekanikiem i odbiorc. */
#define R2C_QLEN 4	/* Dugo kolejki midzy odbiorc i konsumentem -
                      istnieje jedna taka kolejka dla kadego konsumenta. */

DWORD WINAPI producer (PVOID);
DWORD WINAPI consumer (PVOID);
DWORD WINAPI transmitter (PVOID);
DWORD WINAPI receiver (PVOID);

typedef struct THARG_TAG {
	DWORD threadNumber;
	DWORD workGoal;    /* Uywana przez producentw. */
	DWORD workDone;    /* Uywana przez producentw i konsumentw. */
} THARG;

/* Pogrupowane komunikaty wysyane przez przekanik do odbiorcy.		*/
typedef struct T2R_MSG_OBJ_TAG {
	DWORD numMessages; /* Liczba komunikatw.	*/
	MSG_BLOCK messages [TBLOCK_SIZE];
} T2R_MSG_OBJ;

/* Argument dla wtkw przekanikowych i odbiorczych. */
typedef struct TR_ARG_TAG {
	DWORD nProducers;  /* Liczba aktywnych producentw. */
} TR_ARG;

QUEUE_OBJECT p2tq, t2rq, *r2cqArray;

static volatile DWORD ShutDown = 0;
static DWORD EventTimeout = 50;
DWORD trace = 0;


int _tmain (int argc, LPTSTR argv[])
{
	DWORD tStatus, nThread, iThread, goal;
	HANDLE *producerThreadArray, *consumerThreadArray, transmitterThread, receiverThread;
	THARG *producerArg, *consumerArg;
	TR_ARG transmitterArg, receiverArg;

#if (defined (REQUIRES_NT6))
	if (!WindowsVersionOK (6, 0)) 
		ReportError (_T("Ten program wymaga systemu Windows NT 6.0 lub nowszego."), 1, TRUE);
#elif (!defined (NO_SOAW))
	if (!WindowsVersionOK (4, 0)) 
		ReportError (_T("Ten program wymaga systemu Windows NT 4.0 lub nowszego."), 1, FALSE);
#endif

	if (argc < 3) {
		_tprintf (_T("Stosowanie: ThreeStageCS liczba_par cel [sledzenie]\n"));
		return 1;
	}
	srand ((int)time(NULL));	/* Inicjowanie generatora liczb losowych. */
	
	nThread = _ttoi(argv[1]);
	if (nThread > MAX_THREADS) {
		_tprintf (_T("Maksymalna liczba producentow i konsumentow to %d.\n"), MAX_THREADS);
		return 2;
	}
	receiverArg.nProducers = transmitterArg.nProducers = nThread;

	goal = _ttoi(argv[2]);
	if (argc >= 4) trace = _ttoi(argv[3]);

	producerThreadArray = malloc (nThread * sizeof(HANDLE));
	producerArg = calloc (nThread, sizeof (THARG));
	consumerThreadArray = malloc (nThread * sizeof(HANDLE));
	consumerArg = calloc (nThread, sizeof (THARG));
	
	if (producerThreadArray == NULL || producerArg == NULL
	 || consumerThreadArray == NULL || consumerArg == NULL)
		ReportError (_T("Nie mozna zaalokowac pamieci roboczej dla watkow."), 1, FALSE);
	
	QueueInitialize (&p2tq, sizeof(MSG_BLOCK), P2T_QLEN);
	QueueInitialize (&t2rq, sizeof(T2R_MSG_OBJ), T2R_QLEN);
	/* Alokowanie i inicjowanie kolejek midzy odbiorc oraz konsumentem. */
	r2cqArray = calloc (nThread, sizeof(QUEUE_OBJECT));
	if (r2cqArray == NULL) ReportError (_T("Nie mozna zaalokowac pamieci dla kolejek miedzy odbiorca i konsumentem."), 
					20, FALSE);
	
	for (iThread = 0; iThread < nThread; iThread++) {
		/* Inicjowanie kolejki midzy odbiorc i konsumentem dla danego wtku konsumenta. */
		QueueInitialize (&r2cqArray[iThread], sizeof(MSG_BLOCK), R2C_QLEN);
		/* Zapenianie argumentu wtku. */
		consumerArg[iThread].threadNumber = iThread;
		consumerArg[iThread].workGoal = goal;
		consumerArg[iThread].workDone = 0;

		consumerThreadArray[iThread] = (HANDLE)_beginthreadex (NULL, 0,
				consumer, &consumerArg[iThread], 0, NULL);
		if (consumerThreadArray[iThread] == NULL) 
			ReportError (_T("Nie mozna utworzyc watku konsumenta."), 2, TRUE);

		producerArg[iThread].threadNumber = iThread;
		producerArg[iThread].workGoal = goal;
		producerArg[iThread].workDone = 0;
		producerThreadArray[iThread] = (HANDLE)_beginthreadex (NULL, 0,
			producer, &producerArg[iThread], 0, NULL);
		if (producerThreadArray[iThread] == NULL) 
			ReportError (_T("Nie mozna utworzyc watku producenta."), 3, TRUE);
	}

	transmitterThread = (HANDLE)_beginthreadex (NULL, 0, transmitter, &transmitterArg, 0, NULL);
	if (transmitterThread == NULL) 
		ReportError (_T("Nie mozna utworzyc watku przekaznika."), 4, TRUE);
	receiverThread = (HANDLE)_beginthreadex (NULL, 0, receiver, &receiverArg, 0, NULL);
	if (receiverThread == NULL) 
		ReportError (_T("Nie mozna utworzyc watku odbiorcy."), 5, TRUE);
	
	_tprintf (_T("WATEK GLOWNY: wszystkie watki dzialaja.\n"));	
	/* Oczekiwanie na zakoczenie pracy przez producentw. */
	/* Ta implementacja zezwala na utworzenie zbyt wielu wtkw dla funkcji WaitForMultipleObjects, */
	/* jednak mona te wywoywa j w ptli. */
	for (iThread = 0; iThread < nThread; iThread++) {
		tStatus = WaitForSingleObject (producerThreadArray[iThread], INFINITE);
		if (tStatus != 0) 
			ReportError (_T("Blad oczekiwania na watek producenta."), 6, TRUE);
		if (trace >= 1) _tprintf (_T("WATEK GLOWNY: producent %d utworzyl %d jednostek roboczych.\n"),
			iThread, producerArg[iThread].workDone);
	}
	/* Producenci zakoczyli prac. */
	_tprintf (_T("WATEK GLOWNY: wszyscy producenci ukonczyli prace.\n"));

	/* Oczekiwanie na zakoczenie pracy przez konsumentw. */
	for (iThread = 0; iThread < nThread; iThread++) {
		tStatus = WaitForSingleObject (consumerThreadArray[iThread], INFINITE);
		if (tStatus != 0) 
			ReportError (_T("Blad oczekiwania na watek konsumenta."), 7, TRUE);
		if (trace >= 1) _tprintf (_T("WATEK GLOWNY: konsument %d skonsumowal %d jednostek.\n"),
			iThread, consumerArg[iThread].workDone);
	}
	_tprintf (_T("WATEK GLOWNY: wszyscy konsumenci zakonczyli prace.\n"));	

	ShutDown = 1; /* Ustawianie flagi zamykania. */
	tStatus = WaitForSingleObject (transmitterThread, INFINITE);
	if (tStatus != WAIT_OBJECT_0) 
		ReportError (_T("Nieudane oczekiwanie na watek przekaznika."), 8, TRUE);
	tStatus = WaitForSingleObject (receiverThread, INFINITE);

	if (tStatus != WAIT_OBJECT_0) 
		ReportError (_T("Nieudane oczekiwanie na watek odbiorcy."), 9, TRUE);

	QueueDestroy (&p2tq);
	QueueDestroy (&t2rq);
	for (iThread = 0; iThread < nThread; iThread++) {
		QueueDestroy (&r2cqArray[iThread]);
		CloseHandle(consumerThreadArray[iThread]);
		CloseHandle(producerThreadArray[iThread]);
	}
	free (r2cqArray);
	free (producerThreadArray); free (consumerThreadArray);
	free (producerArg); free(consumerArg);
	CloseHandle(transmitterThread); CloseHandle(receiverThread);
	_tprintf (_T("System zakonczyl prace. Zamykanie.\n"));
	return 0;
}

DWORD WINAPI producer (PVOID arg)
{
	THARG * parg;
	DWORD iThread;
	MSG_BLOCK msg;
	
	parg = (THARG *)arg;	
	iThread = parg->threadNumber;

	while (parg->workDone < parg->workGoal) { 
		/* Okresowe tworzenie jednostek roboczych do momentu dojcia do */
        /* docelowego poziomu. Komunikaty maj adres rdowy i docelowy. */
        /* Tu s one takie same, jednak mog by inne. */
		delay_cpu (DELAY_COUNT * rand() / RAND_MAX);
		MessageFill (&msg, iThread, iThread, parg->workDone);
		
		/* Umieszczanie komunikatu w kolejce. */
		QueuePut (&p2tq, &msg, sizeof(msg), INFINITE);
				
		parg->workDone++;		
	}
	
	/* Wysyanie kocowego komunikatu gotowe (z ujemn liczb porzdkow). */
	MessageFill (&msg, iThread, iThread, -1);
	QueuePut (&p2tq, &msg, sizeof(msg), INFINITE);
	return 0;		
}


DWORD WINAPI transmitter (PVOID arg)
{
	/* Pobieranie wielu komunikatw i czenie ich w jeden */
    /* zbiorczy komunikat dla odbiorcy. */

	DWORD tStatus, im;
	T2R_MSG_OBJ t2r_msg = {0};
	TR_ARG * tArg = (TR_ARG *)arg;

	while (!ShutDown) {
		t2r_msg.numMessages = 0;
		/* Pakowanie komunikatw w celu przesania ich do odbiorcy. */
		for (im = 0; im < TBLOCK_SIZE; im++) {
			tStatus = QueueGet (&p2tq, &t2r_msg.messages[im], sizeof(MSG_BLOCK), INFINITE);
			if (tStatus != 0) break;
			t2r_msg.numMessages++;
			/* Zmniejszanie liczby aktywnych konsumentw przy ujemnej liczbie porzdkowej. */
			if (t2r_msg.messages[im].sequence < 0) {
				tArg->nProducers--;
				if (tArg->nProducers <= 0) break;
			}
		}

		/* Przesyanie bloku komunikatw. */
		tStatus = QueuePut (&t2rq, &t2r_msg, sizeof(t2r_msg), INFINITE);
		if (tStatus != 0) return tStatus;
		/* Koczenie pracy przez przekanik, jeli nie ma ju aktywnych konsumentw. */
		if (tArg->nProducers <=0) return 0;
	}
	return 0;
}


DWORD WINAPI receiver (PVOID arg)
{
	/* Pobieranie komunikatw zbiorczych od przekanika, odblokowywanie ich */
    /* i przesyanie do okrelonego konsumenta.	*/

	DWORD tStatus, im, ic;
	T2R_MSG_OBJ t2r_msg;
	TR_ARG * tArg = (TR_ARG *)arg;
	
	while (!ShutDown) {
		tStatus = QueueGet (&t2rq, &t2r_msg, sizeof(t2r_msg), INFINITE);
		if (tStatus != 0) return tStatus;
		/* Przesyanie komunikatw do odpowiedniego konsumenta. */
		for (im = 0; im < t2r_msg.numMessages; im++) {
			ic = t2r_msg.messages[im].destination; /* Docelowy konsument. */
			tStatus = QueuePut (&r2cqArray[ic], &t2r_msg.messages[im], sizeof(MSG_BLOCK), INFINITE);
			if (tStatus != 0) return tStatus;
			if (t2r_msg.messages[im].sequence < 0) {
				tArg->nProducers--;
				if (tArg->nProducers <= 0) break;
			}
		}
		/* Koczenie pracy przez przekanik, jeli nie ma aktywnych konsumentw. */
		if (tArg->nProducers <=0) return 0;
	}
	return 0;
}

DWORD WINAPI consumer (PVOID arg)
{
	THARG * carg;
	DWORD tStatus, iThread;
	MSG_BLOCK msg;
	QUEUE_OBJECT *pr2cq;

	carg = (THARG *) arg;
	iThread = carg->threadNumber;
	
	carg = (THARG *)arg;	
	pr2cq = &r2cqArray[iThread];

	while (carg->workDone < carg->workGoal) { 
		/* Odbieranie i wywietlanie komunikatw. */
		tStatus = QueueGet (pr2cq, &msg, sizeof(msg), INFINITE);
		if (tStatus != 0) return tStatus;
		if (msg.sequence < 0) return 0;  /* Ostatni komunikat. */
				
		if (trace >= 1) _tprintf (_T("Komunikat odebrany przez konsumenta numer %d. Komunikat numer %d.\n"), iThread, msg.sequence);
		if (trace >= 2) MessageDisplay (&msg);

		carg->workDone++;		
	}

	return 0;
}
