/*	SynchObj.c

	Copyright (c) 1998, Johnson M. Hart
	Zazwalam na wykorzystanie w kadym celu pod warunkiem prawidowego zamieszczenia 
	praw autorskich.
	Nie gwarantuj poprawnoci do jakiegokolwiek uytku.

	Biblioteka "obiektw" synchronizacji uzupeniajca standardowe 
	zdarzenia i muteksy z systemw Win32. 
	
	Dla uproszczenia ta implementacja ma wsplny plik nagwkowy (Synchronize.h) z
	inna bibliotek obiektw synchronizacji (EvMxSem.c), ktra ma dziaa 
	w systemie Windows CE, gdzie nie ma semaforw.

	Uwagi na temat implementacji: 
	1.	Wszystkie potrzebne wewntrzne struktury danych s alokowane na stercie procesu.
	2.	W odpowiednich miejscach zwracany jest nowy kod bdu (zobacz plik
		nagwkowy) lub, jeli jest to bd systemu Win32, kod pozostaje niezmieniony.
	3.	Warto zauway uchwyt nowego typu - "SYNCHHANDLE". Obejmuje on uchwyty, liczniki i
		inne informacje. Ta struktura rozronie si wraz z dodawaniem nowych obiektw
		do zestawu. Niektre skadowe s specyficzne dla jednego lub dwh obiektw.
		Struktura jest bardziej oglna ni to potrzebne w niektrych
		specyficznych przykadach.
	4.	Dla sekcji krytycznych uywane s muteksy. Mona je zastpi obiektami
		CRITICAL_SECTION, jeli s one uywane wycznie wewntrz procesu.
		Jeszcze lepiej, jeli obiekt nie jest nazwany. Wtedy wiadomo, e jest uywany
		tylko w procesie, dlatego mona wybra technik w czasie tworzenia obiektu.
		JEDNAK powoduje to utrat moliwoci korzystania z limitu czasu ORAZ 
		moliwo korzystania z nazwy i obsugi wielu procesw. 
	5.	Zasymulowane semafory mona nazywa, a tym samym i dzieli midzy procesami.
	6.	Semafory z systemu Win32 nie s uywane, dlatego ta implementacja (z pustymi nazwami)
		dziaa w systemie Windows CE.
	7.	Implementacja ilustruje kilka ciekawych aspektw synchronizacji. Niektre z nich s 
		specyficzne dla systemw Win32, a inne - oglne. Wspomniano o tym w odpowiednich
		komentarzach.
	8.	Obiekty maj odpowiednik dla funkcji WaitForSingleObject. Nie ma jednak odpowiednika
		dla funkcji WaitForMultipleObjects, poniewa opracowanie wydajnej techniki
		dziaajcej poza jdrem jest tu bardzo trudne, jeli nie niemoliwe.
*/

#include "Everything.h"
#include "Synchronize.h"

static SYNCHHANDLE CleanUp (SYNCHHANDLE);
static SYNCHHANDLE AllocSynchHandle (LPCTSTR, LPTSTR, LPTSTR, LPTSTR, LPBOOL);
/*********************************************
	Rodzina funkcji SEMAFORA Z OCZEKIWANIEM NA WIELE JEDNOSTEK.
	Takie semafory pozwalaj oczekiwa wtkowi atomowo na kilka jednostek.
	DOMYLNIE oczekiwanie odbywa si w trybie "pierwszy moliwy do obsuenia".
	Oznacza to, e wtek z moliwym do obsuenia daniem zostanie zwolniony,
	nawet jeli ma on (niezalenie od priorytetw) zalege danie, ktrego
	nie mona obecnie obsuy.

	Opcjonalne podejcie, ktre mona wczy w czasie tworzenia semafora, to
	"pierwszy przyszed-pierwszy obsuony". Wtek z daniem wikszym ni liczba
	dostpnych jednostek zablokuje wszystkie nastpne wtki - nawet te z daniami moliwymi
	do obsugi lub wyszym priorytetem. Rozwizanie z ksiki jest oparte
	na tym podejciu, jednak tu zaimplementowano je w ramach 
	oglnej platformy.
*/

SYNCHHANDLE CreateSemaphoreMW (

	LPSECURITY_ATTRIBUTES  lpSemaphoreAttributes,	/* Wskanik na atrybuty zabezpiecze. */
    LONG  lInitialCount,	/* Pocztkowa liczba jednostek. */
    LONG  lMaximumCount,	/* Maksymalna liczba jednostek. */
	BOOL  fFirstComeFirstServed,	/*	Domylnie obowizuje tryb 
	                                    "pierwszy moliwy do obsuenia". */
    LPCTSTR  lpName )

/*	Tworzenie semafora z oczekiwaniem na wiele jednostek. 
	Wymaga licznika, muteksu do ochrony stanu semafora i zdarzenia z 
	automatycznym zerowaniem.

	Oto reguy, ktre musz by zawsze przestrzegane w zwizku ze zdarzeniem i muteksem
	(naruszenie tych regu przez funkcje semafora prawie na pewno
	doprowadzi do bdw):
		1.	aden wtek nie moe zgosi ani wyzerowa zdarzenia, a take uzyska
			dostpu do jakiejkolwiek czci struktury SYNCHHANDLE bez wczeniejszego
			zajcia muteksu.
			JEDNAK wtek moe oczekiwa na zdarzenie bez zajmowania muteksu
			(jest to konieczne, poniewa inaczej zdarzenie moe nigdy nie zosta ustawione).
			Jest to model zmiennej warunkowej ze zgaszaniem (zdarzenie z automatycznym
			zerowaniem i funkcja SetEvent).
		2.	Zdarzenie przechodzi w stan zgoszony tylko wtedy, jeil licznik ma
			warto wiksz od zera. Aby to zagwarantowa, licznik naley sprawdza
			po kadym jego zmniejszeniu.
		3.	Licznik semafora zawsze ma warto >= 0 i <= od maksymalnej wartoci.
*/
/*  DODATEK - uyj funkcji SignalObjectAndWait, aby zastpi limity czasu
    przy oczekiwaniu na zdarzenie. */
{
	SYNCHHANDLE hSynch = NULL, hShare = NULL;
	TCHAR MutexName [MAX_PATH] = _T(""), EventName [MAX_PATH] = _T(""),
		MutexaName[MAX_PATH] = _T("");
	BOOL NewObject;

	if (lInitialCount > lMaximumCount || lMaximumCount < 0 || lInitialCount < 0) {
		/* Ze parametry. */
		SetLastError (SYNCH_ERROR);
		return NULL;
	}

	hSynch = AllocSynchHandle (lpName, MutexName, EventName, MutexaName, &NewObject);
	if (hSynch == NULL) {
		SetLastError (SYNCH_ERROR);
		return NULL;
	}

	/*	Tworzenie uchwytw obiektu. Zawsze s tworzone w uchwycie lokalnym procesu. */
	
	hSynch->hMutex = CreateMutex (lpSemaphoreAttributes, FALSE, (LPCTSTR)MutexName);

	/*	Tworzenie zdarzenia. Jest ono pocztkowo zgoszone tylko wtedy, jeli
		pocztkowa warto licznika > 0. */
	hSynch->hEvent = CreateEvent (lpSemaphoreAttributes, FALSE /*  Automatyczne zerowanie. */, 
		lInitialCount > 0, (LPCTSTR)EventName);

	hSynch->hMutexa = NULL;
	hSynch->dwFlags = 6; /* Zdarzenie i muteks, jednak bez muteksu pomocniczego
		(chyba e jest to semafor z oczekiwaniem na wiele jednostek dziaajcy w 
		trybie pierwszy przyszed-pierwszy obsuony). */
	if (fFirstComeFirstServed) {
		hSynch->hMutexa = CreateMutex (lpSemaphoreAttributes, FALSE, (LPCTSTR)MutexaName);
		hSynch->dwFlags = 7; /*  Wszystkie trzy obiekty zostay utworzone. */
	}

	/*	Ustawianie stanu obiektu - zawsze w uchwycie lokalnym i 
		w uchwycie wspuytkowanym (jeli taki istnieje). */

	hSynch->MaxCount = lMaximumCount;
	hSynch->CurCount = lInitialCount;  /* Warto lokalna nie jest zachowywana. */
	hSynch->fFirstComeFirstServed = fFirstComeFirstServed;
	_tcscpy (hSynch->lpName, lpName);

	hShare = hSynch->SharedHandle;
	if (NewObject && hShare != NULL ) { 		
		hShare->MaxCount = lMaximumCount;
		hShare->CurCount = lInitialCount;  /* Warto lokalna nie jest zachowywana. */
		hShare->fFirstComeFirstServed = fFirstComeFirstServed;
		_tcscpy (hShare->lpName, lpName);
	}

	/*  Zwraca uchwyt lub - jeli wystpi bd - warto null po zamkniciu
		wszystkich otwartych uchwytw i zwolnieniu zaalokowanej pamici. */
	return CleanUp (hSynch);	
}


BOOL ReleaseSemaphoreMW (SYNCHHANDLE hSemMW, LONG cReleaseCount, LPLONG lpPreviousCount)
/*	Odpowiednik funkcji ReleaseSemaphore z moliwoci oczekiwania na wiele jednostek. */
{
	BOOL Result = TRUE;
	SYNCHHANDLE hState;

	/*	Uzyskiwanie dostpu do obiektu w celu zagwarantowania, e parametr cReleaseCount nie 
		spowodowa przekroczenia maksymalnej wartoci licznika. */

	/*	Stan jest przechowywany lokalnie, jeli obiekt nie ma nazwy, a w 
		pamici wspuytkowanej, jeeli obiekt jest nazwany. */

	hState = (hSemMW->SharedHandle == NULL) ? hSemMW : hSemMW->SharedHandle;
	__try {
		WaitForSingleObject (hSemMW->hMutex, INFINITE);
		*lpPreviousCount = hState->CurCount;
		if (hState->CurCount + cReleaseCount > hState->MaxCount || cReleaseCount <= 0) {
			SetLastError (SYNCH_ERROR);
			Result = FALSE;
			_leave;
		}
		hState->CurCount += cReleaseCount;

		/*	Ustawianie zdarzenia z automatycznym zerowaniem. Wszystkie wtki oczekujce obecnie na 
			zdarzenie s zwalniane, a zdarzenie zostaje wyzerowane. */
		SetEvent (hSemMW->hEvent);
	}
	__finally {
		ReleaseMutex (hSemMW->hMutex);
	}
	return Result;
}

DWORD WaitForSemaphoreMW (SYNCHHANDLE hSemMW, LONG cSemRequest, DWORD dwMilliseconds)
/*	Odpowiednik funkcji WaitForSingleObject z moliwoci oczekiwania na wiele jednostek.
	Drugi parametr to liczba danych jednostek, a oczekiwanie odbywa si (w zalenoci 
	od opcji wybranej w czasie tworzenia) w trybie "pierwszy przyszed-pierwszy obsuony" lub
	"pierwszy dostpny". */
{
	DWORD WaitResult;
	SYNCHHANDLE hState;

	/*	Stan jest przechowywany lokalnie, jeli obiekt nie ma nazwy, a w 
		pamici wspuytkowanej, jeeli obiekt jest nazwany. */

	hState = (hSemMW->SharedHandle == NULL) ? hSemMW : hSemMW->SharedHandle;

	if (cSemRequest <= 0 || cSemRequest > hState->MaxCount) {
		SetLastError (SYNCH_ERROR);
		return WAIT_FAILED;
	}

	/*	Jeli semafor dziaa w trybie "pierwszy przyszed-pierwszy obsuony", wtek zajmuje
		muteks pomocniczy w celu zablokowania wszystkich pozostaych wtkw.
		Trzeba to zrobi PRZED oczekiwaniem na muteks chronicy stan semafora. */
	if (hSemMW->fFirstComeFirstServed) {
		WaitResult = WaitForSingleObject (hSemMW->hMutexa, dwMilliseconds);
		if (WaitResult != WAIT_OBJECT_0 && WaitResult != WAIT_ABANDONED_0) return WaitResult;
	}

	WaitResult = WaitForSingleObject (hSemMW->hMutex, dwMilliseconds);
	if (WaitResult != WAIT_OBJECT_0 && WaitResult != WAIT_ABANDONED_0) return WaitResult;

	while (hState->CurCount < cSemRequest) { 
		/*	Licznik ma warto mniejsz ni dana liczba jednostek. Wtek musi 
			oczekiwa na zdarzenie (ktre - zgodnie z reguami - jest obecnie wyzerowane),
			aby zasoby semafora stay si dostpne. Najpierw oczywicie trzeba zwolni muteks,
			aby inny wtek mg ustawi zdarzenie.
		*/
		ReleaseMutex (hSemMW->hMutex);
		/*	Oczekiwanie na zdarzenie okrelajce, e inny wtek zwikszy warto
			licznika semafora.
			Jest to zdarzenie z automatycznym zerowaniem i zgaszane za pomoc funkcji 
			SetEvent (a niePulseEvent), dlatego tylko jeden wtek (obecnie oczekujcy lub
			oczekujcy w przyszoci) zostanie zwolniony.
			Inna moliwo to uycie zdarzenia z rcznym zerowaniem i zerowanie
			zdarzenia bezporednio po oczekiwaniu.
		*/
		WaitResult = WaitForSingleObject (hSemMW->hEvent, dwMilliseconds);
		if (WaitResult != WAIT_OBJECT_0) return WaitResult;

		/*	Zajmowanie semafora, aby mona ponownie sprawdzi jego stan w grnej
			czci ptli. Warto zauway, e w tym samym czasie oczekiwa mog
			take inne wtki, ale w danym momencie tylko jeden z nich moe
			sprawdza licznik semafora. */
		WaitResult = WaitForSingleObject (hSemMW->hMutex, dwMilliseconds);
		if (WaitResult != WAIT_OBJECT_0 && WaitResult != WAIT_ABANDONED_0) return WaitResult;
	}
	
	/*	hState->CurCount >= cSemRequest (danie mona speni), a
		dany wtek zajmuje muteks. */
	hState->CurCount -= cSemRequest;
	if (hState->CurCount > 0) SetEvent (hSemMW->hEvent);
	ReleaseMutex (hSemMW->hMutex);
	if (hSemMW->fFirstComeFirstServed) ReleaseMutex (hSemMW->hMutexa);
	
	return WaitResult;

}


BOOL CloseSynchHandle (SYNCHHANDLE hSynch)
/*	Zamykanie uchwytu synchronizacji. 
	Usprawnienie: dodaj sprawdzanie przed zamkniciem uchwytu, czy jest on prawidowy. */
{
	BOOL Result = TRUE;

	if (hSynch->hEvent  != NULL) Result = Result && CloseHandle (hSynch->hEvent);
	if (hSynch->hMutex  != NULL) Result = Result && CloseHandle (hSynch->hMutex);
	if (hSynch->hMutexa != NULL) Result = Result && CloseHandle (hSynch->hMutexa);
	if (hSynch->SharedHandle != NULL) 
		InterlockedDecrement (&(hSynch->SharedHandle->RefCount));
	if (hSynch->ViewOfFile != NULL) UnmapViewOfFile (hSynch->ViewOfFile);
	HeapFree (GetProcessHeap (), 0, hSynch);
	return (Result);
}


static SYNCHHANDLE CleanUp (SYNCHHANDLE hSynch)
{	/*	Przygotowanie do zwrcenia sterowania po utworzeniu uchwytu synchronizacji.
		Jeli wystpi bd, naley zwolni wszystkie zaalokowane zasoby.
		Zmienna "Flags" okrela, ktre obiekty systemu Win32 s potrzebne
		w uchwycie synchronizacji. */
	BOOL ok = TRUE;
	DWORD Flags;

	if (hSynch == NULL)	return NULL;
	Flags = hSynch->dwFlags;
	if ((Flags & 4) == 1 && hSynch->hEvent ==  NULL) ok = FALSE;
	if ((Flags & 2) == 1 && hSynch->hMutex ==  NULL) ok = FALSE;
	if ((Flags & 1) == 1 && hSynch->hMutexa == NULL) ok = FALSE;
	if (!ok) {
		CloseSynchHandle (hSynch);
		return NULL;
	}

	/*	Wszystko zadziaao. */
	return hSynch;
}


static SYNCHHANDLE AllocSynchHandle (LPCTSTR lpName, 
	LPTSTR MutexName, LPTSTR EventName, LPTSTR MutexaName,
	LPBOOL pfNewObject)
/*	Alokowanie pamici dla uchwytu synchronizacji. Uchwyty obiektw bez nazwy
	s tworzone bezporedni na stercie procesu, natomiast obiekty nazwane maj
	dwa uchwyty - jeden zaalokowany lokalnie na uchwyty (lokalne dla procesu)
	i poza procesem we wspuytkowanej pamici odwzorowanej na plik
	stronicowy (na stan wspuytkowanego obiektu).
	Ponadto naley utworzy nazwy trzech obiektw wewntrznych i okreli,
	czy pojawi si nowy obiekt, ktry naley zainicjowa. */
{
	HANDLE hSynchDB;	/*  Muteks do ochrony caej bazy danych obiektu synchronizacji. */
	HANDLE hMap, hFile;
		/*	Uchwyt wspuytkowanej pamici i pliku do 
   		    zarzdzania obiektami synchronizacji. */
	BOOL FirstTime;
	SYNCHHANDLE pView, pNew = NULL, pFirstFree, hLocal;

	hLocal = HeapAlloc (GetProcessHeap(), HEAP_ZERO_MEMORY, SYNCH_HANDLE_SIZE);
	if (hLocal == NULL) return NULL;
	*pfNewObject = TRUE;
	if (lpName == NULL || _tcscmp (lpName, _T("")) == 0) {
		/*  Obiekt nie ma nazwy. */
		hLocal->SharedHandle = NULL;
		return hLocal;
	}

	/*	Obiekt ma nazw. Naley utworzy nazwy dla obiektw wewntrznych. */
	_stprintf (MutexName,  _T("%s%s"), lpName, _T(".mtx"));
	_stprintf (EventName,  _T("%s%s"), lpName, _T(".evt"));
	_stprintf (MutexaName, _T("%s%s"), lpName, _T(".mtxa"));
	
	/*	Blokowanie dostpu do bazy danych obiektu synchronizacji w celu uniemoliwienia
		innym wtkom dostpu i jednoczesnego utworzenia innego obiektu o tej samej nazwie.
		Wszystkie procesy i wtki korzystaj z tego samego muteksu o znanej nazwie. */
	
	hSynchDB = CreateMutex (NULL, FALSE, SYNCH_OBJECT_MUTEX);
	WaitForSingleObject (hSynchDB, INFINITE);

	/*  Dostp do pamici wspuytkowanej, gdzie przechowywane s obiekty synchronizacji.
		Jednak koniecznie trzeba najpierw sprawdzi, czy jest to pierwsze tworzenie 
		danego obiektu. Mona wtedy zainicjowa wspuytkowan tablic odwzorowan
		w pamici. 
		Sprawdzanie odbywa si za pomoc funkcji OpenFileMapping.  */
	
	__try {
		hMap = OpenFileMapping (FILE_MAP_WRITE, FALSE, SYNCH_FM_NAME);
		FirstTime = (hMap == NULL);
		if (FirstTime) /* Do naprawy!! Uchwyt hFile nie zosta utworzony. */
			hMap = CreateFileMapping (hFile, NULL, PAGE_READWRITE, 0, SIZE_SYNCH_DB, SYNCH_FM_NAME);
		if (hMap == NULL) _leave;
		pView = (SYNCHHANDLE)MapViewOfFile (hMap, FILE_MAP_WRITE, 0, 0, SIZE_SYNCH_DB);
		if (pView == NULL) _leave;
		if (FirstTime) memset (pView, 0, SIZE_SYNCH_DB);

		/*	Wyszukiwanie w celu sprawdzenia, czy obiekt o danej nazwie ju istnieje.
			Wpis to odwzorowany rekord uywany do operacji pomocniczych, kiedy
			jest potrzebny. Puste miejsce jest wykrywane, kiedy licznik odwoa
			ma warto 0. */
		pFirstFree = NULL;
		for (pNew = pView+1; pNew < pView + SYNCH_MAX_NUMBER; pNew++) {
			if ((pFirstFree == NULL) && (pNew->RefCount <= 0)) pFirstFree = pNew;
			if ((pNew->lpName != NULL) 
				&& _tcscmp (pNew->lpName, lpName) == 0) break; /* Nazwa istnieje. */
		}

		if (pNew < pView + SYNCH_MAX_NUMBER) { /* Nazwa istnieje. */
			*pfNewObject = FALSE;
		} else if (pFirstFree != NULL) {
			/*	Nazwa nie istnieje, jednak znaleziono puste miejsce. */
			*pfNewObject = TRUE;
			pNew = pFirstFree;
		} else { /* Nazwa nie istnieje i nie ma wolnego miejsca. */
			pNew = NULL;
			*pfNewObject = TRUE;
		}
	}

	__finally {
		if (pNew != NULL) hLocal->ViewOfFile = pView;
		hLocal->SharedHandle = pNew;
		if (hLocal->SharedHandle != NULL) 
			InterlockedIncrement (&(hLocal->SharedHandle->RefCount));
		ReleaseMutex (hSynchDB);
	}
	return hLocal;
}

