﻿// © 2011 IDesign Inc.
// Pytania? Komentarze? Odwiedź
// http://www.idesign.net

using System;
using System.ServiceModel;

[ServiceContract]
interface IPasswordManager
{
   /// <summary>
   /// Zwraca wartość określającą, czy bieżący dostawca członkostwa umożliwia zerowanie hasłe przez samych użytkowników. Jeśli tak, zwraca wartość true; w przeciwnym razie zwraca wartość false.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   bool EnablePasswordReset(string application);

   /// <summary>
   /// Zwraca wartość określającą, czy bieżący dostawca członkostwa umożliwia uzyskiwanie haseł przez użytkowników. Jeśli tak, zwraca wartość true; w przeciwnym razie zwraca wartość false.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   bool EnablePasswordRetrieval(string application);

   /// <summary>
   /// Generuje hasło na podstawie wyrażenia regularnego i pozostałych ograniczeń.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   string GeneratePassword(string application,int length,int numberOfNonAlphanumericCharacters);

   /// <summary>
   /// Zwraca maksymalną liczbę prób logowania, po których konto użytkownika jest blokowane.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   int GetMaxInvalidPasswordAttempts(string application);

   /// <summary>
   /// Zwraca minimalną wymaganą liczbę znaków niealfanumerycznych.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   int GetMinRequiredNonAlphanumericCharacters(string application);

   /// <summary>
   /// Zwraca minimalną wymaganą długość hasła.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   int GetMinRequiredPasswordLength(string application);

   /// <summary>
   /// Zwraca czas (wyrażony w minutach), w którym użytkownik może podjąć nieudane próby logowania lub odpowiadania na zdefiniowane pytanie przed zablokowaniem konta.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   int GetPasswordAttemptWindow(string application);

   /// <summary>
   /// Zwraca wyrażenie regularne dla hasła.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   string GetPasswordStrengthRegularExpression(string application);

   /// <summary>
   /// Zwraca wartość określającą, czy domyślny dostawca członkostwa wymaga od użytkownika odpowiedzi na zdefiniowane pytanie przed wycerowaniem lub odzyskaniem hasła.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   bool RequiresQuestionAndAnswer(string application);

   /// <summary>
   /// Zeruje hasło użytkownika, zastępując je nowym, automatycznie wygenerowanym hasłem. Zwraca nowe hasło. Metoda może być stosowana, pod warunkiem że zapisy w pliku konfiguracyjnym nie wymagają podania odpowiedzi na zdefiniowane pytanie.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   string ResetPassword(string application,string userName);

   /// <summary>
   /// Zeruje hasło użytkownika, zastępując je nowym, automatycznie wygenerowanym hasłem. Zwraca nowe hasło. Wymaga przekazania odpowiedzi na zdefiniowane pytanie.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   string ResetPasswordWithQuestionAndAnswer(string application,string userName,string passwordAnswer);

   /// <summary>
   /// Zwraca hasło użytkownika przechowywane w pamięci członkostwa. Wymaga przekazania odpowiedzi na zdefiniowane pytanie.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   string GetPassword(string application,string userName,string passwordAnswer);

   /// <summary>
   /// Zwraca pytanie zabezpieczające dla danego użytkownika.
   /// </summary>
   [TransactionFlow(TransactionFlowOption.Allowed)]
   [OperationContract]
   string GetPasswordQuestion(string application,string userName);

   /// <summary>
   /// Aktualizuje hasło użytkownika w pamięci członkostwa. Zwraca wartość true, jeśli aktualizacja zakończyła się pomyślnie; w przeciwnym razie zwraca wartość false. Metoda może być stosowana, pod warunkiem że zapisy w pliku konfiguracyjnym nie wymagają podania odpowiedzi na zdefiniowane pytanie.
   /// </summary>
   [TransactionFlow(TransactionFlowOption.Allowed)]
   [OperationContract]
   void ChangePassword(string application,string userName,string newPassword);

   /// <summary>
   /// Aktualizuje hasło użytkownika w pamięci członkostwa. Zwraca wartość true, jeśli aktualizacja zakończyła się pomyślnie; w przeciwnym razie zwraca wartość false.
   /// </summary>
   [OperationContract]
   [TransactionFlow(TransactionFlowOption.Allowed)]
   void ChangePasswordWithAnswer(string application,string userName,string passwordAnswer,string newPassword);
}
