#include <stdio.h>
#include <assert.h>

// Ponisze makro pozwala nam posugiwa si typem "real" uywanym 
// przez fpadd i fpsub jak typem float jzyka C.

#define asreal(x) (*((float *) &x))


// Typ "real" uywany przez fpadd i fpsub. Uyjemy 32-bitowej liczby
// cakowitej bez znaku, aby atwo byo pobra jej bity.

typedef long unsigned real;


// Kilka funkcji pomocniczych do pobierania poszczeglnych pl
// naszych 32-bitowych wartoci zmiennoprzecinkowych:
//
// extractExponent-
//
// Pobiera bity 23..31 (zawierajce wykadnik) oraz odejmuje
// 127, aby zamieni wykadnik z przesuniciem-127 na liczb
// ze znakiem.

inline int extractExponent( real from )
{
    return ((from >> 23) & 0xff) - 127;
}


// extractMantissa-
//
// Pobiera bity 0..22 z liczby i ustawia bit 23 na jeden (bit stay).
// Oczywicie, o ile warto nie jest zerem, bo wtedy funkcja zwraca
// po prostu zero.

inline int extractMantissa( real from )
{
    if( from == 0 ) return 0;
    return ((from & 0xFFFFFF) | 0x800000 );
}

// extractSign-
//
// Zwraca znak mantysy.

inline int extractSign( real from )
{
    return( from >> 31);
}


// packFP:
//
// Pakuje znak, wykadnik i mantys do 32-bitowej liczby 
// "rzeczywistej". Obsuguje wartoci znormalizowane, zdenormalizowane
// oraz zero, ale nie obsuguje wartoci NaN ani nieskoczonoci.

inline real packFP( int sign, int exponent, int mantissa )
{
   return 
        (real)
        ( 
                (sign << 31) 
            |   ((exponent + 127) << 23)  
            |   (mantissa & 0x7fffff)
        );
}


// shiftAndRound:
//
// Przesuwa mantys w prawo o podan liczb bitw.
// Wynik zaokrgla zgodnie z wymogami standardu IEEE, czyli:
//
//  Jeli wysuwane bity maj warto wiksz ni poowa bita, ktry po caej
//   operacji bdzie bitem najmodszym, warto ma by zwikszona przez 
//   dodanie do tego najmodszego bita jedynki.
//  Jeli wysuwane bity maj (po denormalizacji) warto mniejsz ni poowa
//   bita, ktry po caej operacji bdzie bitem najmodszym, warto ma by
//   obcita (czyli po prostu zostawiamy j, jak jest).
//  Jeli wysuwane bity stanowi dokadnie poow bita, ktry po caej
//   operacji bdzie najmodszy, musimy zaokrgli liczb do wartoci 
//   z zerem na najmodszym bicie (jeli bya tam jedynka, zaokrglamy w gr;
//   jeli byo zero, zostawiamy je bez zmian).

void shiftAndRound( long unsigned *valToShift, int bitsToShift )
{
    // Maski uywane do maskowania bitw w celu sprawdzenia ostatniego bita.
    
    static unsigned masks[24] =
    {
        0, 1, 3, 7, 0xf, 0x1f, 0x3f, 0x7f, 
        0xff, 0x1ff, 0x3ff, 0x7ff, 0xfff, 0x1fff, 0x3fff, 0x7fff,
        0xffff, 0x1ffff, 0x3ffff, 0x7ffff, 0xfffff, 0x1fffff, 0x3fffff, 0x7fffff
    };
        
    // HOmasks - maskuj najstarszy bit wartoci maskowanej przez masks.
    
    static unsigned HOmasks[24] =
    {
        0, 
        1, 2, 4, 0x8, 0x10, 0x20, 0x40, 0x80, 
        0x100, 0x200, 0x400, 0x800, 0x1000, 0x2000, 0x4000, 0x8000, 
        0x10000, 0x20000, 0x40000, 0x80000, 0x100000, 0x200000, 0x400000
    };
        
    // shiftedOut- warto wysuwana poza mantys w trakcie denormalizacji
    // (podczas zaokrglania denormalizowanej wartoci).
    
    int             shiftedOut;
    
    assert( bitsToShift <= 23 );
    
    
    // Teraz najpierw pobieramy bity do wysunicia (aby mc sprawdzi, jakie
    // zaokrglenie naley zastosowa).
    
    shiftedOut = *valToShift & masks[ bitsToShift ];
    
    // Przesunicie wartoci w prawo o wskazan liczb bitw:
    
    *valToShift = *valToShift >> bitsToShift;
    
    // Jeli trzeba, warto zaokrglamy:
    
    if( shiftedOut > HOmasks[ bitsToShift ] )
    {
        // Jeli wysunite bity s wiksze od poowy wartoci nowego
        // najmodszego bita, warto zaokrglamy w gr.
       
        *valToShift = *valToShift + 1;
    }
    else if( shiftedOut == HOmasks[ bitsToShift ] )
    {
        // Jeli wysuwane bity to dokadnie poowa wartoci nowego najmodszego
        // bita, zaokrglamy do najbliszej liczby z zerowym najmodszym bitem.
       
        *valToShift = *valToShift + ((*valToShift & 1) == 1);
    }
    // w przeciwnym razie zaokrglamy w d, do poprzedniej wartoci. Uzyskana
    // liczba jest ju obcita (czyli zaokrglona w d), zatem nie musimy
    // ju niczego robi.
}

    
       
       
    


// fpadd:
//
//   Wylicza:
//      dest = left + right
// gdzie wszystkie operandy to liczby "rzeczywiste" (tak naprawd, s to
// 32-bitowe liczby zmiennoprzecinkowe).

void fpadd( real left, real right, real *dest )
{   
    // Ponisze zmienne to pola lewego argumentu:
    
    int             Lexponent;
    long unsigned   Lmantissa;
    int             Lsign;
    
    // Ponisze zmienne to pola prawego argumentu:
    
    int             Rexponent;
    long unsigned   Rmantissa;
    int             Rsign;
    
    // Ponisze zmienne to pola wyniku:
    
    int   Dexponent;
    long  unsigned Dmantissa;
    int   Dsign;

    
    // Pobieramy pola, aby uatwi sobie ich przetwarzanie:
    
    Lexponent = extractExponent( left );
    Lmantissa = extractMantissa( left );
    Lsign     = extractSign( left );
    
    Rexponent = extractExponent( right );
    Rmantissa = extractMantissa( right );
    Rsign     = extractSign( right );
    
    
    // Obsuga argumentw specjalnych (nieskoczono, NaN)
    
    if( Lexponent == 127 )
    {
        if( Lmantissa == 0 )
        {
	    // Jeli lewy argument jest nieskoczony, wynik zaley
	    // od wartoci prawego argumentu.
            
            if( Rexponent == 127 )
            {
		// Jeli wykadnik ma same jedynki (po odkodowaniu 127),
		// mantysa decyduje, czy mamy nieskoczono (mantysa
		// zerowa), QNaN (mantysa = 0x800000), czy SNaN (niezerowa
                // mantysa, nierwna 0x800000). 
                
                if( Rmantissa == 0 )  // Czyby nieskoczono?
                {
                    // nieskoczono + nieskoczono = nieskoczono
                    // -nieskoczono - nieskoczono = -nieskoczono
                    // -nieskoczono + nieskoczono = NaN
                    // nieskoczono - nieskoczono = NaN
                    
                    if( Lsign == Rsign )
                    {
                        *dest = right;
                    }
                    else
                    {
                        *dest = 0x7fC00000;  // +QNaN
                    }
                }
                else  // prawa mantysa nie jest zerem, wic to NaN
                {
                    *dest = right;  // prawy argument to NaN - propagacja
                }
            }
            
        }
        else // lewa mantysa to nie zero, lewy wykadnik to same jedynki
        {
	    //  Jeli lewy operand to NaN, wynikiem bdzie taki sam NaN.
            
            *dest = left;
        }
        
        // Mamy ju wynik, wic go zwracamy.
        
        return;
        
    }
    else if( Rexponent == 127 )
    {
	// Moliwe s dwa przypadki: prawy argument to NaN (wtedy propagujemy
	// to NaN niezalenie od wartoci lewego argumentu) lub jest to
	// +/- nieskoczono. Lewy argument to zwyka liczba, wic i tak
	// dojdzie do propagacji nieskoczonoci, gdy dowolna liczba
	// skoczona i nieskoczono daj nieskoczono.
        
        *dest = right;  // prawy argument to NaN - propagujemy go
        return;
    }
    


    // Teraz ju mamy do czynienia ze zwykymi liczbami zmiennoprzecinkowymi.
    // Dodajmy je do siebie. Najpierw "denormalizujemy" jeden z argumentw
    // (o ile wykadniki nie s takie same).
    // 
    // Algorytm: wybieramy warto z mniejszym wykadnikiem. Przesuwamy
    // jej mantys w prawo o liczb bitw rwn rnicy midzy oboma
    // wykadnikami.
    
    if( Rexponent > Lexponent )
    {
        shiftAndRound( &Lmantissa, (Rexponent - Lexponent));
        Dexponent = Rexponent;
    }
    else if( Rexponent < Lexponent )
    {
        shiftAndRound( &Rmantissa, (Lexponent - Rexponent));
        Dexponent = Lexponent;
    }
    
    // Teraz dodajemy mantysy. Jest tu pewna puapka: jeli znaki s przeciwne,
    // musimy wartoci od siebie odj (jako e format zmiennoprzecinkowy
    // stosuje dopenienie do jedynki, odejmujemy wiksz mantys od mniejszej, 
    // znak wyniku ustawiamy zalenie od znakw pierwotnych i tego, ktra
    // mantysa bya wiksza).

    if( Rsign ^ Lsign )
    {
	// Znaki s rwne, wic musimy wartoci od siebie odj.
        
        if( Lmantissa > Rmantissa )
        {
	    // Lewy argument jest wikszy, wic wynik uzyska jego znak.
       
            Dmantissa = Lmantissa - Rmantissa;
            Dsign = Lsign;
        }
        else
        {
	    // Prawy argument jest wikszy, wic wynik uzyska jego znak.
       
            Dmantissa = Rmantissa - Lmantissa;
            Dsign = Rsign;
        }
        
    }
    else
    {
        // Znaki s takie same, wic dodajemy wartoci do siebie.
        
        Dsign = Lsign;
        Dmantissa = Lmantissa + Rmantissa;
    }
    
    // Normalizujemy wynik.
    //
    // Zauwamy, e podczas dodawania i odejmowania moliwe jest przepenienie
    // na jednym bicie (jeli bdzie miao ono miejsce, przesuwamy mantys 
    // w prawo o jedno miejsce i odpowiednio zwikszamy wykadnik).
    // zauwamy, e w razie wystpienia przepenienia podczas zwikszania
    // wykadnika, uzyskamy nieskoczono (ktra jest oznaczana przez 
    // wykadnik $FF).

   if( Dmantissa >= 0x1000000 )
    {
	// Przy dodawaniu i odejmowanie nigdy nie powstanie wicej ni
	// jeden dodatkowy bit. Dziki cechom uywanego przez nas formatu
	// zmiennoprzecinkowego, maksymalna warto mantysy uzyskana
	// przy dodawaniu i odejmowaniu to 0x1fffffe.  Wobec tego, podczas
	// zaokrglania nie wystpi przepenienie na 25-m bicie.
        
        shiftAndRound( &Dmantissa, 1 ); // Przesuwamy wynik na 24 bity.
        ++Dexponent;                    // Przesuwanie do dzielenie przez dwa,
                                        //  wic zwikszanie wykadnika,
					// oznaczajce mnoenie przez dwa,
					// stanowi jego przeciw-dziaanie.
    }
    else
    {
	// Jeli najstarszy bit jest wyzerowany, normalizujemy wynik 
	// przesuwajc bity wyniku i jednoczenie zmniejszajc wykadnik.
	// Zero bdziemy traktowa jako przypadek szczeglny; zreszt 
	// rzadko si ono zdarza.
        
        if( Dmantissa != 0 )
        {
            
	    // W ptli mnoymy mantys przez dwa (przesuwajc j w lewo),
	    // nastpnie ca liczb dzielimy przez dwa (zmniejszajc
	    // wykadnik). Dzieje si tak tak dugo, a mantysa wyniku
	    // bdzie miaa warto -127 (zero w formie nadmiaru-127).
	    // Jeli wykadnik wyniku zredukuje si do -128, mamy ju liczb
	    // zdenormalizowan i moemy zaprzesta korekt.
                                                         
            while( (Dmantissa < 0x800000) && (Dexponent > -127 ))
            {
                Dmantissa = Dmantissa << 1;
                --Dexponent;
            }
            
        }
        else
        {
	    // Jeli mantysa si wyzerowaa, zerujemy te wszystko inne.
            
            Dsign = 0;
            Dexponent = 0;
        }
    }
    
    // Odtwarzamy wynik i pakujemy go:

    *dest = packFP( Dsign, Dexponent, Dmantissa );
    
}
                                                         
                                                         
                                                         
                                                         

// We could repeat all the code above for subtract, or we could do subtraction
// the easy way by flipping one of the signs and doing an add.  Guess which way
// this is :-)

void fpsub( real left, real right, real *dest )
{

    right = right ^ 0x80000000;
    fpadd( left, right, dest );
}


// Prosta funkcja main robica elementarne testy fpadd i fpsub.

int main( int argc, char **argv )
{
    real l, r, d;
    
    asreal(l) = 1.0;

    asreal(r) = 2.0;
    
    fpadd( l, r, &d );
    printf( "dest = %x\n", d );
    printf( "dest = %12E\n", asreal( d ));
    
    l = d;
    asreal(r) = 4.0;
    fpsub( l, r, &d );
    printf( "dest2 = %x\n", d );
    printf( "dest2 = %12E\n", asreal( d ));
}
