
// Biblioteka standardowa
#include <iostream>
#include <string>

// Biblioteka JSON
#include <nlohmann/json.hpp>
using json = nlohmann::json;

// Narzdzia
#include "expected.h"
#include "mtry.h"
#include "trim.h"

// Nasza implementacja strumienia reaktywnego
#include "filter.h"
#include "join.h"
#include "sink.h"
#include "transform.h"
#include "values.h"

// Implementacja usugi
#include "service.h"

/**
 * Dla funkcji, ktre zwracaj obiekty json. Gdy ich wykonanie
 * moe si nie powie, zwracaj wyjtek.
 */
using expected_json = expected<json, std::exception_ptr>;

/**
 * Podstawowe dane zakadki. Struktura przechowuje adres URL i nazw.
 */
struct bookmark_t {
    std::string url;
    std::string text;
};

/**
 * Tworzy acuch na podstawie zakadki w nastpujcym formacie:
 * [tekst](url)
 */
std::string to_string(const bookmark_t& page)
{
    return "[" + page.text + "](" + page.url + ")";
}

/**
 * Zapisuje informacje zwizane z zakadk w nastpujcym formacie:
 * [tekst](url)
 */
std::ostream& operator<<(std::ostream& out, const bookmark_t& page)
{
    return out << "[" << page.text << "](" << page.url << ")";
}

/**
 * Typ, ktry zawiera zakadk lub bd (wyjtek).
 */
using expected_bookmark = expected<bookmark_t, std::exception_ptr>;

/**
 * Prbuje odczyta zakadk z obiektu JSON. W przypadku bdu
 * nastpi zwrcenie wyjtku.
 */
expected_bookmark bookmark_from_json(const json& data)
{
    return mtry([&] {
            return bookmark_t { data.at("FirstURL"), data.at("Text") };
        });
}


int main(int argc, char *argv[])
{
    using namespace reactive::operators;

    boost::asio::io_service event_loop;

    auto pipeline =
        service(event_loop)
            | transform(trim)

			// Ignorujemy komentarze i puste wiadomoci.
            | filter([] (const std::string &message) {
                return message.length() > 0 && message[0] != '#';
            })

			// Prbujemy parsowa dane wejciowe.
            | transform([] (const std::string &message) {
                return mtry([&] {
                    return json::parse(message);
                });
            })

			// Przeksztacamy wynik na zakadk.
            | transform([] (const auto& exp) {
                return mbind(exp, bookmark_from_json);
            })

            | sink([] (const auto &exp_bookmark) {
				// Jeli z jakigo powodu danie nie byo poprawne,
				// poinformuj o bdzie.
                if (!exp_bookmark) {
                    std::cerr << "BD: danie niezrozumiae.\n";
                    return;
                }

				// Jeli zakadka nie dotyczy C++, poinformuj o bdzie.
                if (exp_bookmark->text.find("C++") != std::string::npos) {
                    std::cerr << "OK: " << to_string(exp_bookmark.get()) << std::endl;
                } else {
                    std::cerr << "BD: odnonik nie dotyczy C++.\n";
                }
            });

	// Uruchomienie usugi Boost.ASIO.
    std::cerr << "Usuga dziaa...\n";
    event_loop.run();
}
