// vim: zdefiniuj parametr set undofile:
// Program: 06-string-concatenation-refs

#include <iostream>
#include <string>
#include <tuple>

template <typename... Strings>
class lazy_string_concat_helper;

template <typename LastString, typename... Strings>
class lazy_string_concat_helper<LastString,
                                Strings...> {
private:
    const LastString& data;
    lazy_string_concat_helper<Strings...> tail;

public:
    lazy_string_concat_helper(
            const LastString& data,
            lazy_string_concat_helper<Strings...> tail)
        : data(data)
        , tail(tail)
    {
    }

    int size() const
    {
        return data.size() + tail.size();
    }

    template <typename It>
    void save(It end) const
    {
        const auto begin = end - data.size();
        std::copy(data.cbegin(), data.cend(),
                  begin);
        tail.save(begin);
    }

    operator std::string() const
    {
        std::string result(size(), '\0');
        save(result.end());
        return result;
    }

    lazy_string_concat_helper<std::string,
                              LastString,
                              Strings...>
    operator+(const std::string& other) const
    {
        return lazy_string_concat_helper
               <std::string, LastString, Strings...>(
                   other,
                   *this
               );
    }
};


template <>
class lazy_string_concat_helper<> {
public:
    lazy_string_concat_helper()
    {
    }

    int size() const
    {
        return 0;
    }

    template <typename It>
    void save(It) const
    {
    }

    lazy_string_concat_helper<std::string>
    operator+(const std::string& other) const
    {
        return lazy_string_concat_helper<std::string>(
                other,
                *this
            );
    }
};


template <typename Stream, typename... Strings>
Stream& operator<<(Stream& stream,
                   const lazy_string_concat_helper<Strings...>& strings)
{
    return stream << static_cast<std::string>(strings);
}


lazy_string_concat_helper<> lazy_concat;

int main(int argc, char* argv[])
{
    // Podczas uywania deklaracji "auto" i szablonw wyrae przechowujcych referencje do wartoci,
	// ktre mog z czasem ulec zmianie, istnieje niebezpieczestwo pojawienia si nieoczekiwanych efektw ubocznych.
    std::string name = "Janina";
    std::string surname = "Nowak";

    const auto fullname =
        lazy_concat + name + " " + surname;

    name = "Jan";

    // Zmienna fullname zostaa zdefiniowana, zanim zmienna name zostaa zmodyfikowana,
    // jednak czenie acuchw zostao wykonane ju po zmianie
    std::cout << fullname << std::endl;
}
