// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm.tests;

import java.io.*;
import java.util.Iterator;

import junit.framework.TestCase;

import com.saorsa.tm.*;

public class CustomerTest extends TestCase {
  public CustomerTest(String aName) {
    super(aName);
  }

  protected void setUp() {
    dave = new Customer("Dave Astels", "542-4562", "Wolfville");
    miro = new Customer("Miro Novak", "555-1111", "Calgary");
  }

  protected void tearDown() {
    Customer.clear();
  }

  public void testEquality() {
    Customer miro2 = new Customer("Miro Novak", "555-1111", "Calgary");
    Customer miro3 = new Customer("Miro Novak", "555-1111", "Calgary");
    assertTrue(!dave.equals(miro));
    assertEquals(miro, miro);
    assertEquals(miro, miro2);
    assertEquals(miro2, miro3);
    assertEquals(miro, miro3);
    assertEquals(miro3, miro);
    assertTrue(!dave.equals(null));
    assertTrue(!dave.equals("bad type"));
  }

  public void testComparing() {
    assertTrue(dave.compareTo(miro) < 0);
    assertTrue(miro.compareTo(dave) > 0);
    assertTrue(dave.compareTo(dave) == 0);
  }

  public void testAddition() {
    Customer.addCustomer(dave);
    assertTrue(Customer.containsCustomer(dave));
    assertTrue(!Customer.containsCustomer(miro));
    Customer.addCustomer(miro);
    assertTrue(Customer.containsCustomer(miro));
  }

  public void testRemoval() {
    Customer.addCustomer(dave);
    Customer.addCustomer(miro);
    Customer.removeCustomer(dave);
    assertTrue(!Customer.containsCustomer(dave));
    assertTrue(Customer.containsCustomer(miro));
    Customer.removeCustomer(miro);
    assertTrue(!Customer.containsCustomer(miro));
  }

  public void testNamedRemoval() {
    Customer.addCustomer(dave);
    Customer.addCustomer(miro);
    Customer.removeCustomerNamed(dave.getName());
    assertTrue(!Customer.containsCustomer(dave));
    assertTrue(Customer.containsCustomer(miro));
    Customer.removeCustomerNamed(miro.getName());
    assertTrue(!Customer.containsCustomer(miro));
  }

  public void testCount() {
    assertEquals(0, Customer.numberOfCustomers());
    Customer.addCustomer(dave);
    assertEquals(1, Customer.numberOfCustomers());
    Customer.addCustomer(miro);
    assertEquals(2, Customer.numberOfCustomers());
    Customer.removeCustomer(dave);
    assertEquals(1, Customer.numberOfCustomers());
    Customer.removeCustomer(miro);
    assertEquals(0, Customer.numberOfCustomers());
  }

  public void testOrdering() {
    Customer.addCustomer(miro);
    Customer.addCustomer(dave);
    Iterator customerIterator = Customer.getAlphabeticalListOfCustomers();
    Customer previousCustomer = Customer.nullCustomer;
    while (customerIterator.hasNext()) {
      Customer thisCustomer = (Customer) customerIterator.next();
      assertTrue(thisCustomer.compareTo(previousCustomer) > 0);
      previousCustomer = thisCustomer;
    }
  }

  public void testLookup() {
    Customer.addCustomer(dave);
    Customer.addCustomer(miro);
    assertEquals(dave, Customer.findCustomerByName(dave.getName()));
    assertEquals(miro, Customer.findCustomerByName(miro.getName()));
    assertEquals(Customer.nullCustomer, Customer.findCustomerByName("Granville Miller"));
  }

  public void testLoading() {
    StringReader reader = new StringReader(customerTestData);
    Customer.load(reader);

    Customer myDave = Customer.findCustomerByName("Dave Astels");
    assertEquals("Dave Astels", myDave.getName());
    assertEquals("542-4562", myDave.getPhoneNumber());
    assertEquals("Wolfville", myDave.getAddress());

    Customer myMiro = Customer.findCustomerByName("Miro Novak");
    assertEquals("Miro Novak", myMiro.getName());
    assertEquals("555-1111", myMiro.getPhoneNumber());
    assertEquals("Calgary", myMiro.getAddress());
  }

  public void testAsXML() {
    Customer.addCustomer(miro);
    Customer.addCustomer(dave);

    try {
      StringWriter writer = new StringWriter();
      Customer.allToXML(writer);
      Sequence.clear();
      StringReader reader = new StringReader(writer.toString());
      Customer.load(reader);

      Customer myDave = Customer.findCustomerByName("Dave Astels");
      assertEquals("Dave Astels", myDave.getName());
      assertEquals("542-4562", myDave.getPhoneNumber());
      assertEquals("Wolfville", myDave.getAddress());

      Customer myMiro = Customer.findCustomerByName("Miro Novak");
      assertEquals("Miro Novak", myMiro.getName());
      assertEquals("555-1111", myMiro.getPhoneNumber());
      assertEquals("Calgary", myMiro.getAddress());

    } catch (IOException ex) {
      fail();
    }
  }

  public void testOutstandingChanges() {
    Customer.addCustomer(dave);
    ChangeTracker tracker = ChangeTracker.getTracker("customer");
    assertTrue(tracker.anyChange());

    Customer.clear();
    assertTrue(!tracker.anyChange());

    Customer aCustomer = new Customer("Dave", "111-1234", "somewhere");
    Customer.addCustomer(aCustomer);
    assertTrue(tracker.anyChange());

    try {
      Customer.save();
    } catch (IOException ex) {}

    assertTrue(!tracker.anyChange());
    aCustomer.setAddress("Somewhere else");
    assertTrue(tracker.anyChange());
  }

  private String customerTestData =
    "<customers>"
      + "  <customer name=\"Dave Astels\" phone=\"542-4562\" address=\"Wolfville\"/>"
      + "  <customer name=\"Miro Novak\" phone=\"555-1111\" address=\"Calgary\"/>"
      + "</customers>";

  private Customer dave = null;
  private Customer miro = null;
}
