// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm;

import java.io.*;
import java.text.MessageFormat;
import java.util.*;

import org.xml.sax.AttributeList;
import reh.SaxMapper.SaxMapper;
import reh.SaxMapper.TagTracker;

/**
 * This class represents an estimate for work which the customer desires.
 */

public class Quote extends Statement {
  public static void reset() {
    Sequence.removeSequence("quote");
  }

  public static void clear() {
    quotes.clear();
    nullQuote.getTracker().reset();
  }

  public static Quote newQuote(Customer somebody, String aDescription) {
    int newQuoteNumber = Sequence.getSequence("quote").next();
    Quote newQuote = new Quote(somebody, aDescription, newQuoteNumber);
    return addQuote(newQuote);
  }

  public static Quote newQuote(
    Customer somebody,
    String aDescription,
    Date aDate) {
    int newQuoteNumber = Sequence.getSequence("quote").next();
    Quote newQuote = new Quote(somebody, aDescription, newQuoteNumber, aDate);
    return addQuote(newQuote);
  }

  protected static Quote addQuote(Quote aQuote) {
    quotes.put(new Integer(aQuote.getNumber()), aQuote);
    aQuote.getTracker().set();
    return aQuote;
  }

  public static Quote findQuoteByNumber(int quoteNumber) {
    Integer key = new Integer(quoteNumber);
    return quotes.containsKey(key) ? (Quote) quotes.get(key) : nullQuote;
  }

  public static Collection allQuotes() {
    return quotes.values();
  }

  public static void updateFrom(Collection updatedQuotes) throws IOException {
    clear();
    for (Iterator quoteIterator = updatedQuotes.iterator();
      quoteIterator.hasNext();
      ) {
      addQuote((Quote) quoteIterator.next());
    }
    save();
  }

  public static void load(Reader reader) {
    QuoteLoader loader = new QuoteLoader();
    clear();
    loader.fromXML(reader);
    nullQuote.getTracker().reset();
  }

  public static void save() throws IOException {
    if (filename != null) {
      FileWriter writer = new FileWriter(filename, false);
      allToXML(writer);
    }
    nullQuote.getTracker().reset();
  }

  public static void setFile(String aFilename) throws FileNotFoundException {
    filename = aFilename;
    FileReader reader = new FileReader(filename);
    load(reader);
  }

  public static void allToXML(Writer writer) throws IOException {
    writer.write("<quotes>\n");
    for (Iterator quoteIterator = quotes.values().iterator();
      quoteIterator.hasNext();
      ) {
      Quote aQuote = (Quote) quoteIterator.next();
      aQuote.toXML(writer);
    }
    writer.write("</quotes>\n");
    writer.flush();
  }

  public static Quote makeEmptyQuote() {
    return newQuote(Customer.nullCustomer, "-- Edit Description --");
  }

  private static Map quotes = new TreeMap();
  public static Quote nullQuote = new Quote(Customer.nullCustomer, "", 0);
  public static String filename = null;

  protected Quote(
    Customer somebody,
    String aDescription,
    int aNumber,
    Date aDate) {
    super(somebody, aDescription, aNumber, aDate);
  }

  protected Quote(Customer somebody, String aDescription, int aNumber) {
    super(somebody, aDescription, aNumber);
  }

  protected ChangeTracker getTracker() {
    return ChangeTracker.getTracker("quote");
  }

  public void toXML(Writer writer) throws IOException {
    boolean hasItems = !parts().isEmpty();
    writer.write(
      MessageFormat.format(
        "\t<quote number=\"{0,number,integer}\" customername=\"{1}\" description=\"{2}\"{3}>\n",
        new Object[] {
          new Integer(number),
          customer.getName(),
          description,
          hasItems ? "" : "/" }));
    if (hasItems) {
      writer.write("\t\t<items>\n");
      MessageFormat format =
        new MessageFormat("\t\t\t<item part=\"{0}\" quantity=\"{1,number,integer}\"/>\n");
      for (Iterator partIterator = parts().iterator(); partIterator.hasNext();) {
        Part aPart = (Part) partIterator.next();
        Object[] args = { aPart.getName(), new Integer(getQuantityOf(aPart))};
        writer.write(format.format(args));
      }
      writer.write("\t\t</items>\n\t</quote>\n");
    }
  }

  private static class QuoteLoader extends SaxMapper {

    Quote currentQuote;

    public Object getMappedObject() {
      return null;
    }

    public TagTracker createTagTrackerNetwork() {
      TagTracker root = new TagTracker() {};

      TagTracker quoteTracker = new TagTracker() {
        public void onStart(String localName, AttributeList atts) {
          String description = atts.getValue("description");
          int number = Integer.parseInt(atts.getValue("number"));
          String customerName = atts.getValue("customername");
          currentQuote =
            new Quote(Customer.findCustomerByName(customerName), description, number);
          Quote.addQuote(currentQuote);
        }
      };
      root.track("/quotes/quote", quoteTracker);

      TagTracker itemTracker = new TagTracker() {
        public void onStart(String localName, AttributeList atts) {
          Part aPart = Part.findPartByName(atts.getValue("part"));
          int quantity = Integer.parseInt(atts.getValue("quantity"));
          currentQuote.setQuantityOf(aPart, quantity);
        }
      };
      quoteTracker.track("items/item", itemTracker);

      return root;
    }

    public void loadFromFile(String id) throws Exception {
      FileInputStream in = new FileInputStream(id);
      fromXML(in);
    }

  }
}