// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm.tests;

import java.io.*;
import java.math.BigDecimal;
import java.util.Iterator;

import junit.framework.TestCase;

import com.saorsa.tm.ChangeTracker;
import com.saorsa.tm.Part;

public class PartTest extends TestCase {
  public PartTest(String name) {
    super(name);
  }

  protected void setUp() {
    plainSwitch = new Part("plain switch", new BigDecimal(1.25));
    anotherPlainSwitch = new Part("plain switch", new BigDecimal(1.25));
    decoraSwitch = new Part("decora switch", new BigDecimal(102.75));
  }

  protected void tearDown() {
    Part.clear();
  }

  public void testEquality() {
    assertEquals(plainSwitch, plainSwitch);
    assertEquals(plainSwitch, anotherPlainSwitch);
    assertTrue(!plainSwitch.equals(decoraSwitch));
  }

  public void testComparing() {
    assertTrue(decoraSwitch.compareTo(plainSwitch) < 0);
    assertTrue(plainSwitch.compareTo(decoraSwitch) > 0);
    assertTrue(decoraSwitch.compareTo(decoraSwitch) == 0);
  }

  public void testAddition() {
    Part.addPart(plainSwitch);
    assertTrue(Part.containsPart(plainSwitch));
    assertTrue(!Part.containsPart(decoraSwitch));
    Part.addPart(decoraSwitch);
    assertTrue(Part.containsPart(decoraSwitch));
  }

  public void testRemoval() {
    Part.addPart(plainSwitch);
    Part.addPart(decoraSwitch);
    Part.removePart(plainSwitch);
    assertTrue(!Part.containsPart(plainSwitch));
    assertTrue(Part.containsPart(decoraSwitch));
    Part.removePart(decoraSwitch);
    assertTrue(!Part.containsPart(decoraSwitch));
  }

  public void testCount() {
    assertEquals(0, Part.numberOfParts());
    Part.addPart(plainSwitch);
    assertEquals(1, Part.numberOfParts());
    Part.addPart(decoraSwitch);
    assertEquals(2, Part.numberOfParts());
    Part.removePart(plainSwitch);
    assertEquals(1, Part.numberOfParts());
    Part.removePart(decoraSwitch);
    assertEquals(0, Part.numberOfParts());
  }

  public void testOrdering() {
    Part.addPart(plainSwitch);
    Part.addPart(decoraSwitch);
    Iterator partIterator = Part.allParts().iterator();
    Part previousPart = Part.nullPart;
    while (partIterator.hasNext()) {
      Part thisPart = (Part) partIterator.next();
      assertTrue(thisPart.compareTo(previousPart) > 0);
      previousPart = thisPart;
    }
  }

  public void testLookup() {
    Part.addPart(plainSwitch);
    Part.addPart(decoraSwitch);
    assertEquals(plainSwitch, Part.findPartByName(plainSwitch.getName()));
    assertEquals(decoraSwitch, Part.findPartByName(decoraSwitch.getName()));
    assertEquals(Part.nullPart, Part.findPartByName("bad part name"));
  }

  public void testLoading() {
    StringReader reader = new StringReader(partTestData);
    Part.load(reader);

    Part plain = Part.findPartByName("plain switch");
    assertEquals("plain switch", plain.getName());
    assertTrue(new BigDecimal(1.25).compareTo(plain.getPrice()) == 0);

    Part decora = Part.findPartByName("decora switch");
    assertEquals("decora switch", decora.getName());
    assertTrue(new BigDecimal(2.75).compareTo(decora.getPrice()) == 0);
  }

  public void testAsXML() {
    Part.addPart(plainSwitch);
    Part.addPart(decoraSwitch);

    try {
      StringWriter writer = new StringWriter();
      Part.allToXML(writer);
      Part.clear();

      StringReader reader = new StringReader(writer.toString());
      Part.load(reader);

      Part plain = Part.findPartByName("plain switch");
      assertEquals("plain switch", plain.getName());
      assertTrue(new BigDecimal(1.25).compareTo(plain.getPrice()) == 0);

      Part decora = Part.findPartByName("decora switch");
      assertEquals("decora switch", decora.getName());
      assertTrue(new BigDecimal(102.75).compareTo(decora.getPrice()) == 0);
    } catch (IOException ex) {
      fail();
    }
  }

  public void testOutstandingChanges() {
    Part.addPart(plainSwitch);
    ChangeTracker tracker = ChangeTracker.getTracker("part");
    assertTrue(tracker.anyChange());

    Part.clear();
    assertTrue(!tracker.anyChange());

    Part.addPart(decoraSwitch);
    assertTrue(tracker.anyChange());
    try {
      Part.save();
    } catch (IOException ex) {}

    assertTrue(!tracker.anyChange());
    plainSwitch.setName("Something else");
    assertTrue(tracker.anyChange());
  }

  private String partTestData =
    "<parts>"
      + "  <part name=\"plain switch\" price=\"1.25\"/>"
      + "  <part name=\"decora switch\" price=\"2.75\"/>"
      + "</parts>";
  Part plainSwitch = null;
  Part anotherPlainSwitch = null;
  Part decoraSwitch = null;
}