// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm.tests;

import java.io.*;
import java.math.BigDecimal;

import junit.framework.TestCase;

import com.saorsa.tm.*;

public class InvoiceTest extends TestCase {
  public InvoiceTest(String name) {
    super(name);
  }

  protected void setUp() {
    Invoice.reset();
    Invoice.clear();
    dave = new Customer("Dave Astels", "542-4562", "Wolfville");
    Customer.addCustomer(dave);
    miro = new Customer("Miro Novak", " 555-1111 ", " Calgary ");
    Customer.addCustomer(miro);
    anInvoice = Invoice.newInvoice(dave, "First Invoice");
    plainSwitch = new Part("plain switch", new BigDecimal(1.25));
    anotherPlainSwitch = new Part("plain switch", new BigDecimal(1.25));
    decoraSwitch = new Part("decora switch", new BigDecimal(2.75));
    decoraOutlet = new Part("decora outlet", new BigDecimal(3.50));
    Part.clear();
    Part.addPart(plainSwitch);
    Part.addPart(decoraSwitch);
    Part.addPart(decoraOutlet);
  }

  public void testEmptyInvoice() {
    assertEquals(0, anInvoice.lineitemCount());
  }

  public void testItemAddition() {
    anInvoice.add(plainSwitch, 2);
    assertEquals(1, anInvoice.lineitemCount());
    anInvoice.add(decoraSwitch, 3);
    assertEquals(2, anInvoice.lineitemCount());
  }

  public void testItemRemoval() {
    anInvoice.add(plainSwitch, 2);
    anInvoice.add(decoraSwitch, 3);
    assertEquals(2, anInvoice.lineitemCount());
    anInvoice.remove(plainSwitch);
    assertEquals(1, anInvoice.lineitemCount());
    anInvoice.remove(decoraSwitch);
    assertEquals(0, anInvoice.lineitemCount());
  }

  public void testQuantity() {
    assertEquals(0, anInvoice.getQuantityOf(plainSwitch));
    anInvoice.add(plainSwitch, 2);
    anInvoice.add(decoraSwitch, 3);
    assertEquals(2, anInvoice.getQuantityOf(plainSwitch));
    assertEquals(3, anInvoice.getQuantityOf(decoraSwitch));
  }

  public void testAdditionalItems() {
    anInvoice.add(plainSwitch, 2);
    anInvoice.add(decoraSwitch, 3);
    anInvoice.add(plainSwitch, 1);
    anInvoice.add(decoraSwitch, 3);
    assertEquals(3, anInvoice.getQuantityOf(plainSwitch));
    assertEquals(6, anInvoice.getQuantityOf(decoraSwitch));
  }

  public void testSettingQuantities() {
    anInvoice.add(plainSwitch, 2);
    anInvoice.add(decoraSwitch, 3);
    anInvoice.setQuantityOf(plainSwitch, 1);
    anInvoice.setQuantityOf(decoraSwitch, 1);
    assertEquals(1, anInvoice.getQuantityOf(plainSwitch));
    assertEquals(1, anInvoice.getQuantityOf(decoraSwitch));
  }

  public void testSumming() {
    anInvoice.add(plainSwitch, 2);
    assertTrue(new BigDecimal(2.5).compareTo(anInvoice.calculateSubTotal()) == 0);
    anInvoice.add(decoraSwitch, 3);
    assertTrue(new BigDecimal(10.75).compareTo(anInvoice.calculateSubTotal()) == 0);
  }

  public void testNumbering() {
    assertEquals(1, anInvoice.getNumber());
    Invoice secondInvoice = Invoice.newInvoice(dave, "Second Invoice");
    assertEquals(2, secondInvoice.getNumber());
    Invoice thirdInvoice = Invoice.newInvoice(dave, "Third Invoice");
    assertEquals(3, thirdInvoice.getNumber());
  }

  public void testAssociatedQuote() {
    Quote aQuote = Quote.newQuote(dave, "A quote for Dave");
    aQuote.add(plainSwitch, 2);
    aQuote.add(decoraSwitch, 3);
    Invoice anAssociatedInvoice =
      Invoice.newInvoiceWithAssociatedQuote(
        aQuote,
        "An Invoice with associated Quote");
    assertSame(dave, anAssociatedInvoice.getCustomer());
    assertEquals(2, anAssociatedInvoice.getQuantityOf(plainSwitch));
    assertEquals(3, anAssociatedInvoice.getQuantityOf(decoraSwitch));
  }

  public void testLookup() {
    Invoice secondInvoice = Invoice.newInvoice(dave, "Second Invoice");
    Invoice thirdInvoice = Invoice.newInvoice(dave, "Third Invoice");

    assertSame(anInvoice, Invoice.findInvoiceByNumber(1));
    assertSame(thirdInvoice, Invoice.findInvoiceByNumber(3));
    assertSame(secondInvoice, Invoice.findInvoiceByNumber(2));
  }

  public void testLoading() {
    StringReader reader = new StringReader(invoiceTestData);
    Invoice.load(reader);

    Invoice anInvoice;

    anInvoice = Invoice.findInvoiceByNumber(3);
    assertEquals(3, anInvoice.getNumber());
    assertEquals(miro, anInvoice.getCustomer());
    assertEquals("Second Invoice", anInvoice.getDescription());
    assertEquals(2, anInvoice.getQuantityOf(decoraSwitch));
    assertEquals(3, anInvoice.getQuantityOf(decoraOutlet));

    anInvoice = Invoice.findInvoiceByNumber(1);
    assertEquals(1, anInvoice.getNumber());
    assertEquals(dave, anInvoice.getCustomer());
    assertEquals("First Invoice", anInvoice.getDescription());
    assertEquals(3, anInvoice.getQuantityOf(plainSwitch));
  }

  public void testAsXML() {
    Invoice secondInvoice = Invoice.newInvoice(dave, "Second Invoice");
    secondInvoice.setQuantityOf(plainSwitch, 3);
    Invoice thirdInvoice = Invoice.newInvoice(miro, "Third Invoice");
    thirdInvoice.setQuantityOf(decoraSwitch, 2);
    thirdInvoice.setQuantityOf(decoraOutlet, 3);

    try {
      StringWriter writer = new StringWriter();
      Invoice.allToXML(writer);

      StringReader reader = new StringReader(writer.toString());
      Invoice.load(reader);

      Invoice anInvoice;

      anInvoice = Invoice.findInvoiceByNumber(3);
      assertEquals(3, anInvoice.getNumber());
      assertEquals(miro, anInvoice.getCustomer());
      assertEquals("Third Invoice", anInvoice.getDescription());
      assertEquals(2, anInvoice.getQuantityOf(decoraSwitch));
      assertEquals(3, anInvoice.getQuantityOf(decoraOutlet));

      anInvoice = Invoice.findInvoiceByNumber(1);
      assertEquals(1, anInvoice.getNumber());
      assertEquals(dave, anInvoice.getCustomer());
      assertEquals("First Invoice", anInvoice.getDescription());

      anInvoice = Invoice.findInvoiceByNumber(2);
      assertEquals(2, anInvoice.getNumber());
      assertEquals(dave, anInvoice.getCustomer());
      assertEquals("Second Invoice", anInvoice.getDescription());
      assertEquals(3, anInvoice.getQuantityOf(plainSwitch));
    } catch (IOException ex) {
      fail();
    }
  }

  public void testOutstandingChanges() {
    ChangeTracker tracker = ChangeTracker.getTracker("invoice");
    assertTrue(tracker.anyChange());

    Invoice.clear();
    assertTrue(!tracker.anyChange());

    Invoice anInvoice = Invoice.newInvoice(dave, "something");
    assertTrue(tracker.anyChange());

    try {
      Invoice.save();
    } catch (IOException ex) {}

    assertTrue(!tracker.anyChange());
    anInvoice.setDescription("Something else");
    assertTrue(tracker.anyChange());
  }

  private String invoiceTestData =
    "<invoices>"
      + "  <invoice number=\"1\" customername=\"Dave Astels\" description=\"First Invoice\">"
      + "    <items>"
      + "      <item part=\"plain switch\" quantity=\"3\"/>"
      + "    </items>"
      + "  </invoice>"
      + "  <invoice number=\"3\" customername=\"Miro Novak\" description=\"Second Invoice\">"
      + "    <items>"
      + "      <item part=\"decora switch\" quantity=\"2\"/>"
      + "      <item part=\"decora outlet\" quantity=\"3\"/>"
      + "    </items>"
      + "  </invoice>"
      + "</invoices>";
  Customer dave = null;
  Customer miro = null;
  Invoice anInvoice = null;
  Part plainSwitch = null;
  Part anotherPlainSwitch = null;
  Part decoraSwitch = null;
  Part decoraOutlet = null;
}