// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm;

import java.io.*;
import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;

import org.xml.sax.AttributeList;
import reh.SaxMapper.SaxMapper;
import reh.SaxMapper.TagTracker;

/**
 * This class represents an invoice which is sent to the customer, 
 * indicating how much they owe, and (possibly) for what.
 */

public class Invoice extends Statement {
    public static void reset() {
        Sequence.removeSequence("invoice");
    }

    public static void clear() {
        invoices.clear();
        nullInvoice.getTracker().reset();
    }

    public static Invoice newInvoice(Customer somebody, String aDescription) {
        int newInvoiceNumber = Sequence.getSequence("invoice").next();
        Invoice newInvoice = new Invoice(somebody, aDescription, newInvoiceNumber);
        return addInvoice(newInvoice);
    }

    public static Invoice newInvoiceWithAssociatedQuote(Quote aQuote, String aDescription) {
        int newInvoiceNumber = Sequence.getSequence("invoice").next();
        Invoice newInvoice = new Invoice(aQuote, aDescription, newInvoiceNumber);
        for (Iterator itemIterator = aQuote.items().iterator(); itemIterator.hasNext();) {
            Lineitem anItem = (Lineitem) itemIterator.next();
            newInvoice.setQuantityOf(anItem.getPart(), anItem.getQuantity());
        }
        return addInvoice(newInvoice);
    }

    protected static Invoice addInvoice(Invoice anInvoice) {
        invoices.put(new Integer(anInvoice.getNumber()), anInvoice);
        anInvoice.getTracker().set();
        return anInvoice;
    }

    public static Invoice findInvoiceByNumber(int invoiceNumber) {
        Integer key = new Integer(invoiceNumber);
        return invoices.containsKey(key) ? (Invoice) invoices.get(key) : nullInvoice;
    }

    public static Collection allInvoices() {
        return invoices.values();
    }

    public static void updateFrom(Collection updatedInvoices) throws IOException {
        clear();
        for (Iterator invoiceIterator = updatedInvoices.iterator(); invoiceIterator.hasNext();) {
            addInvoice((Invoice) invoiceIterator.next());
        }
        save();
    }

    public static void load(Reader reader) {
        InvoiceLoader loader = new InvoiceLoader();
        clear();
        loader.fromXML(reader);
        nullInvoice.getTracker().reset();
    }

    public static void save() throws IOException {
        if (filename != null) {
            FileWriter writer = new FileWriter(filename, false);
            allToXML(writer);
        }
        nullInvoice.getTracker().reset();
    }

    public static void setFile(String aFilename) throws FileNotFoundException {
        filename = aFilename;
        FileReader reader = new FileReader(filename);
        load(reader);
    }

    public static void allToXML(Writer writer) throws IOException {
        writer.write("<invoices>\n");
        for (Iterator invoiceIterator = invoices.values().iterator(); invoiceIterator.hasNext();) {
            Invoice anInvoice = (Invoice) invoiceIterator.next();
            anInvoice.toXML(writer);
        }
        writer.write("</invoices>");
        writer.flush();
    }

    public static Invoice makeEmptyInvoice() {
        return newInvoice(Customer.nullCustomer, "-- Edit Description --");
    }

    private static Map invoices = new TreeMap();
    public static Invoice nullInvoice = new Invoice(Customer.nullCustomer, "", 0);
    public static String filename = null;

    protected Invoice(Customer somebody, String aDescription, int aNumber) {
        super(somebody, aDescription, aNumber);
    }

    protected Invoice(Quote quoteToAssociate, String aDescription, int aNumber) {
        super(quoteToAssociate.getCustomer(), aDescription, aNumber);
        associatedQuote = quoteToAssociate;
    }

    protected ChangeTracker getTracker() {
        return ChangeTracker.getTracker("invoice");
    }

    public void toXML(Writer writer) throws IOException {
        boolean hasItems = !parts().isEmpty();
        writer.write(MessageFormat.format("\t<invoice number=\"{0,number,integer}\" customername=\"{1}\" description=\"{2}\"", new Object[] { new Integer(number), customer.getName(), description}));
        if (hasAssociatedQuote()) {
        	writer.write(MessageFormat.format(" associatedquote=\"{0}\"", new Object[] {new Integer(associatedQuote.getNumber())}));
        }
        if (hasItems) {
        	writer.write(">\n");
            writer.write("\t\t<items>\n");
            MessageFormat format = new MessageFormat("\t\t\t<item part=\"{0}\" quantity=\"{1,number,integer}\"/>\n");
            for (Iterator partIterator = parts().iterator(); partIterator.hasNext();) {
                Part aPart = (Part) partIterator.next();
                Object[] args = { aPart.getName(), new Integer(getQuantityOf(aPart))};
                writer.write(format.format(args));
            }
            writer.write("\t\t</items>\n\t</invoice>\n");
        } else {
        	writer.write("/>\n");
        }
    }

    public boolean hasAssociatedQuote() {
        return associatedQuote != null;
    }

    private Quote associatedQuote;

    private static class InvoiceLoader extends SaxMapper {

        Invoice currentInvoice;

        public Object getMappedObject() {
            return null;
        }

        public TagTracker createTagTrackerNetwork() {
            TagTracker root = new TagTracker() {};

            TagTracker invoiceTracker = new TagTracker() {
                public void onStart(String localName, AttributeList atts) {
                    String description = atts.getValue("description");
                    int number = Integer.parseInt(atts.getValue("number"));
                    String customerName = atts.getValue("customername");
                    // deal with option associatedquote
                    int quoteNumber = Integer.parseInt(atts.getValue("associatedquote"));
                    currentInvoice = new Invoice(Customer.findCustomerByName(customerName), description, number);
                    Invoice.addInvoice(currentInvoice);
                }
            };
            root.track("/invoices/invoice", invoiceTracker);

            TagTracker itemTracker = new TagTracker() {
                public void onStart(String localName, AttributeList atts) {
                    Part aPart = Part.findPartByName(atts.getValue("part"));
                    int quantity = Integer.parseInt(atts.getValue("quantity"));
                    currentInvoice.setQuantityOf(aPart, quantity);
                }
            };
            invoiceTracker.track("items/item", itemTracker);

            return root;
        }

        public void loadFromFile(String id) throws Exception {
            FileInputStream in = new FileInputStream(id);
            fromXML(in);
        }

    }
}
