// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm;

import java.io.*;
import java.text.MessageFormat;
import java.util.*;

import org.xml.sax.AttributeList;
import reh.SaxMapper.SaxMapper;
import reh.SaxMapper.TagTracker;

/** 
 * Represent a customer in the system. 
 * Also keeps track of the set of customers known to the system 
 */

public class Customer implements Comparable {
  public static void addCustomer(Customer theCustomerToAdd) {
    customers.put(
      theCustomerToAdd.getName(),
      theCustomerToAdd);
    tracker.set();
  }

  public static void removeCustomer(Customer theCustomerToRemove) {
    customers.remove(theCustomerToRemove.getName());
    tracker.set();
  }

  public static void removeCustomerNamed(String theNameToRemove) {
    customers.remove(theNameToRemove);
    tracker.set();
  }

  public static boolean containsCustomer(Customer theCustomerToCheckFor) {
    return customers.containsValue(
      theCustomerToCheckFor);
  }

  public static Iterator getAlphabeticalListOfCustomers() {
    return customers.values().iterator();
  }

  public static Collection allCustomers() {
    return customers.values();
  }

  public static Collection allCustomerNames() {
    return customers.keySet();
  }

  public static int numberOfCustomers() {
    return customers.size();
  }

  public static Customer findCustomerByName(String theNameToLookup) {
    if (customers.containsKey(theNameToLookup)) {
      return (Customer) customers.get(theNameToLookup);
    } else {
      return nullCustomer;
    }
  }

  public static void updateFrom(Collection updatedCustomers)
    throws IOException {
    clear();
    for (Iterator customerIterator =
      updatedCustomers.iterator();
      customerIterator.hasNext();
      ) {
      addCustomer((Customer) customerIterator.next());
    }
    save();
  }

  public static void load(Reader reader) {
    CustomerLoader loader = new CustomerLoader();
    loader.fromXML(reader);
    tracker.reset();
  }

  public static void allToXML(Writer writer)
    throws IOException {
    writer.write("<customers>\n");
    for (Iterator customerIterator =
      customers.values().iterator();
      customerIterator.hasNext();
      ) {
      Customer aCustomer =
        (Customer) customerIterator.next();
      aCustomer.toXML(writer);
    }
    writer.write("</customers>");
    writer.flush();
  }

  public static void save() throws IOException {
    if (filename != null) {
      FileWriter writer =
        new FileWriter(filename, false);
      allToXML(writer);
    }
    tracker.reset();
  }

  public static void setFile(String aFilename)
    throws FileNotFoundException {
    filename = aFilename;
    FileReader reader = new FileReader(filename);
    load(reader);
  }

  public static void clear() {
    customers.clear();
    tracker.reset();
  }

  public static Customer makeEmptyCustomer() {
    return new Customer(
      "--Edit this Customer--",
      "555-5555",
      "somewhere");
  }

  private static Map customers = new TreeMap();
  public static Customer nullCustomer =
    new Customer("", "", "");
  public static String filename = null;
  private static ChangeTracker tracker =
    ChangeTracker.getTracker("customer");

  public Customer(
    String aName,
    String aPhoneNumber,
    String anAddress) {
    name = aName;
    phoneNumber = aPhoneNumber;
    address = anAddress;
  }

  public boolean equals(Object anObject) {
    if (anObject == this) {
      return true;
    } else if (anObject instanceof Customer) {
      Customer aCustomer = (Customer) anObject;
      return name.equals(aCustomer.name);
    } else {
      return false;
    }
  }

  public int hashCode() {
    int result = 17;
    result = 37 * result + name.hashCode();
    return result;
  }

  public String getName() {
    return name;
  }

  public void setName(String newName) {
    name = newName;
    tracker.set();
  }

  public String getPhoneNumber() {
    return phoneNumber;
  }

  public void setPhoneNumber(String newPhoneNumber) {
    phoneNumber = newPhoneNumber;
    tracker.set();
  }

  public String getAddress() {
    return address;
  }

  public void setAddress(String newAddress) {
    address = newAddress;
    tracker.set();
  }

  public int compareTo(Object objectToCompareTo) {
    Customer customerToCompareTo =
      (Customer) objectToCompareTo;
    return name.compareTo(customerToCompareTo.name);
  }

  public String toString() {
    StringBuffer buf = new StringBuffer();
    buf.append("Customer[");
    buf.append(name);
    buf.append("]");
    return buf.toString();
  }

  public void toXML(Writer writer)
    throws IOException {
    writer.write(
      MessageFormat.format(
        "\t<customer name=\"{0}\" phone=\"{1}\" address=\"{2}\"/>\n",
        new Object[] { name, phoneNumber, address }));
  }

  private String name;
  private String phoneNumber;
  private String address;

   private static class CustomerLoader
    extends SaxMapper {

    public Object getMappedObject() {
      return null;
    }

    public TagTracker createTagTrackerNetwork() {
      TagTracker root = new TagTracker() {};

      TagTracker customerTracker = new TagTracker() {
        public void onStart(
          String localName,
          AttributeList atts) {
          String name = atts.getValue("name");
          String phone = atts.getValue("phone");
          String address = atts.getValue("address");
          Customer.addCustomer(
            new Customer(name, phone, address));
        }
      };
      root.track(
        "/customers/customer",
        customerTracker);

      return root;
    }

    public void loadFromFile(String id)
      throws Exception {
      FileInputStream in = new FileInputStream(id);
      fromXML(in);
    }

  }
}