// Copyright (C) 2001 Saorsa Development Inc.

package com.saorsa.tm.tests;

import java.io.*;
import java.math.BigDecimal;
import java.util.Date;

import junit.framework.TestCase;

import com.saorsa.tm.*;

public class QuoteTest extends TestCase {
  public QuoteTest(String name) {
    super(name);
  }

  protected void setUp() {
    Quote.reset();
    Quote.clear();
    dave = new Customer("Dave Astels", "542-4562", "Wolfville");
    Customer.addCustomer(dave);
    miro = new Customer("Miro Novak", " 555-1111 ", " Calgary ");
    Customer.addCustomer(miro);
    aQuote = Quote.newQuote(dave, "First Quote", new Date(101, 8, 28));
    // Sept 28, 2001
    plainSwitch = new Part("plain switch", new BigDecimal(1.25));
    anotherPlainSwitch = new Part("plain switch", new BigDecimal(1.25));
    decoraSwitch = new Part("decora switch", new BigDecimal(2.75));
    decoraOutlet = new Part("decora outlet", new BigDecimal(3.50));
    Part.clear();
    Part.addPart(plainSwitch);
    Part.addPart(decoraSwitch);
    Part.addPart(decoraOutlet);
  }

  public void testEmptyQuote() {
    assertEquals(0, aQuote.lineitemCount());
  }

  public void testItemAddition() {
    aQuote.add(plainSwitch, 2);
    assertEquals(1, aQuote.lineitemCount());
    aQuote.add(decoraSwitch, 3);
    assertEquals(2, aQuote.lineitemCount());
  }

  public void testItemRemoval() {
    aQuote.add(plainSwitch, 2);
    aQuote.add(decoraSwitch, 3);
    assertEquals(2, aQuote.lineitemCount());
    aQuote.remove(plainSwitch);
    assertEquals(1, aQuote.lineitemCount());
    aQuote.remove(decoraSwitch);
    assertEquals(0, aQuote.lineitemCount());
  }

  public void testQuantity() {
    assertEquals(0, aQuote.getQuantityOf(plainSwitch));
    aQuote.add(plainSwitch, 2);
    aQuote.add(decoraSwitch, 3);
    assertEquals(2, aQuote.getQuantityOf(plainSwitch));
    assertEquals(3, aQuote.getQuantityOf(decoraSwitch));
  }

  public void testAdditionalItems() {
    aQuote.add(plainSwitch, 2);
    aQuote.add(decoraSwitch, 3);
    aQuote.add(plainSwitch, 1);
    aQuote.add(decoraSwitch, 3);
    assertEquals(3, aQuote.getQuantityOf(plainSwitch));
    assertEquals(6, aQuote.getQuantityOf(decoraSwitch));
  }

  public void testSettingQuantities() {
    aQuote.add(plainSwitch, 2);
    aQuote.add(decoraSwitch, 3);
    aQuote.setQuantityOf(plainSwitch, 1);
    aQuote.setQuantityOf(decoraSwitch, 1);
    assertEquals(1, aQuote.getQuantityOf(plainSwitch));
    assertEquals(1, aQuote.getQuantityOf(decoraSwitch));
  }

  public void testSumming() {
    aQuote.add(plainSwitch, 2);
    assertTrue(new BigDecimal(2.5).compareTo(aQuote.calculateSubTotal()) == 0);
    aQuote.add(decoraSwitch, 3);
    assertTrue(new BigDecimal(10.75).compareTo(aQuote.calculateSubTotal()) == 0);
  }

  public void testNumbering() {
    assertEquals(1, aQuote.getNumber());
    Quote secondQuote = Quote.newQuote(dave, "Second Quote");
    assertEquals(2, secondQuote.getNumber());
    Quote thirdQuote = Quote.newQuote(dave, "Third Quote");
    assertEquals(3, thirdQuote.getNumber());
  }

  public void testLookup() {
    Quote secondQuote = Quote.newQuote(dave, "Second Quote");
    Quote thirdQuote = Quote.newQuote(dave, "Third Quote");

    assertSame(aQuote, Quote.findQuoteByNumber(1));
    assertSame(thirdQuote, Quote.findQuoteByNumber(3));
    assertSame(secondQuote, Quote.findQuoteByNumber(2));
  }

  public void testLoading() {
    StringReader reader = new StringReader(quoteTestData);
    Quote.load(reader);

    Quote aQuote;

    aQuote = Quote.findQuoteByNumber(3);
    assertEquals(3, aQuote.getNumber());
    assertEquals(miro, aQuote.getCustomer());
    assertEquals("Second Quote", aQuote.getDescription());
    assertEquals(2, aQuote.getQuantityOf(decoraSwitch));
    assertEquals(3, aQuote.getQuantityOf(decoraOutlet));

    aQuote = Quote.findQuoteByNumber(1);
    assertEquals(1, aQuote.getNumber());
    assertEquals(dave, aQuote.getCustomer());
    assertEquals("First Quote", aQuote.getDescription());
    assertEquals(3, aQuote.getQuantityOf(plainSwitch));
  }

  public void testAsXML() {
    Quote secondQuote = Quote.newQuote(dave, "Second Quote");
    secondQuote.setQuantityOf(plainSwitch, 3);
    Quote thirdQuote = Quote.newQuote(miro, "Third Quote");
    thirdQuote.setQuantityOf(decoraSwitch, 2);
    thirdQuote.setQuantityOf(decoraOutlet, 3);

    try {
      StringWriter writer = new StringWriter();
      Quote.allToXML(writer);

      StringReader reader = new StringReader(writer.toString());
      Quote.load(reader);

      Quote aQuote;

      aQuote = Quote.findQuoteByNumber(3);
      assertEquals(3, aQuote.getNumber());
      assertEquals(miro, aQuote.getCustomer());
      assertEquals("Third Quote", aQuote.getDescription());
      assertEquals(2, aQuote.getQuantityOf(decoraSwitch));
      assertEquals(3, aQuote.getQuantityOf(decoraOutlet));

      aQuote = Quote.findQuoteByNumber(1);
      assertEquals(1, aQuote.getNumber());
      assertEquals(dave, aQuote.getCustomer());
      assertEquals("First Quote", aQuote.getDescription());

      aQuote = Quote.findQuoteByNumber(2);
      assertEquals(2, aQuote.getNumber());
      assertEquals(dave, aQuote.getCustomer());
      assertEquals("Second Quote", aQuote.getDescription());
      assertEquals(3, aQuote.getQuantityOf(plainSwitch));
    } catch (IOException ex) {
      fail();
    }
  }

  public void testOutstandingChanges() {
    ChangeTracker tracker = ChangeTracker.getTracker("quote");
    assertTrue(tracker.anyChange());

    Quote.clear();
    assertTrue(!tracker.anyChange());

    Quote aQuote = Quote.newQuote(dave, "something");
    assertTrue(tracker.anyChange());

    try {
      Quote.save();
    } catch (IOException ex) {}

    assertTrue(!tracker.anyChange());
    aQuote.setDescription("Something else");
    assertTrue(tracker.anyChange());
  }

  private String quoteTestData =
    "<quotes>"
      + "  <quote number=\"1\" customername=\"Dave Astels\" description=\"First Quote\">"
      + "    <items>"
      + "      <item part=\"plain switch\" quantity=\"3\"/>"
      + "    </items>"
      + "  </quote>"
      + "  <quote number=\"3\" customername=\"Miro Novak\" description=\"Second Quote\">"
      + "    <items>"
      + "      <item part=\"decora switch\" quantity=\"2\"/>"
      + "      <item part=\"decora outlet\" quantity=\"3\"/>"
      + "    </items>"
      + "  </quote>"
      + "</quotes>";
  Customer dave = null;
  Customer miro = null;
  Quote aQuote = null;
  Part plainSwitch = null;
  Part anotherPlainSwitch = null;
  Part decoraSwitch = null;
  Part decoraOutlet = null;
}