// Rysunek 24.34. AddressBookController.java
// Kontroler aplikacji książki adresowej
import java.util.List;
import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.ActionEvent;
import javafx.fxml.FXML;
import javafx.scene.control.Alert;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.control.ListView;
import javafx.scene.control.TextField;

public class AddressBookController {
   @FXML private ListView<Person> listView; // Wyświetla nazwy kontaktów
   @FXML private TextField firstNameTextField;
   @FXML private TextField lastNameTextField;
   @FXML private TextField emailTextField;
   @FXML private TextField phoneTextField;
   @FXML private TextField findByLastNameTextField;

   // Zapewnia interakcję z bazą danych
   private final PersonQueries personQueries = new PersonQueries();

   // Przechowuje listę obiektów Person będącą wynikiem interakcji z bazą danych
   private final ObservableList<Person> contactList = 
      FXCollections.observableArrayList();
   
   // Wypełnia listView i konfiguruje nasłuchiwanie zdarzeń wyboru
   public void initialize() {
      listView.setItems(contactList); // Dowiązuje do contactsList
      getAllEntries(); // Wypełnia contactList, które aktualizuje listView
	  displayContact(listView.getSelectionModel().getSelectedItem());

      // Gdy zmieni się wybór w ListView, wyświetl dane wybranej osoby
      listView.getSelectionModel().selectedItemProperty().addListener(
         (observableValue, oldValue, newValue) -> {
            displayContact(newValue);
         }
      );     
   }

   // Pobiera wszystkie wpisy z bazy danych, aby wypełnić contactList
   private void getAllEntries() {
      contactList.setAll(personQueries.getAllPeople()); 
      selectFirstEntry();
   }

   // Wybiera pierwszy element z listy listView
   private void selectFirstEntry() {
      listView.getSelectionModel().selectFirst();
   }

   // Wyświetla informacje kontaktowe
   private void displayContact(Person person) {
      if (person != null) {
         firstNameTextField.setText(person.getFirstName());
         lastNameTextField.setText(person.getLastName());
         emailTextField.setText(person.getEmail());
         phoneTextField.setText(person.getPhoneNumber());
      }
      else {
         firstNameTextField.clear();
         lastNameTextField.clear();
         emailTextField.clear();
         phoneTextField.clear();
      }
   }

   // Dodaje nowy wpis
   @FXML
   void addEntryButtonPressed(ActionEvent event) {
      int result = personQueries.addPerson(
         firstNameTextField.getText(), lastNameTextField.getText(), 
         emailTextField.getText(), phoneTextField.getText());                                     
      
      if (result == 1) {
         displayAlert(AlertType.INFORMATION, "Dodano wpis", 
            "Poprawne dodanie nowego wpisu.");
      }
      else {
         displayAlert(AlertType.ERROR, "Wpisu nie dodano", 
            "Nie udało się dodać wpisu.");
      }
      
      getAllEntries();
   }

   // Znajdź wpisy dotyczące konkretnego nazwiska
   @FXML
   void findButtonPressed(ActionEvent event) {
      List<Person> people = personQueries.getPeopleByLastName(
         findByLastNameTextField.getText() + "%");

      if (people.size() > 0) { // Wyświetl wszystkie wpisy
         contactList.setAll(people);
         selectFirstEntry();
      }
      else {
         displayAlert(AlertType.INFORMATION, "Nazwiska nie znaleziono", 
            "Nie ma żadnych wpisów dotyczących podanego nazwiska.");
      }
   }

   // Wyświetl wszystkie wpisy
   @FXML
   void browseAllButtonPressed(ActionEvent event) {
      getAllEntries();
   }

   // Wyświetl okno dialogowe Alert
   private void displayAlert(
      AlertType type, String title, String message) {
      Alert alert = new Alert(type);
      alert.setTitle(title);
      alert.setContentText(message);
      alert.showAndWait();
   }
}


/**************************************************************************
 * (C) Copyright 1992-2018 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/
