/******************************************************************************
 *  Kompilacja:  javac Stack.java
 *  Wykonanie:    java Stack < input.txt
 *  Pliki z danymi:   https://introcs.cs.princeton.edu/43stack/tobe.txt
 *
 *  Generyczny stos implementowany z użyciem listy połączonej. Każdy element
 *  stosu jest typu Item.
 *  
 *  % more tobe.txt 
 *  to be or not to - be - - that - - - is
 *
 *  % java Stack < tobe.txt
 *  to be not that or be (2 left on stack)
 *
 ******************************************************************************/

import java.util.Iterator;
import java.util.NoSuchElementException;


/**
 *  Klasa {@code Stack} reprezentuje stos LIFO z generycznymi elementami.
 *  Dostępne są standardowe operacje <em>push</em> i <em>pop</em>, a także metody do
 *  sprawdzania wartości wierzchołka (bez zdejmowania go), sprawdzania, czy stos jest pusty, pobierania liczby
 *  elementów stosu i przechodzenia po elementach w porządku LIFO.
 *  <p>
 *  Ta implementacja wykorzystuje jednokierunkową listę połączoną z klasą zagnieżdżoną reprezentującą
 *  węzły listy.
 *  Operacje <em>push</em>, <em>pop</em>, <em>peek</em>, <em>size</em> i <em>is-empty</em>
 *  działają w stałym czasie.
 *  <p>
 *  Dodatkową dokumentację znajdziesz w <a href="https://introcs.cs.princeton.edu/43stack">podrozdziale 4.3</a> książki
 *  <i>Wprowadzenie do programowania w Javie</i> (Robert Sedgewick i Kevin Wayne). 
 *
 *  @author Robert Sedgewick
 *  @author Kevin Wayne
 *
 *  @param <Item> generyczny typ elementów stosu
 */
public class Stack<Item> implements Iterable<Item> {
    private int n;          // wielkość stosu
    private Node first;     // wierzchołek stosu

    // klasa pomocnicza dla listy połączonej
    private class Node {
        private Item item;
        private Node next;
    }

   /**
     * Inicjuje pusty stos
     */
    public Stack() {
        first = null;
        n = 0;
    }

    /**
     * Zwraca true, jeśli stos jest pusty
     *
     * @return true, jeśli stos jest pusty; false w przeciwnym razie
     */
    public boolean isEmpty() {
        return first == null;
    }

    /**
     * Zwraca liczbę elementów stosu
     *
     * @return  liczbę elementów stosu
     */
    public int size() {
        return n;
    }

    /**
     * Dodaje element do stosu
     *
     * @param  item dodawany element
     */
    public void push(Item item) {
        Node oldfirst = first;
        first = new Node();
        first.item = item;
        first.next = oldfirst;
        n++;
    }

    /**
     * Usuwa i zwraca element ostatnio dodany do stosu
     *
     * @return element ostatnio dodany do stosu
     * @throws NoSuchElementException, jeśli stos jest pusty
     */
    public Item pop() {
        if (isEmpty()) throw new NoSuchElementException("stos pusty");
        Item item = first.item;        // zapisuje zwracany element
        first = first.next;            // usuwa pierwszy węzeł
        n--;
        return item;                   // zwraca zapisany element
    }


    /**
     * Zwraca (bez usuwania) element ostatnio dodany do stosu
     *
     * @return element ostatnio dodany do stosu
     * @throws NoSuchElementException, jeśli stos jest pusty
     */
    public Item peek() {
        if (isEmpty()) throw new NoSuchElementException("stos pusty");
        return first.item;
    }

    /**
     * Zwraca tekstową reprezantację stosu
     *
     * @return sekwencję rozdzielonych spacjami elementów stosu w porządku LIFO
     */
    public String toString() {
        StringBuilder s = new StringBuilder();
        for (Item item : this) {
            s.append(item);
            s.append(' ');
        }
        return s.toString();
    }
       

    /**
     * Zwraca iterator stosu do przechodzenia po elementach w porządku LIFO.
     *
     * @return iterator stosu do przechodzenia po elementach w porządku LIFO
     */
    public Iterator<Item> iterator()  { return new ListIterator();  }

    // iterator nie udostępnia metody remove() (ponieważ jest opcjonalna)
    private class ListIterator implements Iterator<Item> {
        private Node current = first;
        public boolean hasNext()  { return current != null;                     }
        public void remove()      { throw new UnsupportedOperationException();  }

        public Item next() {
            if (!hasNext()) throw new NoSuchElementException();
            Item item = current.item;
            current = current.next; 
            return item;
        }
    }


    /**
     * Testy jednostkowe typu danych {@code Stack}
     */
    public static void main(String[] args) {
        Stack<String> stack = new Stack<String>();
        while (!StdIn.isEmpty()) {
            String item = StdIn.readString();
            if (!item.equals("-")) stack.push(item);
            else if (!stack.isEmpty()) StdOut.print(stack.pop() + " ");
        }
        StdOut.println("(pozostało na stosie: " + stack.size() + ")");
    }
}
