/******************************************************************************
 *  Kompilacja:  javac Graph.java
 *  Wykonanie:    java Graph < input.txt
 *  Zależności: ST.java SET.java In.java StdOut.java
 *  Pliki z danymi:   https://introcs.cs.princeton.edu/45graph/tinyGraph.txt
 *  
 *  Typ danych reprezentujący graf nieskierowany. Zaimplementowany
 *  z użyciem tablicy symboli, w której klucze to wierzchołki (typu String),
 *  a wartości to zbiory sąsiadów (kolekcja SET elementów typu String).
 *
 *  Uwagi
 *  -------
 *   - krawędzie równoległe są niedozwolone
 *   - pętle są dopuszczalne
 *   - listy sąsiedztwa przechowują wiele różnych kopii tego samego ciągu znaków.
 *     Możesz ograniczyć zużycie pamięci, zapisując ciągi znaków zewnętrznie.
 *
 *  % java Graph < tinyGraph.txt
 *  A: B C G H 
 *  B: A C H 
 *  C: A B G 
 *  G: A C 
 *  H: A B 
 *
 *  A: B C G H 
 *  B: A C H 
 *  C: A B G 
 *  G: A C 
 *  H: A B 
 *
 ******************************************************************************/

/**
 *  Klasa {@code Graph} reprezentuje graf nieskierowany z wierzchołkami
 *  o nazwach w postaci ciągów znaków.
 *  Obsługuje operacje: dodaj krawędź, dodaj wierzchołek,
 *  pobierz wszystkie wierzchołki, przejdź po wszystkich sąsiadach danego wierzchołka,
 *  czy istnieje wierzchołek, czy występuje krawędź między dwoma wierzchołkami.
 *  Pętle są dozwolone; krawędzie równoległe są niedopuszczalne.
 *  <p>
 *  Dodatkowa dokumentacja: <a href="https://introcs.cs.princeton.edu/45graph">podrozdział 4.5</a> książki
 *  <i>Wprowadzenie do programowania w Javie</i> (Robert Sedgewick i Kevin Wayne).
 */
public class Graph {

    // Tablica symboli: klucz = wierzchołek (ciąg znaków), wartość = zbiór sąsiednich wierzchołków
    private ST<String, SET<String>> st;

    // Liczba krawędzi
    private int E;

   /**
     * Inicjuje pusty graf bez wierzchołków i krawędzi.
     */
    public Graph() {
        st = new ST<String, SET<String>>();
    }

   /**
     * Inicjuje graf na podstawie podanego pliku, używając określonego ogranicznika.
     *
     * @param filename nazwa pliku
     * @param delimiter ogranicznik
     */
    public Graph(String filename, String delimiter) {
        st = new ST<String, SET<String>>();
        In in = new In(filename);
        while (in.hasNextLine()) {
            String line = in.readLine();
            String[] names = line.split(delimiter);
            for (int i = 1; i < names.length; i++) {
                addEdge(names[0], names[i]);
            }
        }
    }

   /**
     * Zwraca liczbę wierzchołków grafu.
     *
     * @return liczbę wierzchołków grafu
     */
    public int V() {
        return st.size();
    }

   /**
     * Zwraca liczbę krawędzi grafu.
     *
     * @return liczbę krawędzi grafu
     */
    public int E() {
        return E;
    }

    // zgłasza wyjątek, jeśli v to nie wierzchołek
    private void validateVertex(String v) {
        if (!hasVertex(v)) throw new IllegalArgumentException(v + " to nie wierzchołek");
    }

   /**
     * Zwraca stopień wierzchołka v w grafie.
     *
     * @param  v wierzchołek
     * @return stopień {@code v} w grafie
     * @throws IllegalArgumentException, jeśli {@code v} nie jest wierzchołkiem grafu
     */
    public int degree(String v) {
        validateVertex(v);
        return st.get(v).size();
    }

   /**
     * Dodaje krawędź v-w do grafu (jeśli jeszcze nie istnieje).
     *
     * @param  v jeden wierzchołek krawędzi
     * @param  w drugi wierzchołek krawędzi
     */
    public void addEdge(String v, String w) {
        if (!hasVertex(v)) addVertex(v);
        if (!hasVertex(w)) addVertex(w);
        if (!hasEdge(v, w)) E++;
        st.get(v).add(w);
        st.get(w).add(v);
    }

   /**
     * Dodaje wierzchołek v do grafu (jeśli jeszcze nie istnieje).
     *
     * @param  v wierzchołek
     */
    public void addVertex(String v) {
        if (!hasVertex(v)) st.put(v, new SET<String>());
    }


   /**
     * Zwraca wierzchołki grafu.
     *
     * @return zbiór wierzchołków grafu
     */
    public Iterable<String> vertices() {
        return st.keys();
    }

   /**
     * Zwraca zbiór wierzchołków przyległych do v w grafie.
     *
     * @param  v wierzchołek
     * @return zbiór wierzchołków przyległych do {@code v} w danym grafie
     * @throws IllegalArgumentException, jeśli {@code v} nie jest wierzchołkiem w tym grafie
     */
    public Iterable<String> adjacentTo(String v) {
        validateVertex(v);
        return st.get(v);
    }

   /**
     * Zwraca true, jeśli v to wierzchołek w grafie.
     *
     * @param  v wierzchołek
     * @return {@code true}, jeśli {@code v} to wierzchołek w grafie, i
     *         {@code false} w przeciwnym razie
     */
    public boolean hasVertex(String v) {
        return st.contains(v);
    }

   /**
     * Zwraca true, jeśli v-w to krawędź w danym grafie.
     *
     * @param  v jeden wierzchołek krawędzi
     * @param  w drugi wierzchołek krawędzi
     * @return {@code true}, jeśli {@code v-w} to krawędź w grafie;
     *         w przeciwnym razie {@code false}
     * @throws IllegalArgumentException, jeśli {@code v} lub {@code w}
     *         nie jest wierzchołkiem w grafie
     */
    public boolean hasEdge(String v, String w) {
        validateVertex(v);
        validateVertex(w);
        return st.get(v).contains(w);
    }

   /**
     * Zwraca ciąg znaków reprezentujący dany graf
     *
     * @return ciąg znaków reprezentujący dany graf
     */
    public String toString() {
        StringBuilder s = new StringBuilder();
        for (String v : st) {
            s.append(v + ": ");
            for (String w : st.get(v)) {
                s.append(w + " ");
            }
            s.append('\n');
        }
        return s.toString();
    }

   /**
     * Testy jednostkowe typu danych {@code Graph}
     */
    public static void main(String[] args) {

        // tworzenie grafu
        Graph graph = new Graph();
        while (!StdIn.isEmpty()) {
            String v = StdIn.readString();
            String w = StdIn.readString();
            graph.addEdge(v, w);
        }

        // wyświetlanie grafu
        StdOut.println(graph);

        // ponowne wyświetlanie grafu na podstawie iterowania po wierzchołkach i krawędziach
        for (String v : graph.vertices()) {
            StdOut.print(v + ": ");
            for (String w : graph.adjacentTo(v)) {
                StdOut.print(w + " ");
            }
            StdOut.println();
        }

    }

}