﻿# Implementuje prosty program kontroli pisowni, pozwalający na wskazywanie
# własnych słowników pomocniczych. Słownik domyślny jest konstruowany na bazie
# standardowych słowników systemowych; można je podmienić w wywołaniu programu.
#
# Słowniki są prostymi plikami tekstowymi, z jednym słowem w wierszu.
# W przeciwieństwie do słowników uniksowego spell(1), słowniki nie muszą być
# posortowane, co uniezależnia skuteczność kontroli pisowni od schematu 
# lokalizacji, choć schemat taki może wpłynąć na kolejność wyjątków w 
# zestawieniu wynikowym. Domyślna lista słowników może być wskazana wartością
# zmiennej środowiskowej DICTIONARIES, można ją jednak podmienić w wywołaniu.
#
# Wyodrębnianie słów poprzedzane jest zastąpieniem znaków sterujących ASCII,
# cyfr, znaków interpunkcyjnych (z wyjątkiem znaku ') znakiem spacji (kod ASCII
# 32). To, co pozostanie po podmianie, jest interpretowane jako lista słów,
# porównywanych z pozycjami słownika. Dzięki temu program skutecznie kontroluje
# poprawność języków z kodowaniem liter w ASCII albo ISO-8859-n, oraz plików Unicode
# z ośmiobitowym kodowaniem UTF-8.
#
# Na potrzeby porównywania program ujednolica wielkość liter słów tekstu wejściowego
# (poddawane konwersji funkcją tolower()).
#
# W tej wersji, mającej obsługiwać wiele języków, domyślnie program nie próbuje 
# samodzielnie generować końcówek słów, chyba, że zostanie wywołany z opcją +strip.
#
# Końcówki (przyrostki) są definiowane wyrażeniami regularnymi i mogą być przekazywane do
# programu za osobnych pośrednictwem plików wymienionych w wierszu wywołania, albo czerpane
# z domyślnego wewnętrznego zestawu końcówek słów języka angielskiego. Komentarze występujące
# w pliku końcówek rozpoczynają się od znaku (#) i rozciągają się do końca wiersza. Każde 
# wyrażenie regularne końcówki powinno kończyć się znakiem $, kotwiczącym wyrażenie na końcu
# słowa. Każde z tych wyrażeń może być uzupełnione listą ciągów zastępczych, przy założeniu,
# że "" oznacza ciąg pusty. Oto przykład:
#
#	ies$	ie ies y	# flies -> fly, series -> series, ties -> tie
#	ily$	y ily		# happily -> happy, wily -> wily
#
# Choć dozwolone jest włączenie końcówek w ciągach zastępczych, nie jest to konieczne, bo 
# wyszukiwanie słów jest realizowane przed obcinaniem końcówek.
#
# Końcówki są testowane w kolejności zgodnej z ich malejącą długością, więc na początku
# próbowane są najdłuższe dopasowania.
#
# Domyślnie wyjście stanowi uporządkowaną alfabetycznie listę (pozbawioną duplikatów) słów
# spoza słownika -- wyjątków pisowni -- po jednym w wierszu. Opcja verbose, wymusza 
# wypisywanie zestawienia szczegółowego w formacie:
#
#	plik:wiersz:wyjątek
#
# Niektóre uniksowe edytory tekstowe rozpoznają ów format, co pozwala na wykorzystanie
# zestawienia wyjątków do szybkiego poprawienia błędów w tekście.
#
# Stosowanie:
#	awk [-v Dictionaries="slowniksys1 slowniksys1 ..."] -f spell.awk -- \
#	    [=koncowki1 =koncowki1 ...] [+pomocniczy1 +pomocniczy1 ...] \
#	    [-strip] [-verbose] [plik(i)]

BEGIN	{ initialize() }

	{ spell_check_line() }

END	{ report_exceptions() }

function get_dictionaries(        files, key)
{
    if ((Dictionaries == "") && ("DICTIONARIES" in ENVIRON))
	Dictionaries = ENVIRON["DICTIONARIES"]
    if (Dictionaries == "")    # wybierz domyślną listę słowników
    {
        DictionaryFiles["/usr/dict/words"]++
        DictionaryFiles["/usr/local/share/dict/words.knuth"]++
    }
    else                       # wybierz słowniki przekazane w wierszu wywołania
    {
	split(Dictionaries, files)
	for (key in files)
	    DictionaryFiles[files[key]]++
    }
}

function initialize()
{
   NonWordChars = "[^" \
	"'" \
	"ABCDEFGHIJKLMNOPQRSTUVWXYZ" \
	"abcdefghijklmnopqrstuvwxyz" \
	"\200\201\202\203\204\205\206\207\210\211\212\213\214\215\216\217" \
	"\220\221\222\223\224\225\226\227\230\231\232\233\234\235\236\237" \
	"\240\241\242\243\244\245\246\247\250\251\252\253\254\255\256\257" \
	"\260\261\262\263\264\265\266\267\270\271\272\273\274\275\276\277" \
	"\300\301\302\303\304\305\306\307\310\311\312\313\314\315\316\317" \
	"\320\321\322\323\324\325\326\327\330\331\332\333\334\335\336\337" \
	"\340\341\342\343\344\345\346\347\350\351\352\353\354\355\356\357" \
	"\360\361\362\363\364\365\366\367\370\371\372\373\374\375\376\377" \
	"]"
    get_dictionaries()
    scan_options()
    load_dictionaries()
    load_suffixes()
    order_suffixes()
}

function load_dictionaries(        file, word)
{
    for (file in DictionaryFiles)
    {
	## print "DIAGNOSTYKA: Wczytywanie slownika " file > "/dev/stderr"
	while ((getline word < file) > 0)
	    Dictionary[tolower(word)]++
	close(file)
    }
}

function load_suffixes(        file, k, line, n, parts)
{
    if (NSuffixFiles > 0)		# wczytanie reguł przyrostków z plików
    {
	for (file in SuffixFiles)
	{
	    ## print DIAGNOSTYKA: wczytywanie pliku regul " file > "/dev/stderr"
	    while ((getline line < file) > 0)
	    {
		sub(" *#.*$", "", line)		# wycięcie komentarzy
		sub("^[ \t]+", "", line)	# i poprzedzających je odstępów
		sub("[ \t]+$", "", line)	# oraz odstępów uzupełniających
		if (line == "")
		    continue
		n = split(line, parts)
		Suffixes[parts[1]]++
		Replacement[parts[1]] = parts[2]
		for (k = 3; k <= n; k++)
		    Replacement[parts[1]] = Replacement[parts[1]] " " parts[k]
	    }
	    close(file)
	}
    }
    else	      # wczytanie domyślnej tabeli reguł przyrostków dla języka angielskiego
    {
	split("'$ 's$ ed$ edly$ es$ ing$ ingly$ ly$ s$", parts)
	for (k in parts)
	{
	    Suffixes[parts[k]] = 1
	    Replacement[parts[k]] = ""
	}
    }
}

function order_suffixes(        i, j, key)
{
    # Porządkowanie przyrostków wedle malejącego rozmiaru
    NOrderedSuffix = 0
    for (key in Suffixes)
	OrderedSuffix[++NOrderedSuffix] = key
    for (i = 1; i < NOrderedSuffix; i++)
	for (j = i + 1; j <= NOrderedSuffix; j++)
	    if (length(OrderedSuffix[i]) < length(OrderedSuffix[j]))
		swap(OrderedSuffix, i, j)
}

function report_exceptions(        key, sortpipe)
{
    ## print "DIAGNOSTYKA: report_exceptions(): NR = ", NR, " FILENAME =", FILENAME
    sortpipe = Verbose ? "sort -f -t: -u -k1,1 -k2n,2 -k3" : "sort -f -u -k1"
    for (key in Exception)
	print Exception[key] | sortpipe
    close(sortpipe)
}

function scan_options(        k)
{
    for (k = 1; k < ARGC; k++)
    {
	if (ARGV[k] == "-strip")
	{
	    ARGV[k] = ""
	    Strip = 1
	}
	else if (ARGV[k] == "-verbose")
	{
	    ARGV[k] = ""
	    Verbose = 1
	}
	else if (ARGV[k] ~ /^=/)	# plik przyrostków
	{
	    NSuffixFiles++
	    SuffixFiles[substr(ARGV[k], 2)]++
	    ARGV[k] = ""
	}
	else if (ARGV[k] ~ /^[+]/)	# słownik prywatny
	{
	    DictionaryFiles[substr(ARGV[k], 2)]++
	    ARGV[k] = ""
	}
    }

    # Usuwanie pustych argumentów z końca tablicy (dla nawk)
    while ((ARGC > 0) && (ARGV[ARGC-1] == ""))
        ARGC--
}

function spell_check_line(        k, word)
{
    ## for (k = 1; k <= NF; k++) print "DIAGNOSTYKA: word[" k "] = \"" $k "\""
    gsub(NonWordChars, " ")		# eliminowanie znaków nieliterowych
    for (k = 1; k <= NF; k++)
    {
	word = $k
	sub("^'+", "", word)		# wycięcie apostrofów poprzedzających słowo
	sub("'+$", "", word)		# wycięcie apostrofów uzupełniających słowo
	if (word != "")
	    spell_check_word(word)
    }
}

function spell_check_word(word,        key, lc_word, location, w, wordlist)
{
    lc_word = tolower(word)
    ## print "DEBUG: spell_check_word(" word ") -> tolower -> " lc_word
    if (lc_word in Dictionary)		# poprawna pisownia
	return
    else				# potencjalny wyjątek
    {
	if (Strip)
	{
	    strip_suffixes(lc_word, wordlist)
	    ## for (w in wordlist) print "DIAGNOSTYKA: wordlist[" w "]"
	    for (w in wordlist)
		if (w in Dictionary)
			return
	}
	## print "DIAGNOSTYKA: spell_check():", word
	location = Verbose ? (FILENAME ":" FNR ":") : ""
	if (lc_word in Exception)
	    Exception[lc_word] = Exception[lc_word] "\n" location word
	else
	    Exception[lc_word] = location word
    }
}

function strip_suffixes(word, wordlist,        ending, k, n, regexp)
{
    ## print "DIAGNOSTYKA: strip_suffixes(" word ")"
    split("", wordlist)
    for (k = 1; k <= NOrderedSuffix; k++)
    {
	regexp = OrderedSuffix[k]
	## print "DIAGNOSTYKA: strip_suffixes(): testowanie \"" regexp "\""
	if (match(word, regexp))
	{
	    ## print "DIAGNOSTYKA: strip_suffixes(): DOPASOWANIE: RSTART =", RSTART, " RLENGTH =", RLENGTH
	    word = substr(word, 1, RSTART - 1)
	    if (Replacement[regexp] == "")
		wordlist[word] = 1
	    else
	    {
		split(Replacement[regexp], ending)
		for (n in ending)
		{
		    if (ending[n] == "\"\"")
			ending[n] = ""
		    wordlist[word ending[n]] = 1
		}
	    }
	    break
	}
    }
     ## for (n in wordlist) print "DIAGNOSTYKA: strip_suffixes() -> \"" n "\""
}

function swap(a, i, j,        temp)
{
    temp = a[i]
    a[i] = a[j]
    a[j] = temp
}
