﻿#! /bin/sh -
# Kompiluje jeden albo wiecej pakietow na jednym albo rownolegle na wielu wezlach docelowych.
#
# Stosowanie:
#	build-all [ --? ]
#		  [ --all "..." ]
#		  [ --cd "..." ]
#		  [ --check "..." ]
#		  [ --configure "..." ]
#		  [ --environment "..." ]
#		  [ --help ]
#		  [ --logdirectory katalog ]
#		  [ --on "[konto@]wezel[:katalog][,plik-srodowiska] ..." ]
#		  [ --source "katalog ..." ]
#		  [ --userhosts "plik(i)" ]
#		  [ --version ]
#		  pakiet(y)
#
# Opcjonalne pliki konfiguracji:
#	$HOME/.build/directories	lista katalogow archiwow kodu zrodlowego pakietow
#	$HOME/.build/userhosts		lista [konto@]wezel[:katalog][,plik-srodowiska]

IFS='
 	'

PATH=/usr/local/bin:/bin:/usr/bin
export PATH

UMASK=002
umask $UMASK

build_one()
{
	# Stosowanie:
	#	build_one [konto@]wezel[:katalog-kompilacji][,plik-srodowiska]

	arg="`eval echo $1`"

	userhost="`echo $arg | sed -e 's/:.*$//'`"

	user="`echo $userhost | sed -e s'/@.*$//'`"
	test "$user" = "$userhost" && user=$USER

	host="`echo $userhost | sed -e s'/^[^@]*@//'`"

	envfile="`echo $arg | sed -e 's/^[^,]*,//'`"
	test "$envfile" = "$arg" && envfile=/dev/null

	builddir="`echo $arg | sed -e s'/^.*://' -e 's/,.*//'`"
	test "$builddir" = "$arg" && builddir=/tmp

	parbase=`basename $PARFILE`

	# UWAGA: zmiana tych wzorcow wymaga aktualizacji kodu funkcji find_package()
	package="`echo $parbase | \
		sed	-e 's/[.]jar$//' \
			-e 's/[.]tar[.]bz2$//' \
			-e 's/[.]tar[.]gz$//' \
			-e 's/[.]tar[.]Z$//' \
			-e 's/[.]tar$//' \
			-e 's/[.]tgz$//' \
			-e 's/[.]zip$//'`"

	## echo "DIAGNOSTYKA: LOGDIR   = $LOGDIR"
	## echo "DIAGNOSTYKA: PARFILE  = $PARFILE"
	## echo "DIAGNOSTYKA: PAR      = $PAR"
	## echo "DIAGNOSTYKA: package  = $package"
	## echo "DIAGNOSTYKA: arg      = $arg"
	## echo "DIAGNOSTYKA: userhost = $userhost"
	## echo "DIAGNOSTYKA: host     = $host"
	## echo "DIAGNOSTYKA: builddir = $builddir"

	# Kopiowanie pliku archiwum kodu zrodlowego (jesli nie ma go na wezle zdalnym)
	echo $SSH $SSHFLAGS $userhost "test -f $PARFILE"
	if $SSH $SSHFLAGS $userhost "test -f $PARFILE"
	then
		parbaselocal=$PARFILE
	else
		parbaselocal=$parbase
		echo $SCP $PARFILE $userhost:$builddir
		$SCP $PARFILE $userhost:$builddir
	fi

	# Rozpakowanie archiwum, kompilacja i test wynikow -- wszystko w tle

	sleep 1			# gwarantuje niepowtarzalnosc generowanych nazw plikow dziennikow
	now="`date $DATEFLAGS`"
	logfile="$package.$host.$now.log"

	## echo "DIAGNOSTYKA: LOGFILE  = $LOGDIR/$package.$host.$now.log"

	nice $SSH $SSHFLAGS $userhost "
			echo '============================================================' ;
		        test -f $BUILDBEGIN && . $BUILDBEGIN || test -f $BUILDBEGIN && source $BUILDBEGIN || true ;
			echo 'Pakiet:                        $package' ;
			echo 'Archiwum:                      $PARFILE' ;
			echo 'Data:                          $now' ;
			echo 'Konto lokalne:                 $USER' ;
			echo 'Wezel lokalny:                 `hostname`' ;
			echo 'Lokalny katalog dziennika:     $LOGDIR' ;
			echo 'Lokalny plik dziennika:        $logfile' ;
			echo 'Konto zdalne:                  $user' ;
			echo 'Wezel zdalny:                  $host' ;
			echo 'Zdalny katalog kompilacji:     $builddir' ;
			printf 'Data w systemie zdalnym:       ' ;
			date $DATEFLAGS ;
			printf 'architektura systemu zdalnego: ' ;
			uname -a || true ;
			printf 'Zdalna wersja gcc:             ' ;
			gcc --version | head -n 1 || echo ;
			printf 'Zdalna wersja g++:             ' ;
			g++ --version | head -n 1 || echo ;
			echo 'Srodowisko konfiguracji:    `$STRIPCOMMENTS $envfile | $JOINLINES`' ;
			echo 'Srodowisko dodatkowe:        $EXTRAENVIRONMENT' ;
			echo 'Katalog konfiguracji:        $CONFIGUREDIR' ;
			echo 'Znaczniki konfiguracji:      $CONFIGUREFLAGS' ;
			echo 'Zadania kompilacji:          $ALLTARGETS' ;
			echo 'Zadania kontrolne:           $CHECKTARGETS' ;
			echo 'Raport zajetosci dysku dla $builddir/$package:' ;
			df $builddir | $INDENT ;
			echo 'Srodowisko:' ;
			env | env LC_ALL=C sort | $INDENT ;
			echo '============================================================' ;
			umask $UMASK ;
			cd $builddir || exit 1 ;
			/bin/rm -rf $builddir/$package ;
			$PAR $parbaselocal ;
			test "$parbase" = "$parbaselocal" && /bin/rm -f $parbase ;
			cd $package/$CONFIGUREDIR || exit 1 ;
			test -f configure && \
				chmod a+x configure && \
					env `$STRIPCOMMENTS $envfile | $JOINLINES` \
						$EXTRAENVIRONMENT \
							nice time ./configure $CONFIGUREFLAGS ;
			nice time make $ALLTARGETS && nice time make $CHECKTARGETS ;
			echo '============================================================' ;
			echo 'Raport dostepnosci dysku dla $builddir/$package:' ;
			df $builddir | $INDENT ;
			printf 'Data w systemie zdalnym:  ' ;
			date $DATEFLAGS ;
			cd ;
		        test -f $BUILDEND && . $BUILDEND || test -f $BUILDEND && source $BUILDEND || true ;
			echo '============================================================' ;
		" < /dev/null > "$LOGDIR/$logfile" 2>&1 &
}

error()
{
	echo "$@" 1>&2
	usage_and_exit 1
}

find_file()
{
	# Stosowanie:
	#	find_file plik program-i-argumenty
	# Zwraca 0 (sukces) - gdy znaleziono plik, 1 (porazka) - gdy nie znaleziono pliku

	if test -r "$1"
	then
		PAR="$2"
		PARFILE="$1"
		return 0
	else
		return 1
	fi
}

find_package()
{
	# Stosowanie: find_package pakiet-x.y.z
	base=`echo "$1" | sed -e 's/[-_][.]*[0-9].*$//'`
## echo "DIAGNOSTYKA: find_package $1"
## echo "DIAGNOSTYKA: SRCDIRS = $SRCDIRS"
	PAR=
	PARFILE=
	for srcdir in $SRCDIRS
	do
		test "$srcdir" = "." && srcdir="`pwd`"
## echo "DIAGNOSTYKA: srcdir = $srcdir"

		for subdir in "$base" ""
		do
			# UWAGA: zmiana tej listy wymaga aktualizacji ustawien pakietow w funkcji build_one()
			find_file $srcdir/$subdir/$1.tar.gz	"tar xfz"	&& return
			find_file $srcdir/$subdir/$1.tar.Z	"tar xfz"	&& return
			find_file $srcdir/$subdir/$1.tar	"tar xf"	&& return
			find_file $srcdir/$subdir/$1.tar.bz2	"tar xfj"	&& return
			find_file $srcdir/$subdir/$1.tgz	"tar xfz"	&& return
			find_file $srcdir/$subdir/$1.zip	"unzip -q"	&& return
			find_file $srcdir/$subdir/$1.jar	"jar xf"	&& return
		done
	done
}

set_userhosts()
{
	# Stosowanie: set_userhosts plik(i)
	for u in "$@"
	do
		if test -r "$u"
		then
			ALTUSERHOSTS="$ALTUSERHOSTS $u"
		elif test -r "$BUILDHOME/$u"
		then
			ALTUSERHOSTS="$ALTUSERHOSTS $BUILDHOME/$u"
		else
			error "Nie znaleziono pliku: $u"
		fi
	done
}

usage()
{
	cat <<EOF
Stosowanie:
	$PROGRAM [ --? ]
			[ --all "..." ]
			[ --cd "..." ]
			[ --check "..." ]
			[ --configure "..." ]
			[ --environment "..." ]
			[ --help ]
			[ --logdirectory katalog ]
			[ --on "[konto@]wezel[:katalog][,plik-srodowiska] ..." ]
			[ --source "katalog ..." ]
			[ --userhosts "plik(i)" ]
			[ --version ]
			pakiet(y)
EOF
}

usage_and_exit()
{
	usage
	exit $1
}

version()
{
	echo "$PROGRAM wersja $VERSION"
}

warning()
{
	echo "$@" 1>&2
	EXITCODE=`expr $EXITCODE + 1`
}

ALLTARGETS=
altlogdir=
altsrcdirs=
ALTUSERHOSTS=
BUILDBEGIN=./.build/begin
BUILDEND=./.build/end
BUILDHOME=$HOME/.build
CHECKTARGETS=check
CONFIGUREDIR=.
CONFIGUREFLAGS=
DATEFLAGS="+%Y.%m.%d.%H.%M.%S"
EXITCODE=0
EXTRAENVIRONMENT=
INDENT="awk '{ print \"\t\t\t\" \$0 }'"
JOINLINES="tr '\n' '\040'"
LOGDIR=
PROGRAM=`basename $0`
SCP=scp
SSH=ssh
# SCP="echo scp"
# SSH="echo ssh"
SSHFLAGS=${SSHFLAGS--x}
STRIPCOMMENTS='sed -e s/#.*$//'
userhosts=
VERSION=1.0

# Domyslne pliki konfiguracyjne
defaultdirectories=$BUILDHOME/directories
defaultuserhosts=$BUILDHOME/userhosts

# Lista domyslnych lokalizacji archiwow kodu zrodlowego
# i lista domyslnych kont i wezlow kompilacji:
SRCDIRS="`$STRIPCOMMENTS $defaultdirectories 2> /dev/null`"
test -z "$SRCDIRS" && \
	SRCDIRS="
			.
			/usr/local/src
			/usr/local/gnu/src
			$HOME/src
			$HOME/gnu/src
			/tmp
			/usr/tmp
			/var/tmp
		"

while test $# -gt 0
do
	case $1 in

	--all | --al | --a | -all | -al | -a )
		shift
		ALLTARGETS="$1"
		;;

	--cd | -cd )
		shift
		CONFIGUREDIR="$1"
		;;

	--check | --chec | --che | --ch | -check | -chec | -che | -ch )
		shift
		CHECKTARGETS="$1"
		;;

	--configure | --configur | --configu | --config | --confi | \
	--conf | --con | --co | \
	-configure | -configur | -configu | -config | -confi | \
	-conf | -con | -co )
		shift
		CONFIGUREFLAGS="$1"
		;;

	--environment | --environmen | --environme | --environm | --environ | \
	--enviro | --envir | --envi | --env | --en | --e | \
	-environment | -environmen | -environme | -environm | -environ | \
	-enviro | -envir | -envi | -env | -en | -e )
		shift
		EXTRAENVIRONMENT="$1"
		;;

	--help | --hel | --he | --h | '--?' | -help | -hel | -he | -h | '-?' )
		usage_and_exit 0
		;;

	--logdirectory | --logdirector | --logdirecto | --logdirect | \
	--logdirec | --logdire | --logdir | --logdi | --logd | --log | \
	--lo | --l | \
	-logdirectory | -logdirector | -logdirecto | -logdirect | \
	-logdirec | -logdire | -logdir | -logdi | -logd | -log | -lo | -l )
		shift
		altlogdir="$1"
		;;

	--on | --o | -on | -o )
		shift
		userhosts="$userhosts $1"
		;;

	--source | --sourc | --sour | --sou | --so | --s | \
	-source | -sourc | -sour | -sou | -so | -s )
		shift
		altsrcdirs="$altsrcdirs $1"
		;;

	--userhosts | --userhost | --userhos | --userho | --userh | \
	--user | --use | --us | --u | \
	-userhosts | -userhost | -userhos | -userho | -userh | \
	-user | -use | -us | -u )
		shift
		set_userhosts $1
		;;

	--version | --versio | --versi | --vers | --ver | --ve | --v | \
	-version | -versio | -versi | -vers | -ver | -ve | -v )
		version
		exit 0
		;;

	-*)
		error "Nieznana opcja: $1"
		;;

	*)
		break
		;;

	esac
	shift
done

# Wyszukanie programupocztowego
for MAIL in /bin/mailx /usr/bin/mailx /usr/sbin/mailx /usr/ucb/mailx /bin/mail /usr/bin/mail
do
	test -x $MAIL && break
done
test -x $MAIL || error "Nie mozna znalezc klienta poczty elektronicznej"

# Katalogi archiwow kodu przekazane w wywolaniu maja pierwszenstwo przed domyslnymi
SRCDIRS="$altsrcdirs $SRCDIRS"

if test -n "$userhosts"
then
	test -n "$ALTUSERHOSTS" &&
	    userhosts="$userhosts `$STRIPCOMMENTS $ALTUSERHOSTS 2> /dev/null`"
else
	test -z "$ALTUSERHOSTS" && ALTUSERHOSTS="$defaultuserhosts"
	userhosts="`$STRIPCOMMENTS $ALTUSERHOSTS 2> /dev/null`"
fi

# Czy jest cos do roboty?
test -z "$userhosts" && usage_and_exit 1

for p in "$@"
do
	find_package "$p"

	if test -z "$PARFILE"
	then
		warning "Nie mozna znalezc pliku pakietu $p"
		continue
	fi

	LOGDIR="$altlogdir"
	if test -z "$LOGDIR" -o ! -d "$LOGDIR" -o ! -w "$LOGDIR"
	then
		for LOGDIR in "`dirname $PARFILE`/logs/$p" $BUILDHOME/logs/$p /usr/tmp /var/tmp /tmp
		do
			test -d "$LOGDIR" || mkdir -p "$LOGDIR" 2> /dev/null
			test -d "$LOGDIR" -a -w "$LOGDIR" && break
		done
	fi

	msg="Zajrzyj do dziennika kompilacji pakietu $p w `hostname`:$LOGDIR"
	echo "$msg"
	echo "$msg" | $MAIL -s "$msg" $USER 2> /dev/null

	for u in $userhosts
	do
		build_one $u
	done
done

# Ograniczenie wartosci kodu powrotnego do konwencji systemu Unix
test $EXITCODE -gt 125 && EXITCODE=125

exit $EXITCODE
